/************************************************************************

Teeny.cpp: a demonstration app for s11n, copied from a real-life s11n
client.

It demonstrates:

- Using a Serializable hierarchy of types.

- A rather detailed serialization algorithm, specialized to serialize
a particular client-side type much more compactly than the default
algorithms can do. See serialize_coord_char_map(), in Teeny_s11n.hpp.

************************************************************************/

#include <sstream>

#include "Teeny.hpp" // core teeny types

#include <s11n.net/s11n/s11nlite.hpp> // s11nlite framework
#include <s11n.net/s11n/algo.hpp> // create_child()

#include "Teeny_s11n.hpp" // de/serialize_coord_char_map()

#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // CERR

// Register Board with the classloader, etc.:
#define S11N_TYPE teeny::Board
#define S11N_TYPE_NAME "teeny::Board"
#define S11N_BASE_TYPE teeny::Serializable
#include <s11n.net/s11n/reg_s11n_traits.hpp>


int main( int argc, char ** argv )
{
	// Demonstration of using s11n with the objects defined in the
	// teeny namespace.

	using namespace teeny;
	using namespace s11nlite;

	// For this example we don't want s11n to throw exceptions:
	s11n::throw_policy( s11n::ThrowNever );

	// Create a game board...
	Board brd( 10, 10, '.' );
	brd.setTerrain( 0, 3, "******" );
	brd.moveActor( 5, 5, 'A' );
	brd.moveActor( 2, 2, 'B' );

	// Set our preferred output format (not required):
	serializer_class( "funxml" );

	// Now save the board to an arbitrary ostream.
	std::ostringstream os;
	if( ! save<Serializable>( brd, os ) )
	// Note the use of the 'Serializable' template parameter. While not strictly
	// required in this case, it is probably a good idea to be explicit here.
	// Serializable is the base s11n-aware interface for the teeny types, which
	// is why we want to use it, as opposed to Board. See below for more details.
	{
		CERR << "Serializing Board failed!\n";
		return 1;
	}

	// Now load a new object from the stream:
	std::istringstream is( os.str() );
	Serializable * ser = s11nlite::load_serializable<Serializable>( is );
	// Again, note the use of Serializable as our template parameter.
	// This is more important when loading than when saving, as it determines
	// which factory/classloader will be used. Using the subtype here will
	// probably not have the desired effect.
	if( ! ser )
	{
		CERR << "Deser of Board failed!\n";
		return 2;
	}

	// If we need the inherited type, we must attempt a dynamic cast:
	Board * b2 = dynamic_cast<Board*>( ser );
	if( ! b2 )
	{
		CERR << "Loaded Serializable is not of the proper type!\n";
		delete ser;
		return 3;
	}

	CERR << "Deserialized Board:\n";
	save<Serializable>( *b2, std::cout );
	delete ser;
	ser = b2 = 0;
	return 0;

}



namespace teeny
{


	Board::Board() : m_wt(0), m_ht(0) {}

	Board::Board(int w, int h, const avatar_type & fill )
	{
		this->recreate( w, h, fill );
	}

	void Board::setTerrain( int x, int y, avatar_type c )
	{
		this->m_terrain[std::make_pair( x, y )] = c;
	}

// 	Board::avatar_map_type &
// 	Board::terrainMap()
// 	{
// 		return this->m_terrain;
// 	}
	
	const Board::avatar_map_type &
	Board::terrainMap() const
	{
		return this->m_terrain;
	}

	const Board::avatar_map_type &
	Board::actorMap() const
	{
		return this->m_act;
	}

	avatar_type Board::terrainAt( int x, int y ) const
	{
		coord_type key(x,y);
		avatar_map_type::const_iterator cit = m_terrain.find( key );
		if( m_terrain.end() == cit ) return 0;
		return (*cit).second;
	}

	avatar_type Board::actorAt( int x, int y ) const
	{
		coord_type key(x,y);
		avatar_map_type::const_iterator cit = m_act.find( key );
		if( m_act.end() == cit ) return 0;
		return (*cit).second;
	}

	bool Board::findActor( int & atx, int & aty, avatar_type a ) const
	{
		typedef actor_reverse_map_type::const_iterator CIT;
		CIT it = this->m_ract.find( a );
		if( this->m_ract.end() == it )
		{
			return false;
		}
		// todo? double-check against m_act, to make sure the coords zusammen stimmen?
		atx = (*it).second.first;
		aty = (*it).second.second;
		return true;
	}

	Board::coord_type Board::findActor( avatar_type A ) const
	{
		coord_type p(-1,-1);
		this->findActor( p.first, p.second, A );
		return p;
	}

	bool Board::containsActor( avatar_type a ) const
	{
		typedef actor_reverse_map_type::const_iterator CIT;
		return this->m_ract.end() != this->m_ract.find( a );
	}

	int Board::setTerrain( int col, int row, const std::string & in, bool horiz )
	{		
		uint max;
		if( horiz )
		{
			max = this->m_wt-col;
		}
		else
		{
			max = this->m_ht - row;
		}
		if( in.size() < max ) max = in.size();
		uint i = 0;
		for(; i < max; i++ )
		{
			this->m_terrain[Board::coord_type(
						   horiz ? i+col : col,
						   horiz ? row : i+row
						   )] = in[i];
		}
		return i;
	}
	

	void Board::rebuildReverseActorMap()
	{
		this->m_ract.clear();
		avatar_map_type::iterator ait = this->m_act.begin(), aet = this->m_act.end();
		for( ; aet != ait; ++ait )
		{
			this->m_ract[(*ait).second] = (*ait).first;
		}
	}
	bool Board::rotate( int ticks )
	{
		// First naive impl: simply rotate 90 degrees, ticks times.
		// EXTREMELY inefficient, but my brain doesn't handle
		// math very well :/.

		if( ticks < 1 || ticks > 3 ) return false;

		int oldw = this->width();
		int oldh = this->height();
		int neww = oldw, newh = oldh;
		int myx, myy;
		int wdiff;
		int foos;
		coord_type coord;
		avatar_type terra;
		avatar_map_type dter;
		avatar_map_type dact;
		avatar_map_type::iterator ait, aet = this->m_act.end();
		avatar_map_type::iterator tit, tet = this->m_terrain.end();
		for( int i = 0; i < ticks; i++ )
		{
			// simply swap width/height to simulate sideways rotate:
			foos = neww;
			neww = newh;
			newh = foos;
			wdiff = (int) (newh - neww);
			dter.clear();
			dact.clear();
			for( tit = this->m_terrain.begin(); tet != tit; ++tit )
			{
				terra = (*tit).second;
				coord = (*tit).first;
				myx = neww - 1  - coord.second;
				myy = coord.first;
				coord = coord_type(
						   myx, myy
						   );
				dter[coord] = terra;
				ait = this->m_act.find( (*tit).first );
				if( this->m_act.end() != ait )
				{
					dact[coord] = (*ait).second;
				}

			}
 			this->m_wt = neww;
 			this->m_ht = newh;
			this->m_terrain = dter;
			this->m_act = dact;
		}


		this->rebuildReverseActorMap();
		return true;
	}

	bool Board::inBounds( int x, int y ) const
	{
		return ( (x >= (int)0) && (x < this->width()) )
			&&
			( (y >= (int)0) && (y < this->height()) );
	}

	void Board::clear()
	{
// 		this->m_wt = this->m_ht = 0;
		this->m_terrain.clear();
		this->m_act.clear();
		this->m_ract.clear();
	}
	bool Board::empty() const
	{
		return this->m_terrain.empty()
			&& this->m_act.empty();
	}

	void Board::recreate( int w, int h, avatar_type fill )
	{
		this->m_wt = w;
		this->m_ht = h;
		this->clear();
		for( int r = 0; r < h; r++ )
		{
			for( int c = 0; c < w; c++ )
			{
				this->m_terrain[Board::coord_type(c,r)] = fill;
			}
		}
	}	
	
	void Board::dump( std::ostream & os ) const
	{
		typedef avatar_map_type AM;
		typedef actor_reverse_map_type RAM;
		typedef avatar_map_type::const_iterator CIT;
		CIT cit;
		CIT cet = this->m_terrain.end();
		int row = 0, col = 0;
		os << "  ";
		os << std::dec; // somehow stream is getting into octal :?
		for( ; col < this->m_wt; col++ )
		{
			if( col % 10 == 0 ) os << '+';
			else os << (col%10);
		}
		os << "\n";
		
		avatar_type thec;
		AM::const_iterator acit;
		RAM::const_iterator racit;
		Board::coord_type coord;
		for( ; row < this->m_ht; row++ )
		{
			if( row % 10 == 0 ) os << '+';
			else os << (row%10);
			os << " ";
			for( col = 0 ; col < this->m_wt; col++ )
			{
				coord = Board::coord_type( col, row );
				acit = this->m_act.find( coord );
				if( acit != this->m_act.end() )
				{
					os << (*acit).second;
				}
				else
				{
					cit = this->m_terrain.find( coord );
					thec = ( cet == cit ) ? '?' : (*cit).second;
					os << thec;
				}
			}
			os << '\n';
		}
	}
	
	bool Board::serialize( Serializable::S11nNode & n ) const
	{
		NodeTraits::class_name( n, "teeny::Board" );
		serialize_coord_char_map( s11n::create_child( n, "terrain" ),
					  this->m_terrain,
					  this->width(),
					  this->height(), 'X', '?' );
		// todo: use a more compact/editable algo for actors
	        Serializable::S11nNode & ach = s11n::create_child( n, "actors" );
		NodeTraits::set( ach, "ACHTUNG", "The internal format of this serialized data WILL change in the future! Don't rely on it's structure!" );
 		s11n::serialize( ach, this->m_act );
		NodeTraits::set( n, "width", this->m_wt );
		NodeTraits::set( n, "height", this->m_ht );
		return true;
	}

	bool Board::deserialize( const Serializable::S11nNode & n )
	{
		this->clear();
		bool worked = true;
		if( (worked = s11n::deserialize_subnode( n, "actors", this->m_act )) )
		{ // rebuild Actor-to-coordinate reverse map:
			this->rebuildReverseActorMap();
		} else return false;

		this->m_wt = NodeTraits::get( n, "width", this->m_wt );
		this->m_ht = NodeTraits::get( n, "height", this->m_ht );
		const S11nNode * ch = s11n::find_child_by_name( n, "terrain" );
		if( this->m_wt && this->m_ht && ch )
		{
			worked  = deserialize_coord_char_map( *ch,
						    this->m_terrain,
						    this->m_wt,
						    this->m_ht );
		}
		return worked;
	}


      	void Board::moveActor( int x, int y, avatar_type a )
	{
		typedef actor_reverse_map_type RAM;
		typedef avatar_map_type AM;
		RAM::iterator rit = m_ract.find( a );
		if( m_ract.end() != rit )
		{
			// we have the actor...
			Board::coord_type co = (*rit).second;
			AM::iterator ait = m_act.find( co );
			if( m_act.end() != ait )
			{
				m_act.erase( ait );
			}
			else
			{
				/**
				   This is really an internal error: state out of sync.
				   Since the problem is about to be rectified, we'll
				   shamefully ignore it here...
				*/
			} 
		}
		Board::coord_type newc( x, y );
		m_ract[a] = newc;
		m_act[newc] = a;
	}

	bool Board::removeActor( avatar_type a )
	{
		actor_reverse_map_type::iterator mit = this->m_ract.find( a );
		if( this->m_ract.end() == mit ) return false;
		this->m_act.erase( this->m_act.find( (*mit).second ) );
		this->m_ract.erase( mit );
		return true;
	}

        avatar_type
	Board::removeActorAt( int x, int y )
	{
		coord_type coord(x,y);
		avatar_map_type::iterator ait = this->m_act.find( coord );
		if( this->m_act.end() == ait ) return 0;
		avatar_type act = (*ait).second;
		this->m_ract.erase( this->m_ract.find( (*ait).second ) );
		this->m_act.erase( ait );
		return act;
	}

	Board & the_board()
	{
		static Board bob;
		static bool donethat = false;
		if( !donethat && (donethat=true) )
		{
			bob.recreate( 15, 15, '_' );
		}
		return bob;
	}

	bool is_valid_actor( avatar_type a )
	{
		return ((a >= 'a') && (a <= 'z'))
		       ||
			((a >= 'A') && (a <= 'Z'));
	}
	bool is_valid_terrain( avatar_type t )
	{
		return (t >= 32 && t <= 33)
			|| // 34 == "
			(t >= 35 && t <= 38)
			|| // 39 == '
			(t >= 40 && t <=47)
			||
			(t >= 58 && t <= 64)
			||
			(t == 91 || (t >= 93 && t <= 95 ) ) // 92==\ 96==`
			||
			(t >= 123 && t <= 126);

	}

	bool is_valid_avatar( avatar_type t )
	{
		return is_valid_actor(t) || is_valid_terrain(t);
	}

	bool is_digit( avatar_type t )
	{
		return std::isdigit( t );
	}

}// namespace teeny
