#ifndef teeny_CORE_HPP_INCLUDED
#define teeny_CORE_HPP_INCLUDED 1

#include <map>
#include <utility> // pair

// #include <s11n.net/s11n/s11nlite.hpp>
#include <s11n.net/s11n/s11n_node.hpp>
#include <s11n.net/s11n/traits.hpp>

namespace teeny {

	/**
	   avatar_type is the type used to represent Avatars, as defined
	   by the Teeny LRFC.
	   
	   Note that the signess of the char is unimportant because the LRFC
	*/
	typedef char avatar_type;

	/**
	   Defines the base Serialization interface for Teeny-related
	   types. Relies on libs11n's s11nlite.
	*/
	class Serializable
	{
	public:
//  		typedef s11nlite::node_type S11nNode;
//  		typedef s11nlite::node_traits NodeTraits;
 		typedef s11n::s11n_node S11nNode;
 		typedef s11n::node_traits<S11nNode> NodeTraits;

		Serializable(){}
		virtual ~Serializable(){}

		virtual bool serialize( S11nNode & ) const = 0;
		virtual bool deserialize( const S11nNode & ) = 0;
	};

	/**
	   Represents a grid-based battleground.

	   100% experimental crud at this point.
	*/
	class Board : public Serializable
	{

	public:
		typedef std::pair<int,int> coord_type;
		typedef std::map< coord_type, avatar_type > avatar_map_type;

		Board();

		Board(int w, int h, const avatar_type & fill );

		void setTerrain( int x, int y, avatar_type c );
// 		avatar_map_type & terrainMap();

		/**
		   Returns the map of coordinate points to terrain. This map
		   should contain exactly as many elements as
		   (width() * height()).
		*/
		const avatar_map_type & terrainMap() const;

		/**
		   Returns the map of coordinate points to actors.
		*/
		const avatar_map_type & actorMap() const;

		/**

		*/
		int setTerrain( int col, int row, const std::string & in, bool horiz = true );

		/**
		   Removes all actors and terrain and sets the board's
		   size to 0x0. Does not change geometry.
		*/
		void clear();

		/**
		   Returns true if this board has no terrain and no actors.
		*/
		bool empty() const;

		/** Erases map content and creates an empty map of the
		 given geometry.  The given fill char is used to fill
		 the map. The char should be ONLY a valid Terrain
		 Avatar, but that is not enforced by this function.
		*/
		void recreate( int w, int h, avatar_type fill );

		/**
		   Sends an ascii rendering of the board to the given
		   stream. Mainly a debuggering function, but also
		   useful in very rudimentary UIs.
		*/
		void dump( std::ostream & os ) const;

		/**
		   Saves the state of actors and terrain to n, using
		   libs11n conventions.
		*/
		virtual bool serialize( S11nNode & n ) const;

		/**
		   Restores the state of actors and terrain from n.
		   If this function returns false then this object
		   is in an undefined state - no telling what mix
		   of data it might have.
		*/
		virtual bool deserialize( const S11nNode & n );

		/**
		   Moves actor a to x/y. Also adds a, if needed.
		*/
		void moveActor( int x, int y, avatar_type a );

		/**
		   Removes actor a.
		*/
		bool removeActor( avatar_type a );

		/**
		   Removes the actor at x, y. It returns the actor it
		   removes, or 0 if no actor was at that coordinate.
		*/
		avatar_type removeActorAt( int x, int y );

		/**
		   Returns the terrain at the given coordinate point,
		   or 0 if none is there.
		*/
		avatar_type terrainAt( int x, int y ) const;

		/**
		   Returns the Actor Avatar at the given coordinates,
		   or 0 if none is there.
		*/
		avatar_type actorAt( int x, int y ) const;


		/**
		   Returns true if [x,y] lies within this object's
		   geometry.
		*/
		bool inBounds( int x, int y ) const;

		/**
		   Searches the board for actor A. If found, atx and
		   aty are set to it's coordinates and true is
		   returned, else false is returned.

		   The odd order of the arguments is for consistency
		   with the rest of the API: x/y args conventionally
		   come first.

		   This function should be relatively fast, as it uses
		   an internal reverse map of actors to coordinates.
		*/
		bool findActor( int & atx, int & aty, avatar_type A ) const;

		/**
		   A variant of findActor() which returns the position
		   of the given actor as a coord_type object. If A is
		   not found then the returned point will be (-1,-1),
		   which is not a legal coordinate in this Teeny
		   implementation (the LRFC leaves this point, no pun
		   intended, open).
		*/
		coord_type findActor( avatar_type A ) const;

		/**
		   A convenience overload of findActor() which does
		   not return the coordinates of the actor.
		*/
		bool containsActor( avatar_type A ) const;

		int width() const { return m_wt; }
		int height() const { return m_ht; }

		/**
x		   Rotates map clockwise, relocating all Actors and
		   Terrain, as appropriate.

		   ticks must be one of:

		   1 == 90 degrees

		   2 == 180 degrees

		   3 == 270 degrees

		   Returns true if it rotates the map, else false.

		   Achtung: only tested on a small number of
		   cases. May not work for all geometries.
		*/
		bool rotate( int ticks );

	private:
		void rebuildReverseActorMap();
		typedef std::map< avatar_type, coord_type > actor_reverse_map_type;
		// todo:
		// move private members into an opaque internal
		// struct
		int m_wt, m_ht;
		avatar_map_type m_terrain;
		avatar_map_type m_act;
		actor_reverse_map_type m_ract;

	};

	/**
	   Returns a shared Board object.
	*/
	Board & the_board();

	/**
	   Returns true if a is a valid Actor Avatar, per the Teeny
	   LRFC definition.
	 */
	bool is_valid_actor( avatar_type a );

	/**
	   Returns true if a is a valid Terrain Avatar, per the Teeny
	   LRFC definition.
	 */
	bool is_valid_terrain( avatar_type t );

	/**
	   Returns true if either of is_valid_{actor,terrain}(a)
	   return true.
	*/
	bool is_valid_avatar( avatar_type a );

	/**
	   Returns the same as std::isdigit(a).
	*/
	bool is_digit( avatar_type a );

	/**
	   For internal use: a proxy for integrating Teeny's
	   serialization API with that of libs11n. Do not use this from
	   client code.
	*/
	class SerializableAPIProxy
	{
	public:
		bool operator()( Serializable::S11nNode & n, const Serializable & s ) const
		{
			return s.serialize( n );
		}
		bool operator()( const Serializable::S11nNode & n, Serializable & s ) const
		{
			return s.deserialize( n );
		}
	};

} // namespace teeny


////////////////////////////////////////////////////////////////////////
// This reg code must unfortunately be in the header or it will never
// get template-instantiated and will be useless.
#define S11N_TYPE teeny::Serializable
#define S11N_TYPE_NAME "teeny::Serializable"
#define S11N_SERIALIZE_FUNCTOR teeny::SerializableAPIProxy
#define S11N_ABSTRACT_BASE 1
#include <s11n.net/s11n/reg_s11n_traits.hpp>

#endif // teeny_CORE_HPP_INCLUDED
