#ifndef TEENY_SERIALIZE_HPP_INCLUDED
#define TEENY_SERIALIZE_HPP_INCLUDED 1
////////////////////////////////////////////////////////////////////////
// Serialization algos/proxies for use with Teeny and libs11n.
////////////////////////////////////////////////////////////////////////
#include <utility> // make_pair()

// Requires that s11n[lite] headers be included first.
// Now pull in the proxies we will need:
#include <s11n.net/s11n/proxy/listish.hpp> // generic list algos
#include <s11n.net/s11n/proxy/std_vector.hpp>
#include <s11n.net/s11n/proxy/std_map.hpp>
#include <s11n.net/s11n/proxy/pod_char.hpp>
#include <s11n.net/s11n/proxy/pod_int.hpp>

#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // CERR


namespace teeny {


	/**
	   An algorithm to s11n a Board's terrain in a more
	   compact and human-readable/editable form than the default
	   map serialization algo does.

	   cols and rows are the geometry of the map, and (cols*rows)
	   must equal src.size(). If the geometry and map do not match
	   then false will be returned and tgt will contain property
	   ":s11n_error:" describing the error.

	   padding should be a char which the Teeny LRFC defines as
	   NOT legal for use in the source map. e.g., when serializing
	   Terrain Avatars, an alphanum is legal. NEVER use a
	   whitespace as padding, as leading/trailing spaces on
	   records may be ignored by some stream parsers. In fact, the
	   whole point of the padding char is to allow whitespace to
	   exist along the edges of Terrain data, so as to avoid
	   problems with i/o handlers stripping leading/trailing
	   whitespace in the records. (After this was written,
	   SPACE was made an illegal character for Teeny Avatars,
	   making the padding issue largely moot but still useful
	   for some file parsers.)

	   The padding chars are stripped and ignored during
	   deserialization.

	   The 'unknown' char is used as filler when src is missing an
	   entry for a given coordinate point.
	*/
	template <typename NodeType, typename MapType>
	bool serialize_coord_char_map( NodeType & tgt,
				       const MapType & src,
				       int cols,
				       int rows,
				       char padding,
				       char unknown )
	{
		typedef s11n::node_traits<NodeType> TR;
		typedef std::vector<std::string> VEC;
		int srcsz = src.size();
		//CERR << "rows="<<rows<<", cols="<<cols<<", srcsz="<<srcsz<<"\n";
		if( (rows * cols) != srcsz )
		{
			TR::set( tgt, ":s11n_error:", "Geometry hint doesn't match map content." );
			return false;
		}
		VEC vec(rows,std::string());
		TR::class_name( tgt, "terrain_map" );
		typedef typename MapType::const_iterator CIT;
		CIT it = src.begin();
		int x = 0;
		int y = 0;
		std::string srow;
		for( ; y < rows; y++ )
		{
			srow = padding; // we pad the rows so whitespace
			// as terrain isn't a problem for s11n
			// parsers.  We must use a char which Teeny
			// defines as non-Terrain (e.g., alphanumeric).
			for(x = 0 ; x < cols; x++ )
			{
				it = src.find(std::make_pair(x,y));
				if( src.end() == it )
				{
					// internal error
// 					return false;
					srow += unknown;
				}
				else
				{
					srow += (*it).second;
				}
			}
			srow += padding;
			vec[y] = srow;
		}
		TR::set( tgt, "cols", cols );
		TR::set( tgt, "rows", rows );
		return s11n::list::serialize_streamable_list( tgt, "rowdata", vec );
	}

	/**
	   De-s11n's tgt from src, which must have been serialized
	   using serialize_coord_char_map(). It sets cols and rows to the
	   numbers gleened from the serialized data in src. If it
	   returns false then tgt is in an undefined state, and should
	   not be used by client code. cols and rows are guaranteed to
	   not be changed unless this function returns true.
	*/
	template <typename NodeType, typename MapType>
	bool deserialize_coord_char_map( const NodeType & src,
					 MapType & tgt,
					 int & cols,
					 int & rows
					 )
	{
		typedef s11n::node_traits<NodeType> TR;
		uint c = TR::get( src, "cols", (uint)0 );
		uint r = TR::get( src, "rows", (uint)0 );
		if( c < 1 || r < 1 )
		{
			// map geometry not set.
			CERR << "Didn't find proper row/cols ("<<c<<","<<r<<") in map.\n";
			return false;
		}
		typedef std::vector<std::string> VEC;
		VEC vec;
		if( ! s11n::list::deserialize_streamable_list( src, "rowdata", vec ) )
		{
			CERR << "deserialize_streamable_list() failed!\n";
			return false;
		}
		if( vec.size() != r )
		{
			CERR << "vec.size() ["<<vec.size()<<"] != row count ["<<r<<"]\n";
			// number of rows in vector doesn't match the map's geometry
			return false;
		}
		typedef VEC::const_iterator VCIT;
		VCIT vit = vec.begin();
		VCIT vet = vec.end();
		std::string line;
		int atrow = 0;
		for( ; vit != vet; ++vit )
		{
			line = (*vit);
			if( line.size()-2 != c /* 2 is the 'X' padding */ ) 
			{
				CERR << "line.size()-PAD ["<<line.size()-2<<"] != cols ["<<c<<"]\n";
				// column count doesn't match the map's geometry
				return false;
			}
			for( uint atcol = 0; atcol < line.size()-2 /*-2 for padding chars*/; atcol++ )
			{
				//CERR << "atcol/atrow="<<atcol<<","<<atrow<<"\n";
				tgt[std::make_pair(atcol,atrow)] = typename MapType::mapped_type( (char)line[atcol+1] ) /*+1 is for leading pad char */;
			}
			++atrow;
		}
		cols = c;
		rows = r;
		return true;
	}


} // namespace teeny


#endif // TEENY_SERIALIZE_HPP_INCLUDED
