////////////////////////////////////////////////////////////////////////
// A demonstration application for several aspects of s11n:
//
// a) "Intrusive" serialization, in the form of member functions in
//    a base hierarchy.
//
// b) Cooperation between different Serializable hierarchies, in the form
//    of non-related Serializable types which can serialize each other.
//  
// c) The use of DLLs. Several SerializableXXX classes come with this
//    example, a couple of which we will load dynamically via DLLs.
//
// Many thanks to Mike Radford for the initial code.
////////////////////////////////////////////////////////////////////////
#include <sstream>
#include <iostream>

#include <s11n.net/s11n/s11nlite.hpp>
#include <s11n.net/s11n/s11n_debuggering_macros.hpp>

#include "SerializableBase.hpp"
#include "SerializableFinal.hpp"

int
main(int argc, char **argv)
{
	/**************************************************************
	Here we demonstrate the de/serialization of multiple
	hierarchies of Serializables. Our classes are:

	- SerializableBase is an abstract type defining a virtual
	serialization interface (two operators).

	- SerializableOne and SerializableTwo publically subclass
	SerializableBase, re-implementing the serialization operators.

	- SerializableFinal is a separate Serializable type, unrelated
	to the others but with the same serialization interface.

	This example is intended to be compiled such that:

	This file links with SerializableFinal.o and
	SerializableBase.o, and Serializable{One,Two} are built as
	DLLs named Serializable{One,Two}.so, and are in
	s11n::plugin::path() (tip: try putting them in the same dir as
	this program).

	This app must be linked with the -rdyanmic (or equivalent)
	linker flag, or classloading will not work. (Don't ask me for
	the details - i don't know them.)
	**************************************************************/


	try
	{

	/****************************************************************
	 // The "direct" way to load a class, whether from a DLL or not:
 		SerializableBase * b = s11n::cl::classload<SerializableBase>( subtype );
 		if( ! b )
 		{
 			CERR << "Failed loading class '"<<subtype<<"'\n";
 			return 1;
 		}
 		CERR << "b == " << std::hex << b << "  :)\n";
 		delete b;
 		b = 0;
	 // We comment out this example so that we don't load the DLL
	 // here (we want it to happen during deserialization, as demonstrated below).
 	 // const std::string subtype = "SerializableOne";
	****************************************************************/


		/**************************************************************
		 Now we will demonstrate how s11n can load classes
		 from DLLs, without client-side code having to know
		 it's using this feature.  The feature has a few
		 limitations, but clients can overcome most of them
		 with little effort.
		**************************************************************/

		std::ostringstream os;

		// The following is a serialized SerializableFinal object:
		os << "(s11n::parens)\n"
		   << "s11n_node=(SerializableFinal \n"
		   << "base=(SerializableOne (common 4) (num 1))" // SerializableOne should be built as SerializableOne.soa DLL
		   << ")\n";

		// Simulate a file, reading the serialized data:
		std::istringstream is( os.str() );

		// Loading the object is a one-liner:
		SerializableFinal * f = s11nlite::load_serializable<SerializableFinal>( is );
		if( ! f )
		{
			throw s11n::s11n_exception( "Deser of SerializableFinal failed!", __FILE__, __LINE__ );
		}
		s11nlite::serializer_class( "s11n::io::funxml_serializer" ); // set our preferred output format, if desired.
		CERR << "Our firest deserialized-then-reserialized object looks like:\n";

		s11nlite::save( *f, std::cout );
		delete f;

		// Another serialized SerializableFinal object:
		os.str("");
		os << "#SerialTree 1\n"
		   << "s11n_node class=SerializableFinal {\n"
		   << "base class=SerializableTwo {\n" // SerializableTwo should be built as SerializableTwo.so.
		   << "common 4\n"
		   << "}\n}\n";

		std::istringstream is2(os.str()); // using is.str(os.str()) isn't working for me???
		f = s11nlite::load_serializable<SerializableFinal>( is2 );
		if( ! f )
		{
			throw s11n::s11n_exception( "Deser of SerializableFinal failed!", __FILE__, __LINE__ );
		}

		CERR << "Our second deserialized-then-reserialized object looks like:\n";
		s11nlite::serializer_class( "s11n::io::simplexml_serializer" );
		s11nlite::save<SerializableFinal>( *f, std::cout );
		/**************************************************************
		^^^^^ Note that the explicit <SerializableFinal> is
		not strictly necessary here, because f is a concrete
		SerializableFinal object. It is good practice to be
		explicit, however, because there are cases involving
		subtypes of Serializables where implicite typing won't
		provide the expected result. See the library manual
		for full details.
		**************************************************************/

		// Let's see that again:
		s11nlite::serializer_class( "s11n::io::compact_serializer" );
		CERR << "Or, in 'compact' format:\n";
		s11nlite::save<SerializableFinal>( *f, std::cout );

		delete f;
	}
	catch( const std::exception & ex )
	{
		CERR << "EXCEPTION: " << ex.what() << "\n";
		return 1;
	}

	return 0;
}
