#ifndef s11n_DATA_NODE_IO_H_INCLUDED
#define s11n_DATA_NODE_IO_H_INCLUDED

////////////////////////////////////////////////////////////////////////
// data_node_io.hpp
// some i/o interfaces & helpers for s11n
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////


#include <string>
#include <sstream>
#include <list>
#include <map>
#include <deque>
#include <iostream>
#include <memory>// auto_ptr

#include <cassert>
#include <typeinfo>



// #include <s11n.net/cl/cllite.hpp> // for opening DLLs

#include <s11n.net/s11n/phoenix.hpp> // phoenix class

#include <s11n.net/s11n/exception.hpp>
#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // COUT/CERR
#include <s11n.net/s11n/classload.hpp> // classloader()
#include <s11n.net/s11n/serialize.hpp> // unfortunately dep
#include <s11n.net/s11n/traits.hpp> // s11n_traits & node_traits

#include <s11n.net/s11n/export.hpp> // for exporting symbols to DLL

////////////////////////////////////////////////////////////////////////////////
// NO DEPS ON s11n_node.hpp ALLOWED!
////////////////////////////////////////////////////////////////////////////////


namespace s11n {

        namespace io {

		/**
		   Returns an output stream for the given file
		   name. Caller owns the returned pointer, which may
		   be 0.

		   The returned stream supports libzl and libbz2 if your
		   libs11n is built with libs11n_zfstream support.
		*/
		std::ostream * get_ostream( const std::string name );

		/**
		   Returns an input stream for the given file
		   name. Caller owns the returned pointer, which may
		   be 0.

		   The returned stream supports libzl and libbz2 if your
		   libs11n is built with libs11n_zfstream support.

		   If ExternalData is false then name is assumed to be
		   a string containing input, and a string-reading stream
		   is returned.
		*/
		std::istream * get_istream( const std::string name, bool ExternalData = true );

                /**
                   Convenience function for grabbing the first line of a file.

                   If ExternalData == true then returns the first line of the
                   file, else returns up to the first newline of src.
                */
                std::string get_magic_cookie( const std::string & src, bool ExternalData = true );

                /**
                   Convenience function for grabbing the first line of a
                   stream.

                   Returns the first line of the given stream, or an
                   empty string on error.
                */
                std::string get_magic_cookie( std::istream & is );

                /**
                   data_node_serializer provides an interface for
                   saving/loading a given abstract data node type
                   to/from streams.

                   It is designed for containers which comply with
                   s11n's Data Node interface and conventions.


                   Conventions:

                   Must provide:

                   typedef NodeT node_type

                   Two de/serialize functions, following the
                   stream-based interface shown here (filename-based
                   variants are optional, but convenient for clients).

                */
                template <typename NodeT>
                class S11N_EXPORT_API data_node_serializer
                {
                public:
                        /**
                           The underlying data type used to store
                           serialized data.
                        */
                        typedef NodeT node_type;


                        data_node_serializer()
                        {
                                this->magic_cookie( "WARNING: magic_cookie() not set!" );
                                // ^^^ subclasses must do this.
				typedef ::s11n::node_traits<node_type> NTR;
				NTR::name( this->metadata(), "serializer_metadata" );
                                // this->metadata().name( "serializer_metadata" );
                        };
                        virtual ~data_node_serializer(){};


                        /**
                           A convenience typedef, mainly for subclasses.
                        */
                        typedef std::map<std::string,std::string> translation_map;

                        /**
                           Returns a map intended for use with
                           strtool::translate_entities().
                           
                           The default implementation returns an empty map.
                           
                           Subclasses should override this to return a translation
                           map, if they need one. The default map is empty.

                           Be aware that this may very well be called
                           post-main(), so subclasses should take that into
                           account and provide post-main()-safe maps! (Tip:
                           see phoenix::phoenix.)
                        */
                        virtual const translation_map & entity_translations() const
                        {
                                typedef ::s11n::phoenix<translation_map,data_node_serializer<node_type> > TMap;
                                return TMap::instance();
                        }



                        /**
                           Must be implemented to format node_type to the given ostream.

                           It should return true on success, false on error.

                           The default implementation always returns false.

                           Note that this function does not use
                           s11n::serialize() in any way, and is only
                           coincidentally related to it.
                        */
                        virtual bool serialize( const node_type & /*src*/, std::ostream & /*dest*/ )
                        {
                                return false;
                        }

                       /**
                           Overloaded to save dest to the given filename.

                           The default implementation treats destfile
                           as a file name and passes the call on to
                           serialize(node_type,ostream).  The output
                           file is compressed if zfstream::compression_policy()
                           has been set to enable it.

                           Returns true on success, false on error.

                           This function is virtual so that
                           Serializers which do not deal with
                           i/ostreams (e.g., those which use a
                           database connection) can override it to
                           interpret destfile as, e.g., a
                           database-related string (e.g., connection,
                           db object name, or whatever).

			   Fixed in 1.0.2: returns false when destfile
			   is empty.
                        */
                        virtual bool serialize( const node_type & src, const std::string & destfile )
                        {
				if( destfile.empty() ) return false;
                                std::ostream * os = ::s11n::io::get_ostream( destfile );
                                if( ! os ) return false;
                                bool b = this->serialize( src, *os );
                                delete( os );
                                return b;
                        }

                        /**
                           Must be implemented to parse a node_type from the given istream.

                           It should return true on success, false on error.

                           The default implementation always returns 0 and does nothing.

                           Note that this function does not use
                           s11n::deserialize() in any way, and is only
                           coincidentally related to it.
                        */
                        virtual node_type * deserialize( std::istream & )
                        {
                                return 0;
                        }


                        /**
                           Overloaded to load dest from the given filename.

                           It supports zlib/bz2lib decompression for
                           files if your s11n lib supports them.

                           This is virtual for the same reason as
                           serialize(string).

                        */
                        virtual node_type * deserialize( const std::string & src )
                        {
                                typedef std::auto_ptr<std::istream> AP;
                                AP is = AP( ::s11n::io::get_istream( src ) );
                                if( ! is.get() ) return 0;
                                return this->deserialize( *is );
                        }


                        /**
                           Gets this object's magic cookie.

                           Cookies are registered with
                           <code>class_loader< data_node_serializer<NodeType> ></code>
                           types to map files to file input parsers.
                        */
                        std::string magic_cookie() const
                        {
                                return this->m_cookie;
                        }

                protected:
                        /**
                           Sets the magic cookie for this type.
                        */
                        void magic_cookie( const std::string & c )
                        {
                                this->m_cookie = c;
                        }

                        /**
                           metadata is an experimental feature
                           allowing serializers to store arbitrary
                           serializer-specific information in their
                           data steams.
                         */
                        node_type & metadata()
                        { return this->m_meta; }
                        /**
                           A const overload of metadata().
                         */
                        const node_type & metadata() const
                        { return this->m_meta;}
                private:
                        std::string m_cookie;
                        node_type m_meta;
                }; // data_node_serializer<>

                /**

                Tries to load a NodeType object from the given
                node. It uses the cookie from the input stream and
                s11n::cl::classload<SerializerBaseType>() to find a matching
                Serializer.

                0 is returned on error, else a new pointer, which the
                caller owns.

                Achtung: the first line of input from the input stream
                is consumed by this function (to find the cookie), and
                the cookie is not passed on to the handler! The only
                reliable way around this [that i know of] is to buffer
                the whole input as a string, and i don't wanna do that
                (it's bad for massive data files).

                ACHTUNG: Only usable for loading ROOT nodes.

                Special feature:

                If the first line of the stream is
                "#s11n::io::serializer CLASSNAME" then the CLASSNAME
                token is expected to be a Serializer class name. This
                function will try to classload that object.  If
                successful it will use that type to deserialize the
                input stream. If that fails, it will return 0.
                */
                template <typename NodeType>
                NodeType *
                load_node_classload_serializer( std::istream & is )
                {
                        typedef data_node_serializer<NodeType> ST;
                        std::string cookie; // = get_magic_cookie( is );
                        // CERR << "cookie="<<cookie<<std::endl;
                        if( ! std::getline( is, cookie ) )
                        {
                                CERR << "Odd: got a null cookie from the istream.\n";
                                return 0; // happens post-main() on valid streams sometimes!?!?!
                        }

                        // CERR << "Dispatching to node loader for cookie ["<<cookie<<"]\n";
			try
			{
				typedef std::auto_ptr<ST> AP;
				AP ser;
				std::string opencmd = "#s11n::io::serializer ";
				std::string::size_type at = cookie.find( opencmd );
				if( std::string::npos == at )
				{ // try new approach, added in 1.1.0:
					opencmd = "#!/s11n/io/serializer ";
					at = cookie.find( opencmd );
				}

				if( 0 == at )
				{
					std::string dll = cookie.substr( opencmd.size() );
					//CERR << "Trying to load Serializer from cookie: " << dll << "\n";
					ser = AP( ::s11n::cl::classload<ST>( dll ) );
				}
				else
				{
					ser = AP( ::s11n::cl::classload<ST>( cookie ) );
				}
			
				if( ! (ser.get()) )
				{
					CERR << "Did not find serializer for cookie ["<<cookie<<"]."<<std::endl;
					return 0;
				}
				return ser->deserialize( is );
			}
			catch( const s11n_exception & sex )
			{
				throw sex;
			}
			catch( ... ) // todo: consider allowing ser->deserialize() to pass through exceptions
			{
				if( ::s11n::throw_policy() & ThrowOnFailedDeserialize )
				{
					throw ::s11n::s11n_exception( std::string("Stream-level deserialization failed. Cookie=")+cookie,
								      __FILE__, __LINE__ );
				}
				return 0;
			}
			return 0;
                }

                /**
                   Returns a node pointer, parsed from the given stream, using
                   <code>s11n::io::data_node_serializer<NodeType></code>
                   as the base type for looking up a stream handler.

                   ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeType>
                NodeType * load_node( std::istream & is )
                {
                        return load_node_classload_serializer< NodeType >( is );
                }

                /**
                   Overloaded form of load_node( istream ), provided for
                   convenience.

                   If ExternalData is true, input is treated as a file,
                   otherwise it is treated as a string containing input
                   to parse.

                   ACHTUNG: Only usable for loading ROOT nodes.

                   Maintenance note: ExternalData==false may be extremely
                   inefficient, as src may get copied one additional
                   time.
                */
                template <typename NodeType>
                NodeType * load_node( const std::string & src, bool ExternalData = true )
                {
                        typedef std::auto_ptr<std::istream> AP;
                        AP is = AP( ::s11n::io::get_istream( src, ExternalData ) );
                        if( ! is.get() ) return 0;
                        return load_node<NodeType>( *is );
                }

                /**
                   Tries to load a SerializableT from the given stream.
                   On success returns a new object, else 0.

                   The caller owns the returned pointer.

                   ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeT,typename SerializableT>
                SerializableT * load_serializable( std::istream & src )
                {
                        typedef std::auto_ptr<NodeT> AP;
                        AP node = AP( load_node<NodeT>( src ) );
                        if( ! node.get() )
                        {
                                CERR << "load_serializable<>(istream) Could not load a root node from the input.\n";
                                return 0;
                        }
                        return s11n::deserialize<NodeT,SerializableT>( *node );
                }

                /**
                   An overloaded form which takes an input string. If
                   ExternalData is true the string is treated as a file
                   name, otherwise it is processed as an input stream.

                   ACHTUNG: Only usable for loading ROOT nodes.
                */
                template <typename NodeT,typename SerializableT>
                SerializableT * load_serializable( const std::string & src, bool ExternalData = true )
                {
                        typedef std::auto_ptr<std::istream> AP;
                        AP is = AP( ::s11n::io::get_istream( src, ExternalData ) );
                        if( ! is.get() )
                        {
                                CERR << "load_serializable<>(string) Could not load a root node from the input.\n";
                                return 0;
                        }
                        return load_serializable<NodeT,SerializableT>( *is );
                }

        } // namespace io

} // namespace s11n

#endif // s11n_DATA_NODE_IO_H_INCLUDED
