#ifndef s11n_net_s11nlite_CLIENT_API_HPP_INCLUDED
#define s11n_net_s11nlite_CLIENT_API_HPP_INCLUDED 1
////////////////////////////////////////////////////////////////////////
//    License: Do As You Damned Well Please
//    Author: stephan@s11n.net

namespace s11nlite { /** here to please doxygen :( */ }

#include <memory> // auto_ptr
#include <iterator> // insert_interator<>
#include <functional> // for_each

#include <s11n.net/s11n/s11n_config.hpp>

#include <s11n.net/s11n/s11n.hpp> // the whole s11n framework
#include <s11n.net/s11n/io/data_node_io.hpp> // s11n::io::data_node_serializer class
#include <s11n.net/s11n/io/serializers.hpp> // create_serializer()

namespace s11nlite {

        /**
           The sharing namespace contains "sharing context" classes.
        */
        namespace sharing {
                /**
                   Internal marker class.
		*/
                struct sharing_context;
        }

	/**
	   client_api is an abstraction of the 1.0.x s11nlite
	   interface (GAWD, NOT ANOTHER ABSTRACTION LAYER!), providing
	   an interface which works with all compatible Data Node
	   types.

	   NodeType must be compatible with the s11n's "Data Node"
	   type conventions (e.g., s11n::s11n_node).

	   The intention of this class is for projects to have easy
	   access to an s11nlite-like interface for their own node
	   types. s11nlite, as of s11n 1.2, will be based off of
	   this type.

	   Another use of this class is in conjunction with
	   s11n::fac::instance_hook: clients may specialize that type
	   to make this type's instance() function return a custom
	   client_api object. The intention of this feature is to allow
	   clients to extend the s11nlite interface "from inside" while
	   allowing client code to keep using the s11nlite interface.
	   This allows, amongst other things, extending s11nlite
	   to support special i/o channels (as provided, e.g.,
	   by pclasses.com's libraries) without actually touching
	   s11nlite. See http://s11n.net/ps11n/ for where this is
	   headed...

	   Particularly observant users might notice that many of this
	   type's functions which "probably should be" const are not
	   const. This is because subclasses are encouraged to add
	   features behind the basic API, and many would not be
	   practical without a non-const object.


	   TODOs:

	   - Figure out how best to accomodate the
	   serializer_interface type as a template parameter. Right
	   now that's not feasible because the API depends too much on
	   the s11n::io namespace code.
	*/
	template <typename NodeType>
	class client_api
	{
	private:
		/** Class name of preferred serializer type. */
		std::string m_serclass;
	public:

		client_api() : m_serclass(s11n_S11NLITE_DEFAULT_SERIALIZER_TYPE_NAME)
		{
		}

		virtual ~client_api()
		{
		}

		/**
		   node_type is the type used to store/load a Serializable
		   object's data.
		*/
		typedef NodeType node_type;

		/** The s11n::node_traits type for node_type. */
		typedef s11n::node_traits<node_type> node_traits;


		/**
		   This is the base-most type of the serializers used by s11nlite clients.
		*/
		typedef s11n::io::data_node_serializer<node_type> serializer_interface;


		/**
		   Returns create_serialize( serializer_class() ).
		   The caller owns the returned pointer.
		*/
		serializer_interface * create_serializer()
		{
			return this->create_serializer( this->serializer_class() );
		}

		/**
		   Returns a new instance of the given serializer class, or 0
		   if one could not be loaded. classname must represent a subtype
		   of serializer_interface.

		   The caller owns the returned pointer.

		   You can also pass a serializer's cookie here, and that
		   should return the same thing as it's class name would.

		   The internally-supported serializes all support a "friendly
		   form" of the name, an alias registered with their
		   classloader. Passing either this name or the cookie of the
		   Serializer should return the same thing as the classname
		   itself would.

		   Subclasses may do, e.g., lookups for
		   externally-linked serializers.
		*/
		virtual serializer_interface *
		create_serializer( const std::string & classname )
		{
			return ::s11n::io::create_serializer<node_type>( classname );
		}


		/**
		   Sets the current Serializer class used by s11nlite's
		   create_serializer(). Pass it a class name, or one of
		   the convenience names, e.g.:

		   compact, funtxt, funxml, simplexml, parens
		*/
		void serializer_class( const std::string & classname )
		{
			this->m_serclass = classname;
		}



		/**
		   Gets the name of the current Serializer type.
		*/
		std::string serializer_class() const
		{
			return this->m_serclass;
		}

		/**
		   Returns true if key can be used to create a Serializer
		   object via a call to serializer_class(). Subclasses
		   may do additional work here, like look up DLLs, which is
		   why the function is not const.
		*/
		virtual bool provides_serializer( const std::string & key )
		{
			typedef ::s11n::fac::factory_mgr<serializer_interface> FacMgr;
			return FacMgr::instance().provides( key );
		}


		/**
		   See s11n::serialize().
		*/
		template <typename SerializableType>
		bool serialize( node_type & dest,
				const SerializableType & src )
		{
			return s11n::serialize<node_type,SerializableType>( dest, src );
		}

		/**
		   See s11n::serialize().
		*/
		template <typename SerializableType>
		bool serialize_subnode( node_type & dest,
					const std::string & subnodename,
					const SerializableType & src )
		{
			std::auto_ptr<node_type> n(node_traits::create());
			node_traits::name( *n, subnodename );
			if( serialize<SerializableType>( *n, src ) )
			{
				node_traits::children(dest).push_back( n.release() );
				return true;
			}
			return false;
		}

       
		/**
		   Saves the given node to the given ostream using the default
		   serializer type.

		   Returns true on success, false on error.

		   ONLY use this for saving root nodes!
		*/
		virtual bool save( const node_type & src, std::ostream & dest )
		{
			typedef std::auto_ptr<serializer_interface> AP;
			AP s = AP(this->create_serializer());
			if( ! s.get() ) return false;
			return s->serialize( src, dest );
		}

		/**
		   Saves the given node to the given filename using the default
		   serializer type.

		   Returns true on success, false on error.

		   ONLY use this for saving root nodes!

		   Subclasses are free to interpret the filename
		   however they like, e.g., as a URL or database
		   record name.
		*/
		virtual bool save( const node_type & src, const std::string & filename )
		{
			typedef std::auto_ptr<serializer_interface> AP;
			AP s = AP(this->create_serializer());
			if( ! s.get() ) return false;
			return s->serialize( src, filename );
		}


		/**
		   Saves the given Serializable to the given ostream using the default
		   serializer type.

		   Returns true on success, false on error.

		   ONLY use this for saving root nodes!
		*/
		template <typename SerializableType>
		bool save( const SerializableType & src, std::ostream & dest )
		{
			node_type n;
			if( ! this->serialize( n, src ) ) return false;
			return this->save( n, dest );
		}
		/**
		   Saves the given Serializable to the given filename using the default
		   serializer type.
        
		   Returns true on success, false on error.

		   ONLY use this for saving root nodes!
		*/
		template <typename SerializableType>
		bool save( const SerializableType & src, const std::string & dest )
		{
			typedef std::auto_ptr<std::ostream> AP;
			AP os = AP( s11n::io::get_ostream( dest ) );
			if( ! os.get() ) return 0;
			return this->save( src, *os );
		}

		/**
		   Tries to load a node from the given filename.

		   The caller owns the returned pointer.

		   Subclasses are free to interpret the filename
		   however they like, e.g., as a URL or database
		   record name.
		*/        
		virtual node_type * load_node( const std::string & src )
		{
			return s11n::io::load_node<node_type>( src );
		}


		/**
		   Tries to load a node from the given input stream.

		   The caller owns the returned pointer, which may be
		   0.

		   Only usable for loading ROOT nodes.
		*/
		virtual node_type * load_node( std::istream & src )
		{
			return s11n::io::load_node<node_type>( src );
		}



		/**
		   Returns s11n::deserialize<node_type,SerializableType>(src).

		   Caller owns the returned pointer, which may be 0.

		   Note that this function is non-const because deserialization
		   may indirectly classload other types or affect this object.
		*/
		template <typename SerializableType>
		SerializableType * deserialize( const node_type & src )
		{
			return s11n::deserialize<node_type,SerializableType>( src );
		}



		/**
		   Tries to deserialize src into target. Returns true on
		   success. If false is returned then target is not guaranteed
		   to be in a useful state: this depends entirely on the
		   object (but, it could be argued, if it was in a useful
		   state it's deserialize operator would have returned true!).
		*/
		template <typename DeserializableT>
		bool deserialize( const node_type & src, DeserializableT & target )
		{
			return s11n::deserialize<node_type,DeserializableT>( src, target );
		}


		/**
		   Exactly like deserialize(), but operates on a subnode of
		   src named subnodename. Returns false if no such file is
		   found.
		*/
		template <typename DeserializableT>
		bool deserialize_subnode( const node_type & src,
					  const std::string & subnodename,
					  DeserializableT & target )
		{
			return s11n::deserialize_subnode<
				node_type,
				DeserializableT>( src, subnodename, target );
		}

		/**
		   Exactly like deserialize(), but operates on a subnode of
		   src named subnodename. Returns 0 if no such file is
		   found.
		*/
		template <typename DeserializableT>
		DeserializableT * deserialize_subnode( const node_type & src,
						       const std::string & subnodename )
		{
			return s11n::deserialize_subnode<
                        node_type,
				DeserializableT>( src, subnodename );
		}


		/**
		   Tries to load a data_node from src, then deserialize that
		   to a SerializableType.
		*/
		template <typename SerializableType>
		SerializableType * load_serializable( std::istream & src )
		{
			typedef std::auto_ptr<node_type> AP;
			AP n = AP( this->load_node( src ) );
			if( ! n.get() ) return 0;
			return this->deserialize<SerializableType>( *n );
		}

		/**
		   Overloaded form which takes a file name.

		   See s11n::io::get_istream() for a description of
		   the AsFile parameter.
		*/
		template <typename SerializableType>
		SerializableType * load_serializable( const std::string & src, bool AsFile = true )
		{
			typedef std::auto_ptr<std::istream> AP;
			AP is = AP( s11n::io::get_istream( src, AsFile ) );
			if( ! is.get() ) return 0;
			return this->load_serializable<SerializableType>( *is );
		}

		/**
		   See s11n::clone().
		*/
		template <typename SerializableType>
		SerializableType * clone( const SerializableType & tocp )
		{
			node_type node;
			if( ! this->serialize( node, tocp ) ) return 0;
			return this->deserialize<SerializableType>( node );
		}

		/**
		   See s11n::s11n_cast().
		*/
		template <typename Type1, typename Type2>
		bool cast( const Type1 & t1, Type2 & t2 )
		{
			node_type n;
			return this->serialize( n, t1 ) && this->deserialize( n, t2 );
		}

		/**
		   Returns a shared instance of this class.

		   TODO: fix the shared instancing such that clients can swap out
		   the internal instance with their own.
		*/
		static client_api<node_type> &
		instance()
		{
// 			return ::s11n::fac::instance_hook< client_api<node_type> >::instance();
			return ::s11n::phoenix< client_api<node_type> >::instance();
		}

	}; // client_api<> class 

} // namespace s11nlite

#endif // s11n_net_s11nlite_CLIENT_API_HPP_INCLUDED
