////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////
#ifndef s11n_net_s11n_v1_1_LIST_HPP_INCLUDED
#define s11n_net_s11n_v1_1_LIST_HPP_INCLUDED 1

#include <stdio.h> // snprintf()

#include <list>
#include <vector>
#include <set>
#include <memory>
#include <iterator> // insert_iterator
#include <algorithm> // for_each()

#include <s11n.net/strtool/strtool.hpp> // to/from string
#include <s11n.net/s11n/traits.hpp> // node_traits<>
#include <s11n.net/s11n/serialize.hpp> // core serialize funcs
#include <s11n.net/s11n/algo.hpp> // dump_node_debug()
#include <s11n.net/s11n/abstract_creator.hpp> // abstract_creator class

namespace s11n {


	/**
           The s11n::list namespace defines functors and algorithms for
           working with std::list/vector-style containers.
        */
        namespace list {

                /**
                   serialize_list() supports list/vector-like types containing
                   any Serializable type. Serializes (src.begin(),src.end()]
                   to dest. Each item has a node name of that given to the
                   ctor or an unspecified dummy value.
           
                   Returns true on success. If false is returned then dest is
                   in an undefined state: some number of serializations may
                   have succeeded before the failure. This operation stops
                   processing at the first serialization error.

		   If serialization of a child fails, the child is not
		   added to dest and any exception is propagated back
		   to the caller.

                   Compatible ListTypes must support:
           
                   - a value_type typedef describing the type of it's contents.

                   - push_back( value_type )

                   - const_iterator typedefs.

                   Restrictions on value_type's type:

                   - Must be a Serializable. This includes any i/o streamable
                   type which has a proxy installed by s11n (includes all PODs
                   and std::string by default).

                   - may be a pointer type.

                   Some s11n-side requirements:

                   - ListType must be registered with the ListType classloader.

                   ACHTUNG: never pass the same destination container to 
                   this function more than once or you will get duplicate and/or
                   incorrect data.
                */
                template <typename NodeType, typename SerType>
                bool serialize_list( NodeType & dest, const SerType & src )
                {
                        typedef ::s11n::node_traits<NodeType> TR;
			typedef ::s11n::s11n_traits<SerType> STR;
                        TR::class_name( dest, STR::class_name(&src) );
                        typename SerType::const_iterator it = src.begin();
                        // reminder: (*it) may be a pointer- or a value-type,
                        // as long as it's a Serializable.
                        for( ; src.end() != it; ++it )
                        {
				std::auto_ptr<NodeType> ch( TR::create() );
                                if( serialize( *ch, *it ) )
                                {
					TR::children(dest).push_back( ch.release() );
                                        continue;
                                }
				CERR << "serialize_list: a child failed to serialize: " << TR::name(*ch) << " @ " << std::hex << ch.get() << "\n";
				s11n::dump_node_debug( *ch, std::cerr );
				return false;
                        }
                        return true;
                }

                /**
                   Identical to the two-argument form of serialize_list(), but
                   serializes src into a subnode of dest, named subnodename.

		   If serialization into dest child fails, the child
		   node is not added to dest and the error (possibly
		   an exception) is propagated back to the caller.
                */
                template <typename NodeType, typename SerType>
                bool serialize_list( NodeType & dest,
                                     const std::string & subnodename,
                                     const SerType & src )
                {
                        typedef node_traits<NodeType> TR;
                        std::auto_ptr<NodeType> ch( TR::create() );
                        TR::name( *ch, subnodename );
                        if( serialize_list<NodeType,SerType>( *ch, src ) )
                        {
				TR::children(dest).push_back( ch.release() );
                                return true;
                        }
                        return false;
                }

                /**
                   For each (src.children().begin(),end()] an object
                   of type SerType::value_type is created, deserialized,
                   and is added to dest via push_back( item ).

                   See serialize_list() for the list type requirements.

                   If SerType::value_type is a pointer type then dest owns any
                   newly-created pointers, and it's owner (or the container
                   itself) is responsible for cleaning them up. (Non-pointer
                   types need no explicit clean-up, of course.)

                   Returns true if all deserializations succeed.  Stops
                   processing and returns false at the first error, in which
                   case dest is in an undefined state: some children may or
                   may not have been successfully deserialized into it. The
                   immediate implication of this is that it will fail if src
                   contains any other children than the type which dest
                   expects!

		   Exceptions: this function throws if an underlying
		   call to deserialize() throws. If it throws then
		   the currently-deserializing node is not added to the
		   dest list and it is deallocated if needed.
                */
                template <typename NodeType, typename SerType>
                bool deserialize_list( const NodeType & src, SerType & dest )
                {
                        // note: some (most) of the odd-looking code
                        // here is to allow this func to support both
                        // pointer- and non-pointer value_types in SerType.
                        // The other half is debug code.
                        typedef typename SerType::value_type VT;
                        typedef s11n::abstract_creator<VT> ACVT;
                        typedef typename NodeType::child_list_type::const_iterator CHIT;
                        typedef node_traits<NodeType> TR;

                        const NodeType * nch = 0;
                        CHIT it = TR::children(src).begin();
                        CHIT et = TR::children(src).end();
                        ::s11n::object_reference_wrapper<SerType> dwrap(dest);
                        VT ser; // reminder: might be a pointer type
                        std::string implclass;
                        static const char * errprefix = "deserialize_list(node,list) ";
#define ERRPRE if(0) CERR << errprefix << "srcnode="<<std::dec<<&src << ": "
                        ERRPRE <<"starting... dest list size="<<dest.size()<<". src child count="<<TR::children(src).size()<<"\n";
                        int bogocount = 0;
                        for( ; et != it; ++it )
                        {
                                nch = *it;
                                if( ! nch )
                                {

                                        ERRPRE << " problemo: got a null child entry. "
                                             << "Returning false and leaving list in it's current state.\n";
                                        return false;
                                }
                                implclass = TR::class_name(*nch);
				try
				{
					// instantiate a new child object to deserialize to...
					ERRPRE << errprefix << " loop count="<<bogocount<<", implclass="<<implclass<<"\n";
					if( ! ACVT::create( ser, implclass ) )
					{
						CERR << "Internal error: abstract_creator<> "
						     << "could not create a new object of type '"
						     << implclass<<"'!\n";
						return false;
					}
					// deserialize the data into the new child...
					if( ! deserialize( *nch, ser ) )
					{
						CERR << "deserialize_list(): deser of a child failed!\n";
						CERR << "name="<< TR::name(*nch)<< ". implclass="<< implclass<<" @ " << std::hex<<nch <<"\n";
						s11n::dump_node_debug( *nch, std::cerr );
						ACVT::release( ser );
						return false;
					}
				}
				catch(...)
				{
					ACVT::release( ser );
					throw;
				}
				// insert the child into the destination Deserializable...
                                dwrap().insert( dwrap().end(), ser );
                                if( 0 )
                                {
                                        ERRPRE << "child node="<<std::hex<<nch<<std::dec<<", deserialized to ser. implclass="<<implclass<<". It's data node says:\n";
                                        ::s11n::dump_node_debug( *nch, std::cerr );
                                }
                        }
                        if( 0 ) {
                                CERR << "deserialize_list(). INPUT Node tree looks something like:\n";
                                ::s11n::dump_node_debug( src, std::cerr );
                        }
                        ERRPRE <<"returning true. dest list size="<<dest.size()<<"...\n";
#undef ERRPRE
                        return true;
                }

                /**
                   Identical to the two-argument form of deserialize_list(), but
                   deserializes a subnode of src, named subnodename. If no such
                   child is found in src then false is returned.
                */
                template <typename NodeType, typename SerType>
                bool deserialize_list( const NodeType & src,
                                       const std::string & subnodename,
                                       SerType & dest )
                {
                        const NodeType * ch = s11n::find_child_by_name( src, subnodename );
                        if( ! ch ) return false;
                        return deserialize_list<NodeType,SerType>( *ch, dest );
                }        

                /**
                   serialize_streamable_list serializes objects of type
                   <tt>std::list&lt;X&gt;</tt> (and compatible list types,
                   such as std::vector). It stores them in such a way that
                   they can be loaded into any compatible container via
                   deserialize_streamable_list().

                   Conventions:

                   - NodeType must support a generic set(Key,Val) function, as
                   implemented by the s11n::data_node interface.

                   - ListType must conform to std::list conventions and it's
                   value_type must be a non-pointer type which is
                   i/ostreamable (this includes all PODs and
                   std::string). Pointers are not supported by
                   this function.

                   ACHTUNG:

                   - Never call this on a node for which you store other
                   properties, as deserialize_streamable_list() will consume them
                   if you use that function.

                   - This function sets dummy property keys, both to please
                   the conventions of keys having non-empty values and to keep
                   the list in the proper order. It uses keys which should be
                   portable to, e.g., standard XML.

                   Always returns true - the bool return value is for API consistency.

                   ACHTUNG 2:

                   - The return type of this function was changed from size_t
                   to bool in version 0.9.10.
                */
                template <typename NodeType, typename ListType>
                bool serialize_streamable_list( NodeType & dest, const ListType & src )
                {
                        typedef typename ListType::const_iterator CIT;
                        typedef node_traits<NodeType> TR;
			typedef ::s11n::s11n_traits<ListType> STR;
                        TR::class_name( dest, STR::class_name(&src) );
                        size_t i = 0;
                        // We zero-pad all keys to be the same length, so that
                        // they will be kept in the proper order in the
                        // target's property list (annoying, but necessary).
                        // We prepend a non-numeric char to make this output
                        // compatible with standard XML
                        static const int bsize = 10;
                        char num[bsize];
                        char fmt[bsize];
                        size_t sz = src.size();
                        int places = 1; // # of digits to use
                        for( ; sz >= 0x0f; sz = (size_t)(sz/0x0f)) { ++places; }
                        snprintf( fmt, bsize, "x%%0%dx", places );
                        CIT it = src.begin();
                        for( ; src.end() != it; ++it )
                        {
                                snprintf( num, bsize, fmt, i );
                                ++i;
                                TR::set( dest, num, (*it) );
                        }
                        return true;
                }


                /**
                   Identical to serialize_streamable_list(dest,src), but
                   creates a subnode in dest, named subnodename, where the
                   data is stored.

                   ACHTUNG:

                   - The return type of this function was changed from size_t
                   to bool in version 0.9.10.
                */
                template <typename NodeType, typename ListType>
                bool serialize_streamable_list( NodeType & dest,
                                                const std::string & subnodename,
                                                const ListType & src )
                {
                        NodeType & n = s11n::create_child( dest, subnodename );
                        return serialize_streamable_list<NodeType,ListType>( n, src );
                }


                /**
                   Deserializes dest from src. It reads in all properties from
                   src, ignoring their keys and taking only their values. This
                   is suitable for use with the result of a
                   serialize_streamable_list() call. See that function for more
                   information, including the conventions which must be
                   supported by NodeType and ListType.
           
                   Always returns true - the bool return value is for API consistency.

                   ACHTUNG:

                   - The return type of this function was changed from size_t
                   to bool in version 0.9.10.

                */
                template <typename NodeType, typename ListType>
                bool deserialize_streamable_list( const NodeType & src, ListType & dest )
                {
                        typedef node_traits<NodeType> TR;
                        typedef typename ListType::value_type VT;
			typedef typename TR::property_map_type PMT;
			typedef typename PMT::const_iterator CIT;
                        CIT it = TR::properties(src).begin();
			CIT et = TR::properties(src).end();
                        VT defaultval;
                        for( ; et != it; ++it )
                        {
                                dest.insert( dest.end(),
                                             ::strtool::from<VT>( (*it).second, defaultval ) );
                        }
                        return true;
                }


                /**
                   Identical to deserialize_streamable_list(), but looks for
                   the data in a subnode of src named subnodename.

                   Returns false if no child could be found.

                   ACHTUNG:

                   - The return type of this function was changed from size_t
                   to bool in version 0.9.10.

                */
                template <typename NodeType, typename ListType>
                bool deserialize_streamable_list( const NodeType & src,
                                                  const std::string & subnodename,
                                                  ListType & dest )
                {
                        const NodeType * ch = s11n::find_child_by_name( src, subnodename );
                        if( ! ch ) return false;
                        return deserialize_streamable_list<NodeType,ListType>( *ch, dest );
                }



                /**
                   list_serializable_proxy is a functor for de/serializing lists
                   of Serializables.
                */
                class list_serializable_proxy
                {
                public:
                        list_serializable_proxy()
                        {}

                        /**
                           see serialize_list().

                        */
                        template <typename NodeType, typename SerType>
                        bool operator()( NodeType & dest, const SerType & src ) const
                        {
                                return serialize_list( dest, src );
                        }

                        /** see deserialize_list(). */
                        template <typename NodeType, typename SerType>
                        bool operator()( const NodeType & src, SerType & dest ) const
                        {
                                return deserialize_list( src, dest );
                        }
                };

                /**
                   streamable_list_serializable_proxy is a functor for de/serializing lists
                   of i/ostreamable Serializables (e.g., PODs).
                */
                class streamable_list_serializable_proxy
                {
                public:
                        streamable_list_serializable_proxy()
                        {}

                        /**
                           see serialize_streamable_list().

                        */
                        template <typename NodeType, typename SerType>
                        bool operator()( NodeType & dest, const SerType & src ) const
                        {
                                return serialize_streamable_list( dest, src );
                        }

                        /** see deserialize_streamable_list(). */
                        template <typename NodeType, typename SerType>
                        bool operator()( const NodeType & src, SerType & dest ) const
                        {
                                return deserialize_streamable_list( src, dest );
                        }
                };



        } // namespace list
} // namespace s11n


#endif // s11n_net_s11n_v1_1_LIST_HPP_INCLUDED
