////////////////////////////////////////////////////////////////////////
//
////////////////////////////////////////////////////////////////////////
#ifndef s11n_net_s11n_v1_1_MAP_HPP_INCLUDED
#define s11n_net_s11n_v1_1_MAP_HPP_INCLUDED 1

#include <map>
#include <list>
#include <utility> // pair

#include <iterator> // insert_iterator
#include <algorithm> // for_each()

#include <s11n.net/s11n/variant.hpp> // to/from string
#include <s11n.net/s11n/serialize.hpp> // core serialize funcs
#include <s11n.net/s11n/algo.hpp> // dump_node_debug()
#include <s11n.net/s11n/abstract_creator.hpp> // abstract_creator class

namespace s11n {



        /**
           The s11n::map namespace defines functors and algorithms for
           working with std::map and std::pair containers.
        */
        namespace map {

        
                /**

                serialize_streamable_map() is intended for use as a
                serialization proxy for the s11n framework. It
                de/serializes objects of type <tt>std::map&lt;X,Y&gt;</tt>
                into a destination node.

                NodeType must support:

                set( const std::string & key, const std::string & value )

                - MapType must conform to std::map conventions and
                it's key_type and mapped_type must both be Value Types
                which are i/ostreamable (this includes all PODs and
                std::string). Pointers as keys or values are not
                supported by this functor.

                Always returns true.

                ACHTUNG: return type changed in 0.9.12.
                */

                template <typename NodeType, typename MapType>
                bool serialize_streamable_map( NodeType & dest, const MapType & src )
                {
			typedef ::s11n::node_traits<NodeType> TR;
                        typedef typename MapType::value_type VT;
                        typedef typename VT::first_type FT;
                        typedef typename VT::second_type ST;
                        VT p;
                        typename MapType::const_iterator it = src.begin();
			::s11n::variant v1, v2;
                        for( ; src.end() != it; ++it )
                        {
				v1 = (*it).first;
				v2 = (*it).second;
                                TR::set(dest, v1.str(), v2.str() );
                        }
                        return true;
                }

                /**
                   Exactly like serialize_streamable_map(dest,src) except that a subnode,
                   named subnodename, of dest is created to store the data. 

                   ACHTUNG: return type changed in 0.9.12.
                */
                template <typename NodeType, typename MapType>
                bool serialize_streamable_map( NodeType & dest,
                                                 const std::string & subnodename,
                                                 const MapType & src )
                {
                        return serialize_streamable_map( s11n::create_child( dest, subnodename ), src );
                }

                /**
                   This is the converse of serialize_streamable_map(). It tries to
                   read in all properties stored in src and stick them into
                   dest.

                   NodeType must support begin() and end() and they must
                   return iterators to pair&lt;X,Y&gt;, where X and Y must
                   meet the same requirements as the key and value types for
                   MapType in serialize_streamable_map(). MapType must support:

                   void insert( MapType::value_type );

                   (Duh.)

                   Always returns true.

                   ACHTUNG: return type changed in 0.9.12.
                   
                */
                template <typename NodeType, typename MapType>
                bool deserialize_streamable_map( const NodeType & src, MapType & dest )
                {
			typedef ::s11n::node_traits<NodeType> NTR;
                        typedef typename MapType::value_type VT; // pair
                        typedef typename VT::first_type T1;
                        typedef typename VT::second_type T2;

                        const T1 default1 = T1();
                        const T2 default2 = T2();
                        typedef typename NTR::property_map_type PMT;
			typedef typename PMT::const_iterator CIT;
                        CIT it = NTR::properties(src).begin();
			CIT et = NTR::properties(src).end();
			::s11n::variant v1, v2;
                        for( ; et != it; ++it )
                        {
				v1 = (*it).first;
				v2 = (*it).second;
                                dest.insert( std::make_pair( v1.template cast_to<T1>( default1 ),
							     v2.template cast_to<T2>( default2 )
							     ) );
                        }
                        return true;
                }

                /**
                   Exactly like deserialize_streamable_map(dest,src) except
                   that a subnode of dest, named subnodename, is sought to
                   pull the data from.
                */
                template <typename NodeType, typename MapType>
                bool deserialize_streamable_map( const NodeType & src,
                                                   const std::string & subnodename,
                                                   MapType & dest )
                {
                        const NodeType * ch = s11n::find_child_by_name( src, subnodename );
                        if( ! ch ) return false;
                        return deserialize_streamable_map<NodeType,MapType>( *ch, dest );
                }


                /**
                   Serializes a std::pair-compatible type into a "custom"
                   format, suitable for saving pairs in standard XML
                   (de/serialize_streamable_map() can't do this when keys are
                   not valid XML keys, e.g., numeric). Use
                   deserialize_streamable_pair() to decode the data.

                   The destination node gets these two properties:

                   - first = src.first

                   - second = src.second

                   PairType must comply with:

                   - first/second types must be i/o streamable (i.e.,
                   convertable to strings).

                   Returns true on success... and never fails. Honestly. It'll
                   fail at compile-time if it's going to fail.


                   use deserialize_streamable_pair() to convert them back to pairs,
                   or fish out the "first" and "second" properties manually.
                */
                template <typename NodeType, typename PairType>
                bool serialize_streamable_pair( NodeType & dest, const PairType & src )
                {
                        typedef s11n::node_traits<NodeType> TR;
			typedef ::s11n::s11n_traits<PairType> STR;
                        TR::class_name( dest, STR::class_name(&src) );
                        typedef typename PairType::first_type FT;
                        typedef typename PairType::second_type ST;
                        dest.set( "first", src.first );
                        dest.set( "second", src.second );
                        return true;
                }

                /**
                   The quasi-counterpart of serialize_streamable_pair(). It's
                   non-conventional args and return type are a result of
                   map::value_type having a const .first element, which
                   prohibits us assigning to it. See deserialize_pair() for
                   more info on that.
                */
                template <typename PairType, typename NodeType>
                PairType deserialize_streamable_pair( const NodeType & src  )
                {
                        typedef typename PairType::first_type T1;
                        typedef typename PairType::second_type T2;
                        T1 default1 = T1();
                        T2 default2 = T2();
                        return std::make_pair( src.get( "first", default1 ),
                                               src.get( "second", default2 )
                                               );
                }


                /**
                   Similar to serialize_streamable_map(), but puts each key/value
                   pair into it's own node, using serialize_streamable_pair(). The
                   end effect is that it's output is more verbose, but may be
                   compatible with more file formats, regardless of the actual
                   key type. e.g., numeric keys are supported by standard XML
                   (though they are by the s11n XML parsers), and this algorithm
                   structures the data such that this is not a problem.

                   Returns the number of pairs stored.

                   MapType must meet these conditions:

                   value_type must be a pair containing i/ostreamable types
                   (e.g. PODs/strings).

                   ACHTUNG: return type changed in 0.9.12.

                */
                template <typename NodeType, typename MapType>
                bool serialize_streamable_map_pairs( NodeType & dest, const MapType & src )
                {
                        typedef s11n::node_traits<NodeType> TR;
			typedef ::s11n::s11n_traits<MapType> STR;
                        TR::class_name( dest, STR::class_name(&src) );
                        typedef typename MapType::value_type VT;
                        typedef typename VT::first_type FT;
                        typedef typename VT::second_type ST;
                        typename MapType::const_iterator it = src.begin();
                        for( ; src.end() != it; ++it )
                        {
                                if( ! serialize_streamable_pair( create_child( dest, "pair" ), *it ) )
                                {
                                        return false;
                                }
                        }
                        return true;
                }

                /**
                   The load-time counterpart to serialize_streamable_map_pairs().

                   ACHTUNG: return type changed in 0.9.12.

                   If it returns false, no appropriate child nodes
                   were found or one of them failed to deserialize.
                */
                template <typename NodeType, typename MapType>
                bool deserialize_streamable_map_pairs( const NodeType & src, MapType & dest )
                {
                        typedef typename MapType::value_type VT; // pair
                        typedef typename VT::first_type T1;
                        typedef typename VT::second_type T2;
                        typedef std::list<const NodeType *> ChList;
                        typedef typename ChList::const_iterator ChIt;

                        static const T1 default1 = T1();
                        static const T2 default2 = T2();

                        ChList namedch;
                        if( 0 == ::s11n::find_children_by_name( src, "pair", namedch ) ) return false;

                        const NodeType * ch = 0;
                        ChIt it = namedch.begin();
                        VT p = VT(default1,default2);
                        for( ; namedch.end() != it; ++it )
                        {
                                ch = *it;
                                dest.insert( deserialize_streamable_pair<VT>( *ch ) );
                        }
                        return true;
                }


                /**
                   serialize_pair() can serialize any std::pair type which
                   meets these conditions:

                   - PairType's first_type and second_type types must both be
                   Serializables. They may be pointer or value types.

		   If serialization of one child fails, the whole
		   process fails and neither child is added to the
		   dest node. The error, possibly an exception, is
		   propagated back to the caller.

                   ACHTUNG: never pass the same destination container
                   more than once or you will get duplicate and/or
                   incorrect data.
                */
                template <typename NodeType, typename PairType>
                bool serialize_pair( NodeType & dest, const PairType & src )
                {
                        typedef s11n::node_traits<NodeType> NT;
			typedef ::s11n::s11n_traits<PairType> STR;
                        NT::class_name( dest, STR::class_name(&src) );
                        std::auto_ptr<NodeType> ch( NT::create() );
			NT::name(*ch,"first");
                        if( ! serialize( *ch, src.first ) )
                        {
                                potentially_throw( ThrowOnFailedSerialize, "serialize_pair: first child failed serialize!" );
				return false;
                        }
                        std::auto_ptr<NodeType> ch2( NT::create() );
			NT::name(*ch2,"second");
                        if( ! serialize( *ch2, src.second ) )
                        {
                                potentially_throw( ThrowOnFailedSerialize, "serialize_pair: second child failed serialize!" );
                                return false;
                        }
			NT::children(dest).push_back( ch.release() );
			NT::children(dest).push_back( ch2.release() );
                        return true;
                }

                /**
                   The counterpart to serialize_pair().

                   Note: std::map&lt;X,Y&gt;::value_type is not the same as pair&lt;X,Y&gt;,
                   but is pair&lt;const X,Y&gt;, so you cannot simply iterate over a map and
                   pass each pair to this function, because this function cannot assign
                   to the first element of such a pair.

		   Exceptions: this function throws if an underlying
		   call to deserialize() throws. If it throws then
		   the currently-deserializing node is not added to the
		   dest pair and it is deallocated if needed.
                */
                template <typename NodeType, typename PairType>
                bool deserialize_pair( const NodeType & src, PairType & dest )
                {
                        typedef ::s11n::node_traits<NodeType> NTR;
                        //CERR << "deserialize_pair: deserialize " << src.impl_class() << "\n";
                        typedef typename PairType::first_type FT;
                        typedef typename PairType::second_type ST;
                        const NodeType * ch = 0;
                        typedef ::s11n::pointer_stripper<PairType> PS;

                        // Note: the abstract_creator code below is simply
                        // to treat pointer and value types identically
                        // with this same code base. See it's docs for what
                        // it does (or doesn't do, in the case of reference
                        // types).

                        typedef s11n::abstract_creator<FT> AC1st;
                        typedef s11n::abstract_creator<ST> AC2nd;
                        //////////////////////////////////////// .first
                        ch = find_child_by_name( src, "first" );
                        if( ! ch )
                        {
                                CERR << "deserialize_pair: deserialize: no 'first' node found!\n";
                                return false;
                        }
                        std::string implclass = NTR::class_name( *ch );
                        FT f; // value of .first
                        if( ! AC1st::create( f, implclass ) )
                        {
                                CERR << "Internal error: could not create first element."
                                     <<"type='"<<NTR::class_name( *ch )<<"'!\n";
                                return false;
                        }
			try
			{
				if( ! deserialize( *ch, f ) )
				{
					CERR << "pair deserialize(..., first ) failed.\n";
					AC1st::release( f );
					return false;
				}
			}
			catch(...)
			{
				AC1st::release(f);
				throw;
			}
                        //////////////////////////////////////// .second:
                        ST s; // value of .second
                        ch = find_child_by_name( src, "second" );
                        if( ! ch )
                        {
                                CERR << "deserialize_pair: deserialize: no 'second' node found!\n";
                                AC1st::release( f );
                                return false;
                        }
                        implclass = NTR::class_name( *ch );
                        if( ! AC2nd::create( s, implclass ) )
                        {
                                CERR << "Internal error: could not create second element."
                                     <<"type='"<<implclass<<"'!\n";
                                AC1st::release( f );
                                return false;
                        }
			try
			{
				if( ! deserialize( *ch, s ) )
				{
					CERR << "deserialize_pair(): deserialize(node, second) failed. ";
					// "Node name=["<<NTR::name(*ch)<<"] class=["<<implclass<<"].\n";
					::s11n::dump_node_debug( *ch, std::cerr );
					AC1st::release( f );
					AC2nd::release( s );
					return false;
				}
			}
			catch(...)
			{
				AC1st::release(f);
				AC2nd::release(s);
				throw;
			}
                        dest.first = f;
                        dest.second = s;
                        return true;
                }



                /**
                   pair_serializable_proxy is a Serializable Proxy for std::pairs.
                */
                struct pair_serializable_proxy
                {
                        /**
                           See serialize_pair().
                        */
                        template <typename NodeType, typename PairType>
                        bool operator()( NodeType & dest, const PairType & src ) const
                        {
                                return serialize_pair( dest, src );
                        }
                        /**
                           See deserialize_pair().
                        */
                        template <typename NodeType, typename PairType>
                        bool operator()( const NodeType & src, PairType & dest ) const
                        {
                                return deserialize_pair( src, dest );
                        }
                };

        

                /**
                   Serialize the given map into dest. MapType's pairs must be
                   Serializable and must contain Serializable types, but their
                   "pointerness" is irrelevant.

                   ACHTUNG: never pass the same destination container to the
                   operators more than once or you will get duplicate and/or
                   incorrect data.

                   See deserialize_map() for important info.

		   If serialization of a child fails, the child is not
		   added to dest and any exception is propagated back
		   to the caller.
                */
                template <typename NodeType, typename MapType>
                bool serialize_map( NodeType & dest, const MapType & src )
                {
                        typedef typename MapType::const_iterator CIT;
                        typedef node_traits<NodeType> TR;
			typedef ::s11n::s11n_traits<MapType> STR;
                        TR::class_name( dest, STR::class_name(&src) );

                        CIT b = src.begin(), e = src.end();
                        for( ; e != b; ++b )
                        {
                                std::auto_ptr<NodeType> ch( TR::create() );
                                TR::name(*ch, "pair" );
                                if( serialize_pair( *ch, *b ) )
                                {
					dest.children().push_back( ch.release() );
					continue;
                                }
				CERR << "serialize_map: child failed serialize.\n";
				return false;
                        }
                        return true;
                }

                /**
                   Identical to the two-argument form, but creates a
                   subnode of dest, named subnodename, and serializes
                   to that node.

		   If serialization into dest child fails, the child
		   node is not added to dest and the error (possibly
		   an exception) is propagated back to the caller.
                */
                template <typename NodeType, typename MapType>
                bool serialize_map( NodeType & dest,
                                    const std::string & subnodename,
                                    const MapType & src )
                {
                        typedef node_traits<NodeType> TR;
                        std::auto_ptr<NodeType> ch( TR::create() );
                        TR::name( *ch, subnodename );
                        if( serialize_map<NodeType,MapType>( *ch, src ) )
                        {
				TR::children(dest).push_back( ch.release() );
				return true;
                        }
			return false;
                }

                /**
                   The counterpart of serializer_map(), deserializes src into the
                   given map. MapType must be Serializable and contain pairs
                   which themselves are Serializables... ad inifinitum..

                   Minor caveat:

                   This operation will only work with maps containing std::pair
                   types, not map-like classes which use a different pair
                   type. :( The reason is that map&lt;X,Y&gt;::value_type is
                   not a pair of (X,Y), but (const Y,Y), which means we cannot
                   use the map's value_type for a deser operation because we
                   cannot assign to it's .first element (i.e., can't
                   deserialize it). To get around that we "manually" create a
                   new std::pair type using map's key_type and mapped_type
                   typedefs, which "loses" the constness for use so we can
                   assign to the first_type during deserialization, and then
                   insert that pair into the deserializing map.


                */
                template <typename NodeType, typename MapType>
                bool deserialize_map( const NodeType & src, MapType & dest )
                {
                        typedef node_traits<NodeType> TR;
                        typedef typename NodeType::child_list_type::const_iterator CIT;
                        //typedef typename SerializableType::value_type VT;
                        // ^^^ no, because VT::first_type is const!
                        // Thus we hand-create a compatible pair type:
                        typedef typename MapType::key_type KType;
                        typedef typename MapType::mapped_type VType;
                        typedef std::pair< KType, VType > PairType;
                        PairType pair;
                        CIT b = TR::children(src).begin(), e = TR::children(src).end();
                        const NodeType *ch = 0;
                        for( ; e != b ; ++b )
                        {
                                ch = *b;
                                if( ! deserialize_pair( *ch, pair ) )
                                {
                                        CERR << "deserializer_map: child failed deser.\n";
                                        return false;
                                }
                                dest.insert( pair );
                        }
                        return true;
                }


                /**
                   Identical to the two-argument form, but tries to deserialize
                   from a subnode of src named subnodename. If no such node is found
                   then false is returned.
                */
                template <typename NodeType, typename MapType>
                bool deserialize_map( const NodeType & src,
                                      const std::string & subnodename,
                                      MapType & dest )
                {
                        const NodeType * ch = s11n::find_child_by_name( src, subnodename );
                        if( ! ch ) return false;
                        return deserialize_map<NodeType,MapType>( *ch, dest );
                }


                /**
                   A proxy which can serialize std::maps which contain Streamable
                   Types.

                   It uses de/serialize_streamable_map(), so see those
                   functions for details.
                */
                struct streamable_map_serializable_proxy
                {
                        /**
                           Serializes src to dest.

                           ACHTUNG: never pass the same destination container
                           to this operator more than once or you will get
                           duplicate and/or incorrect data.
                        */
                        template <typename NodeType, typename SerializableType>
                        bool operator()( NodeType & dest , const SerializableType & src ) const
                        {
                                //s11n::serialize_streamable_map_pairs( dest, src );
                                serialize_streamable_map( dest, src );
                                return true;
                        }

                        /**
                           Deserializes dest from src.
                        */
                        template <typename NodeType, typename SerializableType>
                        bool operator()( const NodeType & src , SerializableType & dest ) const
                        {
                                //s11n::deserialize_streamable_map_pairs( src, dest );
                                deserialize_streamable_map( src, dest );
                                return true;
                        }
                };



                /**
                   map_serializable_proxy is a Serialization Proxy for std::maps.

                   See de/serialize_map(): this functor simply wraps those.
                */
                struct map_serializable_proxy
                {

                        /**
                           Serializes src into dest. Returns true on success,
                           false on error. Uses serialize_map(), so see that
                           function for details.
                        */
                        template <typename NodeType, typename MapType>
                        bool operator()( NodeType & dest , const MapType & src ) const
                        {
                                return serialize_map( dest, src );
                        }
                        /**
                           Deserializes src into dest. Returns true on
                           success, false on error. Uses serialize_map(), so
                           see that function for details.
                        */
                        template <typename NodeType, typename MapType>
                        bool operator()( const NodeType & src , MapType & dest ) const
                        {
                                return deserialize_map( src, dest );
                        }
                };

        } // namespace map



} // namespace s11n

#endif // s11n_net_s11n_v1_1_MAP_HPP_INCLUDED
