#ifndef s11n_net_s11n_v1_1_FUNCTOR_HPP_INCLUDED
#define s11n_net_s11n_v1_1_FUNCTOR_HPP_INCLUDED 1
/////////////////////////////////////////////////////////////////////////
// algo.hpp: generic functors and algos for use with libs11n.
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
/////////////////////////////////////////////////////////////////////////

#include <string>

#include "s11n_debuggering_macros.hpp" // COUT/CERR
#include "serialize.hpp" // core de/serialize interface
#include "traits.hpp" // node_traits

namespace s11n
{

        /**
           For each item in [first,last), copies the item to OutputIt
           if pred(*item) returns true.

           Code copied from:

           http://www.bauklimatik-dresden.de/privat/nicolai/html/en/cpp.html
        */

        template <typename InputIt, typename OutputIt, typename UnaryPredicate>
        OutputIt copy_if (InputIt first,
                          InputIt last,
                          OutputIt result,
                          UnaryPredicate pred) 
        {
                while (first != last)
                {
                        if (pred(*first)) *result++ = *first;
                        ++first;
                }
                return result;
        }

 

        /**
           Deletes an object passed to it.

           This type accepts non-pointer types for "destruction." This
           is a non-operation, and is supported to allow other
           template code to generically free objects without needing
           to know if they are pointers. This allows some
           formerly-non-consolidatable reference-vs-pointer-type code to
           share a single implementation, as "delete(myobj)" is not
           valid for non-pointer types, but object_deleter()(myobj)
           is.
        */
        struct object_deleter
        {
                /**
                   Deletes t.
                */
                template <typename T>
                void operator()( T * t ) const
                {
                        // ACH!!!! If we use (const T *) for the arguments
                        // then the (const T &) version is called
                        // even when a pointer type is passed in!!!!

                        // i don't fully understand why
                        // delete( const T * ) is legal, 
                        // considering that it triggers a dtor,
                        // and dtors are non-const.
                        //CERR << "object_deleter deleting "<< std::hex<<t<<"\n";
                        delete( t );
                }
                /**
                   Does nothing: is here to allow some
                   reference-vs-pointer-type transparency.
                */
                template <typename T>
                void operator()( const T & t ) const
                {
                        //CERR << "object_deleter no-op\n";
                }
        };



        /**
           For each item in [begin,end) object_deleter()(*item) is called.

           After this call the container from which the iterators come
           still contains the items but they are invalid - deleted
           pointers. The client should call erase(begin,end) to delete
           the entries, or use convenience function
           free_list_entries(), which accepts a list-style
           container and empties the list.
        */
        template <typename IterT>
        void delete_objects( IterT begin, IterT end )
        {
                std::for_each( begin, end, object_deleter() );
        }



        /**
           object_reference_wrapper is a type for giving access
           to T objects via their dot operator, regardless of whether
           they are pointers or not.

           Intended for use with value_types which come from, e.g.,
           std::list, so objects of those types can be called using
           the same syntax regardless of whether they are pointer
           types or not.

           e.g., assuming MyType might be a pointer or a reference,
           we can ignore that difference for call-syntax purposes
           with:

<pre>           
object_reference_wrapper&lt;MyType&gt; wrap;
wrap(myobj).memberfunc();
</pre>

or:

<pre>
object_reference_wrapper&lt;MyType&gt; wrap(myobj);
wrap().memberfunc();           
</pre>
        */
        template <typename T>
        struct object_reference_wrapper
        {
                typedef T value_type;
                typedef T base_value_type;
                object_reference_wrapper() : m_ptr(0) {}
                object_reference_wrapper( value_type &obj ) : m_ptr(&obj) {};
                /**
                   Sets this object's proxy object to t and returns t.
                 */
                base_value_type & operator()( value_type & t )
                {
                        this->m_ptr = &t;
                        return t;
                }
                /**
                   Returns this object's wrapped object.

                   ACHTUNG: this function WILL Cause Grief if it is called
                   on a default-constructed version of this object: you must
                   set this object's wrapped value using the ctor (or via
                   copying) before calling this.
                 */
                base_value_type & operator()() const { return *(this->m_ptr); }

                /**
                   Returns true if this object is wrapping a non-0 object, else
                   false.
                */
                bool good() const
                {
                        return 0 != this->m_ptr;
                }

        private:
                value_type * m_ptr;
        };

        /**
           A specialization to wrap pointers to (T *) such that they
           can be accessed, via this wrapper, using a dot instead of
           <tt>-&gt;</tt>.
        */
        template <typename T>
        struct object_reference_wrapper<T *>
        {
                typedef T * value_type;
                typedef T base_value_type;
                object_reference_wrapper() : m_ptr(0) {}
                object_reference_wrapper( value_type & obj ) : m_ptr(obj) {};
                /** Sets this object's proxied object to t and Returns t. */
                base_value_type & operator()( value_type & t )
                {
                        this->m_ptr = t;
                        return *t;
                }
                /** Returns this object's wrapped object.

                   ACHTUNG: this function WILL Cause Grief if it is
                   called on a default-constructed version of this
                   object: you must set this object's wrapped value
                   using the ctor, operator(T), or via copying this
                   object from a properly-initialized one before
                   calling this.
                */
                base_value_type & operator()() const { return *(this->m_ptr); }
        private:
                base_value_type * m_ptr;
        };


        /**
           const_object_reference_wrapper is identical in usage to
           object_reference_wrapper, except that it deals with const
           objects. It is a separate functor to avoid ambiguity and
           some impossible overloads.
        */
        template <typename T>
        struct const_object_reference_wrapper
        {
                typedef T value_type;
                typedef T base_value_type;
                const_object_reference_wrapper() : m_ptr(0) {}
                const_object_reference_wrapper( const value_type &obj ) : m_ptr(&obj) {};
                /**
                   Sets this object's proxied obj to t and returns t.
                 */
                const base_value_type & operator()( const value_type & t )
                {
                        this->m_ptr = t;
                        return t;
                }
                /**
                   Returns this object's wrapped object.

                   ACHTUNG: this function WILL Cause Grief if it is
                   called on a default-constructed version of this
                   object: you must set this object's wrapped value
                   using the ctor, operator(T), or via copying this
                   object from a properly-initialized one before
                   calling this.
                 */
                const base_value_type & operator()() const { return *this->m_ptr; }
        private:
                const value_type * m_ptr;
        };

        /**
           A specialization to wrap pointers to (T *) such that they
           can be accessed, via this wrapper, using a dot instead of
           <tt>-&gt;</tt>.
        */
        template <typename T>
        struct const_object_reference_wrapper<T *>
        {
                typedef T * value_type;
                typedef T base_value_type;
                const_object_reference_wrapper() : m_ptr(0) {}
                const_object_reference_wrapper( const value_type & obj ) : m_ptr(obj) {};
                /** Returns (*t). */
                const base_value_type & operator()( value_type & t )
                { 
                        this->m_ptr = t;
                        return *t;
                }
                /** Returns this object's wrapped object. */
                const base_value_type & operator()() const { return *(this->m_ptr); }
        private:
                const base_value_type * m_ptr;
        };


        /**
           A functor allowing pairs of PODs and pointers to be
           mixed together in any combination and be deallocated in a
           uniform way. Admitedly, it's a waste of time for value types,
           but the point is a single functor which can work for all
           pair types.

           sample:

           std::for_each( map.begin(), map.end(), pair_entry_deallocator() );
           
        */
        struct pair_entry_deallocator
        {
                /**
                   Deallocates p.first and p.second. PairType must not
                   be a pointer type.
                */
                template <typename PairType>
                void operator()( PairType & p ) const
                {
                        object_deleter od;
                        //CERR << "pair_entry_deallocator wiping out something...\n";
                        od( p.first );
                        od( p.second );
                }
        };


        /**
           Deallocates all objects in list. If the entries are
           pointers they are deleted, else this is a no-op for each
           entry.

           After this call, list will be empty.
        */
        template <typename ListType>
        void free_list_entries( ListType & list )
        {
                object_reference_wrapper<ListType> wr(list); // in case ListType is a pointer type
                if( ! wr.good() ) return;
                delete_objects( wr().begin(), wr().end() );
                wr().clear();
        }


       /**
           Deletes any pointers in map, and does nothing for reference
           types, then calls map.clear(). This is used to treat
           arbitrary maps containing pointers or value types
           identically, by applying the same set of deallocation rules
           for both cases, simplifying some template implementations.

           After calling this, map will be empty.
        */
        template <typename MapType>
        void free_map_entries( MapType & map )
        {
                object_reference_wrapper<MapType> wr(map); // in case MapType is a pointer type
                if( ! wr.good() ) return;
                std::for_each( wr().begin(), wr().end(), pair_entry_deallocator() );
                wr().clear();
        }


        /**
           child_pointer_deep_copier is a functor to deep-copy
           a list of pointers into another list. Designed for
           use with std::for_each and the like.

	   Assuming T is the type contained in ListType, stripped
	   of any pointer part, then the following must hold:


           - List must support <code>push_back( T * )</code>.

	   - This must be a valid expression:

	   <code>T * t = new T( *p );</code>
	   
	   Where p is a passed to this type's operator().

           ACHTUNG: This is only useful for non-polymorphic
           copying.

           It might be interesting to note that copying
           monomorphic s11n::s11n_node objects this way is
           "pseudo-polymorphic" - the copy itself is
           monomorphic but the data needed to deserialize the
           proper type from the node is maintained.
        */
        template <typename ListType>
        class child_pointer_deep_copier
        {
        private:
                template <typename T> struct PtrStrip { typedef T type; };
                template <typename T> struct PtrStrip<T *> { typedef T type; };
        public:
                typedef ListType list_type;
                typedef typename ListType::value_type full_value_type;
                typedef typename PtrStrip<full_value_type>::type value_type; // that is, list_type::value_type, minus any pointer part.
                /**
                   Target list must outlive this object.
                */
                child_pointer_deep_copier( list_type & target ) : m_childs(&target)
                {}

                /**
                   Inserts a copy of p into this object's list and returns true.

                   Returns true if p is successfully copied.

                   If an exception thrown while copying, this function
                   will catch it and not modify the underlying
                   container. In that case, false is returned.

                   The target list takes ownership of the new copy of p.
                */
                bool operator()( const value_type * p ) throw()
                {
                        if( ! this->m_childs || ! p ) return false;
                        value_type * cp = 0;
                        try
                        {
                                cp = new value_type( *p );
                                if( ! cp ) return false;
                        }
                        catch(...)
			{
                                delete( cp );
                                return false;
                        }
                        this->m_childs->push_back( cp );
                        return true;
                }
        private:
                list_type * m_childs;
        };


        /**
           Helper to avoid using bind1st/bind2nd. (Sorry, i'll get rid of this someday.)
        */
        template <typename ComparedType>
        struct equal_to
        {
                typedef ComparedType compared_type;
                explicit equal_to( const compared_type & d ) : m_data(d)
                {}

                /**
                   Returns true if p compares equal to the value
                   passed to this object's ctor.
                */
                void operator()( const compared_type & p )
                {
                        return p == this->m_data;
                }
        private:
                compared_type m_data;
        };

        /**
           Functor to return true if given NameableT objects match a
           certain name. NameableT must support:

           std::string name() const;

           This class is admittedly to avoid the use of bind1st/bind2nd
           :/.
        */
        template <typename NameableT>
        struct same_name
        {
                typedef NameableT nameable_type;
                explicit same_name( const std::string & n ) : m_name(n)
                {}

                bool operator()( const nameable_type * x ) const
                {
                        if( ! x  ) return this->m_name.empty();
                        return x->name() == this->m_name;
                }
        private:
                std::string m_name;
        };

        /**
           Helper class to strip pointers from value_type
           typedefs. It's value_type represents T, minus any
           leading pointer part.
                   
           example:
<pre>
pointer_stripper<std::list<int *>::value_type>::value_type
</pre>
		is int, not (int *).

        */
        template <typename T>
        struct pointer_stripper
        {
                /**
                   value_type is always equivalent to T.
                */
                typedef T value_type;
        };

        /**
           Specialization to make (T *) equivalent to
           (T).
        */
        template <typename T>
        struct pointer_stripper<T *>
        {
                /**
                   value_type is always equivalent to T.
                */
                typedef T value_type;
        };
	/** Specialization to make (T&) equiavalent to (T). */
        template <typename T>
        struct pointer_stripper<T &>
        {
                /**
                   value_type is always equivalent to T.
                */
                typedef T value_type;
        };

	/**
	   Internal debuggering function.

	   Dumps some info about n to the given stream. Mainly useful
	   for debugging failed deserialization.
        */
        template <typename NodeType>
        void dump_node_debug( const NodeType & n, std::ostream & os )
        {
                typedef node_traits<NodeType> NTR;
                os << "node dump: ["<<NTR::name(n)<<"]["<<NTR::class_name(n)<<"]@"<<std::hex<<&n<<"\n";
                typedef typename NTR::property_map_type PMT;
		typedef typename PMT::const_iterator PT;
                PT b = NTR::properties(n).begin(), e = NTR::properties(n).end();
                os << "==================== properties:\n";
                for( ; e != b; ++b )
                {
                        os << (*b).first << " = " << (*b).second << "\n";
                }
                os << "[end node]\n";
        }

        /**
           A helper functor to loop over serializable
           children.

           Designed for use with std::for_each().

           NodeType must be compatible with
           s11n::data_node.

           Please see the operator() docs for important usage
           information, especially if you want to use this
           object outside the context of for_each().
        */
        template <typename NodeType>
        struct data_node_child_serializer
        {
                typedef NodeType node_type;
                /**
                   Creates an object for serializing

                   Preconditions:

                   - dest must outlive this object.
                   More correctly, this object's
                   operator() must not be called after
                   dest is destroyed.

                */
                data_node_child_serializer( node_type & dest, const std::string & subname )
                        : result(true), m_name(subname), m_root(&dest)
                {
                }
 
                /**
                   Serializes src into a subnode of dest using the
                   name given in this object's ctor.

                   Note that during an, e.g., for_each() this object
                   will return false on a failed serialize, and will
                   CONTINUE to return false on additional serializations.
                   This is to avoid the possibility that for_each()
                   fails on the first item of a list, handles 3000 items,
                   and then the whole thing fails because of the first one.
                   Thus, this operator will never process another request
                   once it has returned false ONCE.
                */
                template <typename SerializableT>
                bool operator()( const SerializableT * src )
                {
                        if( ! this->result ) return false;
                        return this->result = 
                                serialize_subnode( *this->m_root,
                                                   this->m_name,
                                                   *src );
                }
                template <typename SerializableT>
                bool operator()( const SerializableT & src )
                {
                        return this->operator()( &src );
                }

                /**
                   For use as a "return value catcher" for std::for_each().
                   See operator() for how it is set. The starting value
                   is true, which means that looping over an empty list
                   with this object will return a true result (which is
                   the convention in s11n).
                */
                bool result;
        private:
                std::string m_name;
                node_type * m_root;
        };

        /**
           A helper functor deserialize a set of data_nodes.

           Designed for use with std::for_each().

           ListType must be compatible with
           <code>std::list<SomeSerializableType *></code>.

        */
        template <typename ListType>
        struct data_node_child_deserializer
        {
                /**
                   The ListType templatized type. Must support:

                   push_back(value_type *);
                 */
                typedef ListType list_type;
                /**
                   value_type is equivalent to a SerializableType.
                */
                typedef typename pointer_stripper<
			typename list_type::value_type
 			>::value_type  value_type;

                /**
                   Creates an object for deserializing Serializables
                   into a target list.

                   If tolerant is true then this object will always
                   try to deserialize new items passed to it's
                   operator(). If it is false (the default) it will
                   stop processing after a single failure.

                   Preconditions:

                   - dest must outlive this object. More correctly,
                   this object's operator() must not be called after
                   either dest is destroyed.

                */
                data_node_child_deserializer( list_type & dest,
                                              bool tolerant = false )
                        : result(true), m_tolerant(tolerant), m_list(&dest)
                {
                }

                /**
                   Tries to deserialize a (value_type *) from src. On error it returns 0,
                   else it inserts the new child into this object's destination container
                   and returns true. "Error" means one of:

		   - src is 0.

		   - this object was creates as intolerant, and a PREVIOUS operator() call
		   failed.

		   - s11n::serialize<NodeType,value_type>(*src) fails.


                   Postconditions:

                   - If successful, this object's destination container owns the pointers
		   added to it.

		   - On error the destination list is not modified.
                */
                template <typename NodeType>
                bool operator()( const NodeType * src )
                {
                        if( ! result && ! this->m_tolerant ) return false; // once an item fails, stop processing them.
                        if( ! src ) return result = false;
                        value_type * ch = 0;
			result = false; // in case deserialize throws
                        ch = ::s11n::deserialize<NodeType,value_type>( *src );
                        //CERR << "data_node_child_deserializer<>("<<src->name()<<") = "
                        //     << std::hex << ch <<"\n";
                        if( ! ch ) return result = false;
                        this->m_list->push_back( ch );
                        return result = true;
                }

                template <typename NodeType>
                bool operator()( const NodeType & src )
                {
                        return this->operator()( &src );
                }

                /**
                   For use as a "return value catcher" for std::for_each().

                   If tolerant is true then this will only reveal the value of
                   the most-recently-processed node.
                */
                bool result;
        private:
                bool m_tolerant;
                list_type * m_list;
        };




        /**
           A Serializable Proxy for streamable types. It "should" work
           with any type which meets these conditions:

           - complementary i/ostream operators are implemented (as
           member or free functions).

           - supports a copy ctor (for deserialization).


	   The class name of serialized objects will be taken
	   from s11n_traits<>::class_name(), which isn't truly
	   valid because most streamable types are never registed
	   via s11n_traits. Hmmm.

           Its output is significantly bigger than using, e.g. node
           properties to store them, but with this proxy any
           streamable can be treated as a full-fledged Serializable,
           which allows some generic container-based serialization to
           be done regardless of the underlying types (see
           list_serializable_proxy for an example).

	   ACHTUNG: never pass the same Serializable to the
	   operators more than once or you will get duplicate and/or
	   incorrect data.
        */
        struct streamable_type_serialization_proxy
        {
                /**
                */
                streamable_type_serialization_proxy()
                {}

                /**
                   Creates a property in dest, called 'v', and sets
                   its value to src using node_traits<NodeType>::set(dest,"v",src).

                   Always returns true unless no class name can be found
		   for src, in which case it returns false or throws
		   (depending on throw_policy()) to avoid inserting
		   a non-name node into the data tree (which might
		   result in unreadable data later).
                */
                template <typename NodeType, typename SerType>
                bool operator()( NodeType & dest, const SerType & src ) const
                {
                        typedef node_traits<NodeType> NTR;
			typedef s11n_traits<SerType> STR;
			std::string cname = STR::class_name(&src);
			if( NTR::class_name(dest).empty() )
			{
				potentially_throw( ThrowOnFailedSerialize, "Could not determine s11n class name for ostream-based serialization",__FILE__,__LINE__);
				return false;
			}
                        NTR::class_name( dest, cname );
                        NTR::set( dest, "v", src );
                        return true;
                }

                /**
                   Looks for a property set by the serialize operator and sets
                   dest to its value. The default for dest, in the case
                   of a missing property or nonconvertable value is
                   dest itself.

                   If the property is missing this function does
                   nothing and returns false, or throws if
                   throw_policy() dictates ThrowOnFailedDeserialize.

                   On success dest gets assigned the property's value
                   and true is returned. This function cannot catch a
                   case of inability to convert the value into a
                   SerType: client code interested in doing so should
                   compare dest's value to a known error value after
                   this function returns or throw from that type's
		   istream operator on error.
                */
                template <typename NodeType, typename SerType>
                bool operator()( const NodeType & src, SerType & dest ) const
                {
                        typedef node_traits<NodeType> NTR;
                        if( ! NTR::is_set( src, "v" ) )
                        {
                                potentially_throw( ThrowOnFailedDeserialize, "streamable_serializable_proxy: deser failed: property 'v' missing!" );
                                return false;
                        }
                        dest = NTR::get( src, "v", SerType() /* should never happen */ );
                        return true;
                }
        };


        /**
           Adds ch as a child of parent. Parent takes over ownership
           of ch.

           NodeType must have a node_traits<> specialization.
        */
        template <typename NodeType, typename ChildType>
        void add_child( NodeType & parent, ChildType * ch )
        {
                typedef ::s11n::node_traits<NodeType> NTR;
                NTR::children( parent ).push_back( ch );
        }


        /**
           Creates a new node, named nodename, as a child of parent.

           Returns a reference to the new child, which parent now
           owns.

           NodeType must have a node_traits<> specialization or work
           using the default
        */
        template <typename NodeType>
        NodeType & create_child( NodeType & parent,
                                  const std::string nodename )
        {
                typedef ::s11n::node_traits<NodeType> NTR;
                NodeType * n = NTR::create( nodename );
                NTR::children( parent ).push_back( n );
                return *n;
        }



        /**
           Each child in parent.children() which has the given name is
           copied into the target container.

           Returns the number of items added to target.

           DestContainerT must support an insert iterator which will
           insert the pointer type contained in the list returned by
           parent.children(). i.e., it must hold (const
           SomeSerializableType *).

           Ownership of the child does not change by calling this
           function. Normally they are owned by the parent node
           (unless the client explicitely does something to change
           that).
        */
        template <typename NodeT, typename DestContainerT>
        size_t find_children_by_name( const NodeT & parent,
				      const std::string & name,
				      DestContainerT & target )
        {
		typedef node_traits<NodeT> TR;
                size_t c = target.size();
                ::s11n::copy_if( TR::children(parent).begin(),
				 TR::children(parent).end(),
				 std::insert_iterator<DestContainerT>( target,
								       target.begin() ),
				 same_name<NodeT>( name )
			       // ^^^^ BUG: this only works when NodeT::name() is public,
			       // as it avoids node_traits<>!
			       // In 1.1 s11n_node::name() might be made private, and node_traits<>
			       // will be his only friend. Then this bug will show up.
			       // s11n::data_node is frozen: he won't change, but s11nlite
			       // might switch to using s11n_node in 1.1!
                                         );
                // you gotta love the STL, man.
                return target.size() - c;
        }

        /**
           Finds the FIRST child in parent with the given name and
           returns a pointer to it, or 0 if no such child is found.

           Ownership of the child does not change by calling this
           function: parent still owns it.
        */
        template <typename NodeT>
        const NodeT *
        find_child_by_name( const NodeT & parent, const std::string & name )
        {
		typedef node_traits<NodeT> TR;
                typedef typename NodeT::child_list_type::const_iterator CIT;
                CIT it = std::find_if( TR::children(parent).begin(),
                                       TR::children(parent).end(),
                                       ::s11n::same_name<NodeT>( name )
                                       );
                return (TR::children(parent).end() == it) ? 0 : *it;
        }

        /**
           A non-const overload of find_child_by_name(). Functionally
           identical to the const form, except for the constness of
           the parent argument and return value.

           Ownership of the returned pointer is not changed by calling
           this function (normally parent owns it, but clients may
           change that without affecting this function). When in
           doubt, i.e. during "normal usage", do NOT delete the returned
           pointer, because the parent node owns it.
        */
        template <typename NodeT>
        NodeT *
        find_child_by_name( NodeT & parent, const std::string & name )
        {
		typedef node_traits<NodeT> TR;
                typedef typename NodeT::child_list_type::iterator IT;
                IT it = std::find_if( TR::children(parent).begin(),
				      TR::children(parent).end(),
				      ::s11n::same_name<NodeT>( name )
                                       );
                return (TR::children(parent).end() == it) ? 0 : *it;
        }



	/**
	   UNTESTED!

	   A functor to compare the data in two nodes for value
	   equality.
	*/
	struct node_compare_eq
	{

		/**
		   UNTESTED!

		   Compares the following node data, gleaned from the
		   node_traits<NodeT> API:

		   - class_name()
		   
		   - name()
		   
		   - properties()
		   
		   - children(), processed recursively.
		   
		   If any mismatch is found, false is returned, else true is
		   returned.
		*/
		template <typename NodeT>
		bool operator()( const NodeT & n1, const NodeT & n2 )
		{
			if( &n1 == &n2 ) return true;
			typedef ::s11n::node_traits<NodeT> NTR;
			if( NTR::class_name( n1 ) != NTR::class_name( n2 ) )
			{
				return false;
			}
			if( NTR::name( n1 ) != NTR::name( n2 ) )
			{
				return false;
			}
			if( NTR::properties( n1 ) != NTR::properties( n2 ) )
			{
				return false;
			}
			if( NTR::children( n1 ).size() != NTR::children( n2 ).size() )
			{
				return false;
			}
			typedef typename NTR::child_list_type CList;
			typedef typename CList::const_iterator CIT;
			const CList & cl1 = NTR::children( n1 );
			const CList & cl2 = NTR::children( n2 );
			CIT c1;
			CIT c2;
			for( ; cl1.end() != c1; ++c1, ++c2 )
			{
				if( ! this->operator()( *(*c1), *(*c2) ) )
				{
					return false;
				}
			}
			return true;

		}
	};


	/**
	   UNTESTED!
	*/
	template <typename NodeT>
	bool node_compare( const NodeT & n1, const NodeT & n2 )
	{
		return node_compare_eq()( n1, n2 );
	}

} // namespace

#endif // s11n_net_s11n_v1_1_FUNCTOR_HPP_INCLUDED
