#ifndef s11n_cl_S11N_CLASSLOAD_HPP_INCLUDED
#define s11n_cl_S11N_CLASSLOAD_HPP_INCLUDED 1

#include <s11n.net/s11n/s11n_debuggering_macros.hpp>
#include <s11n.net/s11n/factory.hpp>
#include <s11n.net/s11n/exception.hpp>
#include <s11n.net/s11n/s11n_config.hpp>

#if s11n_CONFIG_ENABLE_PLUGINS
#  include <s11n.net/s11n/plugin/plugin.hpp>
#endif

// #include <s11n.net/s11n/s11n_debuggering_macros.hpp> // only temporary, for CERR

namespace s11n {
        /**
           The s11n::cl namespace encapsulates the classloader-related
           API for libs11n.

           Note that the registration functions in this namespace
           register with the s11n::fac family of classes. Clients wishing to use
           their own factories should:

           - register with their appropriate classloader.

           - specialize the object_factory class template
           so that it forwards calls to their classloader.

           This layer is used for classloading anything which s11n
           needs to dynamically load, including:

           - Serializables

           - Serializers

           - FlexLexers

           It supports loading types via DLLs if the underlying factory
	   object supports them.

	   Changes from 1.0.x to 1.1.x:

	   - Now internally uses s11n::factory_mgr instead of the
	   cllite API.
        */
        namespace cl {

                /**
                   A default object factory functor for use
                   with the s11n::s11n_traits type.

                   Clients may specialize this to return objects from
                   their own factories. By default it uses s11n::fac's
		   mini-framework, and thus can load any type registered with
                   that API.
                */
                template <typename BaseType>
                struct S11N_EXPORT_API object_factory
                {
                        /** Same as BaseType. */
                        typedef BaseType base_type;

                        /**
			   The default implementation returns
			   ::s11n::fac::factory<BaseType>().create(classname). The
			   caller owns the returned pointer, which may
			   be 0.

			   If create() returns null and s11n is built with the s11n::plugin
			   layer, then s11n::plugin::open(classname) is used to search
			   for a DLL. If that succeeds, we do another lookup in the factory,
			   the presumption being that the DLL registered its class with
			   the factory.

			   TODO: cache lookups, so we don't do a DLL
			   search on every failed classload
			   lookup. Then again, deserialization fails
			   in most cases on the first classload, so
			   this won't be called 1000 times while
			   deseri'ing a list of pointers of some type
			   which cannot be loaded.
                        */
                        base_type * operator()( const std::string & classname ) const
                        {
                                try
                                {
					base_type * ret =  ::s11n::fac::factory<BaseType>().create( classname );
					if( ret )
					{
						return ret;
					}
#if s11n_CONFIG_ENABLE_PLUGINS
// 					CERR << "Trying to find plugin for " << classname << "...\n";
					std::string where = ::s11n::plugin::open( classname );
					if( !where.empty()  )
					{
// 						CERR << "We got one: " << where << "\n";
						return ::s11n::fac::factory<BaseType>().create( classname );
					}
#endif // s11n_CONFIG_ENABLE_PLUGINS
                                }
                                catch(...)
                                {
					potentially_throw( ::s11n::ThrowOnFactoryError, "Factory load failed for class '"+classname+"'." );
                                }
				return 0;
                        }
                };


                /**
		   Returns object_factory<BaseType>()(key).
                */
                template <typename BaseType>
                BaseType * classload( const std::string key )
                {
                        return object_factory<BaseType>()( key );
                }

                /**
                   Registers SubType with BaseType's classloader, using
                   a default object factory.

                   SubType must be a subtype of BaseType, or may be
                   the same as BaseType. If BaseType is abstract it
                   must first be registered as an abstract base, e.g.,
                   using classloader_register_abstract().  In that
                   case BaseType should not be registered with this
                   function.

                */
                template <typename BaseType, typename SubType>
                void classloader_register( const std::string & classname )
                {
			// CERR << "classloader_register("<<classname<<")\n";
			::s11n::fac::register_subtype<BaseType,SubType>( classname,
									 ::s11n::fac::create_hook<BaseType,SubType>::create
									 );
                }

                /**
                   Registers BaseType with it's own classloader.
                */
                template <typename BaseType>
                void classloader_register_base( const std::string & classname )
                {
                        classloader_register<BaseType,BaseType>( classname );
                }

                /**
                   Registers BaseType as an abstract type. That is, it
                   assigns it a factory which always returns 0.

                   If types you register with classloader_register()
                   (and friends) have an abstract BaseType then that
                   BaseType should be registered via this function (or
                   equivalent).
                */
                template <typename BaseType>
                void classloader_register_abstract( const std::string & classname )
                {
			::s11n::fac::register_abstract<BaseType>( classname );
                }


		/**
		   Aliases the given classname with the underlying
		   factory layer, such that classload<BaseType>(_alias)
		   will return the same as classload<BaseType>(classname).

		   Added in 1.1.0.
		*/
                template <typename BaseType>
                void classloader_alias( const std::string & _alias,
					const std::string & classname)
                {
			::s11n::fac::factory<BaseType>().aliases().alias( _alias, classname );
                }


        } // namespace cl
} // namespace s11n


#endif // s11n_cl_S11N_CLASSLOAD_HPP_INCLUDED
