#ifndef s11n_net_s11n_v1_1_EXCEPTION_HPP_INCLUDED
#define s11n_net_s11n_v1_1_EXCEPTION_HPP_INCLUDED 1

#include <string>
#include <exception>

namespace s11n {


	/**
	   The base-most exception type used by s11n.
	*/
        struct s11n_exception : public std::exception
        {
	public:
		virtual ~s11n_exception() throw() {}
                explicit s11n_exception( const std::string & What );
		/**
		   Creates an error string consisting of:

		   file:linenum: What
		*/
                s11n_exception( const std::string & What, const std::string & file, unsigned int linenum );
                virtual const char * what() const throw();
        private:
                std::string m_what;
        };

	/**
	   An exception type to enable to selective catching of
	   classloader-related exceptions. These need to be caught
	   separately from s11n_exceptions in many cases because
	   sometimes a classloader can try other alternatives on an
	   error.
	*/
	struct factory_exception : public s11n_exception
	{
	public:
		virtual ~factory_exception() throw() {}
		explicit factory_exception( const std::string & What ) : s11n_exception( What ) {}
                factory_exception( const std::string & What, const std::string & file, unsigned int line ) : s11n_exception( What,file,line ) {}
	};


	/**
	   Really for use by clients, i/o layers, and s11nlite, not by
	   the s11n core.
	*/
	struct io_exception : public s11n_exception
	{
	public:
		virtual ~io_exception() throw() {}
		explicit io_exception( const std::string & What ) : s11n_exception( What ) {}
                io_exception( const std::string & What, const std::string & file, unsigned int line ) : s11n_exception( What,file,line ) {}
	};

	/**
	   A bitmaskable set of hints to the s11n framework about when
	   it should (not) throw exceptions for various types of
	   errors.

	   Note that this is not used in the public API, but only as a
	   convenience within the library, and optionally by add-on
	   code which wants to follow the same conventions. The actual
	   type ThrowPolicyMask should not be used as API parameters:
	   use throw_policy_mask instead.
	*/
	enum ThrowPolicyMask {

	/**
	   Hint to serialize() ops to throw instead of returning false.
	 */
	ThrowOnFailedSerialize = 0x0001,

	/**
	   Hint to deserialize() ops to throw instead of returning false.
	 */
	ThrowOnFailedDeserialize = 0x0002,

	/**
	   Special case for use by SAM<T*>. Causes SAM<T*> to throw
	   if it is asked to de/ser a null pointer (default behaviour
	   is to return false).
	 */
	ThrowOnSerializeNullPointer = 0x0004,
	/**
	   Convenience entry.
	*/
	ThrowOnFailedDeSer = (ThrowOnFailedSerialize | ThrowOnFailedDeserialize | ThrowOnSerializeNullPointer),

	/**
	   Hint to throw on classloading errors.
	*/
	ThrowOnFactoryError = 0x0010,

	/**
	   For use by i/o layers and s11nlite, not the core.
	*/
	ThrowOnIOError = 0x0020,

	/**
	   Hint to wrap unknown exceptions as s11n_exceptions. Default is to
	   throw them as-is to the client.
	 */
	ThrowWrapUnknownExceptions = 0x0040,

	/**
	   Hint to rethrow unknown exceptions (default is to follow
	   normal error routine, which normally means returning false
	   or null). This and ThrowWrapUnknownExceptions are mutually
	   exclusive, and the library does not guaranty which takes
	   precedent if both are set.
	*/
	ThrowRethrowUnknownExceptions = 0x0080,

	/**
	   A hint to algorithms. The core cannot recognize such errors.
	*/
	ThrowOnMissingProperty = 0x0100,

	/**
	   Highest internally-reserved throw mask. Clients wishing to
	   "extend" this enum should start after this one.
	*/
	ThrowReservedMax = 0xFFFFF, // yes, 5

	/** Hint to never throw. */
	ThrowNever = 0x00000000,

	/** Hint to throw on any error. */
	ThrowAlways = 0xFFFFFFFF,

	/**
	   The build-time default throwing policy.
	   This is the policy used by throw_policy()
	   until another policy is explicitely set.
	*/
	ThrowDefaultPolicy = ThrowReservedMax
	};

	/**
	   The "public API" variation of the ThrowPolicyMask, used to
	   allow us to logically extend the enum's range without
	   having to touch its code.
	*/
	typedef unsigned long throw_policy_mask;

	/**
	   Sets the library-wide throwing policy and returns the old policy.
	*/
	throw_policy_mask throw_policy( throw_policy_mask m ) throw();

	/**
	   Returns the library-wide throwing policy.
	*/
	throw_policy_mask throw_policy() throw();

	/**
	   A convenience function to throw an s11n_exception if
	   a condition matches the current throw_policy().

	   The exception thrown may be a subtype of s11n_exception,
	   depending on condition. The rules are, in the order they
	   are checked:

	   - If !(throw_policy() & condition) no exception is thrown
	   and the function returns. This is the only case for which
	   it will actually return.

	   - A condition containing ThrowOnIOError will cause an
	   io_exception to be thrown if throw_policy() has that bit
	   set.

	   - A condition containing ThrowOnFactoryError will cause a
	   factory_exception to be thrown.

	   - Anything else causes an s11n_exception to be thrown.

	   This function exists more for testing exception behaviour
	   than for practical use.

	*/
	void potentially_throw( throw_policy_mask condition, const std::string & errstring );

	/**
	   Like the two-arg potentially_throw(), but prefixes
	   "filename:linenum: " to the error string. It is expected that
	   one passes __FILE__ and __LINE__ here.
	*/
	void potentially_throw( throw_policy_mask condition, const std::string & errstring,
				const std::string & filename, unsigned int linenum );

	/**
	   A helper type to set a throw policy upon construction and
	   reset it at destruction. This is used to simplify keeping
	   the global policy in a known state as it is changed. When
	   exceptions are thrown, throw_policy_changers in the
	   exception's scope will reset the policy to its previous
	   state as they are destroyed, ensurning policy consistency,
	   at least in single-threaded apps.

	   It is not expected that clients will use this type - it is
	   primarily intended for testing the exception guarantees of
	   s11n.
	*/
	struct throw_policy_changer
	{
		/**
		   Sets ::s11n::throw_policy() to the given mask.

		   If override is true then during the dtor throw_policy()
		   will be set to the value it contains before this
		   function is called. If override is false then the
		   current mask is ORed with the given mask.
		*/
		throw_policy_changer( throw_policy_mask mask,
				      bool override = true );

		/**
		   Sets throw_policy() to the value it contained
		   before this object's ctor modified it.
		*/
		~throw_policy_changer() throw();

	private:
		throw_policy_mask m_oldmask;
	};

} // namespace s11n

#define S11N_THROW(WHAT) throw ::s11n::s11n_exception(WHAT,__FILE__,__LINE__)

#endif // s11n_net_s11n_v1_1_EXCEPTION_HPP_INCLUDED
