#ifndef s11n_DATA_SERIALIZE_H_INCLUDED
#define s11n_DATA_SERIALIZE_H_INCLUDED
////////////////////////////////////////////////////////////////////////
// serialize.hpp:
//
// Defines the core de/serialize() functions (and close friends).
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#include <string>
#include <memory> // auto_ptr

#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // COUT/CERR
#include <s11n.net/s11n/classload.hpp> // classload()
#include <s11n.net/s11n/traits.hpp> // s11n_traits
#include <s11n.net/s11n/exception.hpp> // s11n_exception and friends

#include <sstream> // only for some debugging output

////////////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.hpp/s11n_node.hpp ALLOWED!
////////////////////////////////////////////////////////////////////////////////

#define NODEERR if(0) CERR
// ^^^ a debugging macro

namespace { // anon namespace important for linking reasons, to avoid ODR violations.

        /***
            s11n_api_marshaler is an internal API marshaller for s11n.
	    See the lib manual for full details.

            For the default implementation s11n_traits<SerializableT>
            is used to marshal the de/serialize() calls.
            is used to marshal the de/serialize() calls.
        */
        template <typename SerializableT>
        struct s11n_api_marshaler
        {
                /**
                   Same as SerializableT.
                */
                typedef SerializableT serializable_type;

                /**
                   Returns s11n_traits<serializable_type>::serialize_functor()( dest, src ).

                   It also sets dest's class_name() to
                   s11n_traits<serializable_type>::class_name(&src),
                   but this is generally only suitable for monomorphic
                   serializable_types, and serialization
                   implementations should overwrite that if needed.
                */
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type & src )
                {
                        typedef ::s11n::node_traits<NodeType> NTR;
                        typedef ::s11n::s11n_traits<serializable_type> STR;
                        NTR::class_name( dest, STR::class_name(&src) );
			typename STR::serialize_functor sf;
                        return sf( dest, src );
                }

                /**
                   Returns s11n_traits<SerializableT>::deserialize_functor()(src,dest).
                */
                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type & dest )
                {
                        typedef ::s11n::s11n_traits<serializable_type> STR;
			typename STR::deserialize_functor df;
                        return df( src, dest );
                }
        };

        /**
           A specialization to handle pointer types the same as
           reference/value types. It simply translates the pointers
           into references.
        */
        template <typename SerializableT>
        struct s11n_api_marshaler<SerializableT *>
        {
                /**
                   The SerializableT templatized type, minus any
                   pointer part.
                */
                typedef SerializableT serializable_type;

                /**
                   Convenience typedef: this class' parent type.
                 */
                typedef s11n_api_marshaler<serializable_type> parent_type;

                /**
                   Returns parent_type::serialize( dest, *src );

                   src must be a valid pointer.
                */
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type * src )
                {
                        if( ! src )
			{
				potentially_throw( ::s11n::ThrowOnSerializeNullPointer, "serialize() attempted on a NULL pointer." );
				return false;
			}
                        return parent_type::serialize( dest, *src );
                }

                /**
                   Returns parent_type::deserialize( src, *dest );

                   dest must be a valid pointer.
                */
                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type * dest )
                {
                        if( ! dest )
			{
				potentially_throw( ::s11n::ThrowOnSerializeNullPointer, "serialize() attempted on a NULL pointer." );
				return false;
			}
                        return parent_type::deserialize( src, *dest );
                }
        };

} // anon namespace


/**********************************************************************

General Conventions:

	NodeType should conform to the conventions laid out
	by s11n::data_node.


    SerializableTypes/BaseTypes:

	- BaseT must have the following in it's interface:

	- bool SerializeFunction( NodeType & dest ) const;

	- bool DeserializeFunction( const NodeType & dest );

	SerializeFunction/DeserializeFunction need not be virtual,
	though they may be.

    Proxy functors:

	Serialization functor must have:

	bool operator()( NodeType & dest, const BaseT & src ) const;

	Deserialization functor must have:

	bool operator()( const NodeType & src, BaseT & dest ) const;

	They may be the same functor type - const resolution will
	determine which s11n uses. Sometimes this causes ambiguity,
	and may require 2 functors.

	These signatures apply for all functors designed to work as
	de/serializers.

**********************************************************************/


namespace s11n {



        /**
           Serializes src to target using the default API marshaling
           mechanism.

	   This function honors the conventions set out by
	   throw_policy() by implementing the following behaviour:

	   On success it always returns true.

	   If a the underlying operation returns false, it will
	   return false unless the throw_policy() specifies
	   one of the following:

	   a) all s11n_exceptions are rethrown because we assume they
	   were thrown with the throw_policy() in mind.

	   b) caught std::exceptions will translate to an
	   s11n_exception if the policy dictates
	   ThrowOnFailedSerialize, else the translate to a return of
	   false.

	   c) Unknown exceptions will be translated to an s11n_exception
	   if the policy dictates (ThrowWrapUnknownExceptions | ThrowOnFailedSerialize),
	   otherwise they are rethrown if the policy dictates ThrowRethrowUnknownExceptions,
	   else they translate to a return of false.
        */
        template <typename DataNodeType, typename SerializableT>
        bool serialize( DataNodeType & target, const SerializableT & src )
        {
		typedef s11n_traits<SerializableT> STR;
                try
                {
                        if( ! s11n_api_marshaler<SerializableT>::serialize( target, src ) )
			{
				potentially_throw( ThrowOnFailedSerialize, "serialize() failed on target of type "+STR::class_name(&src) );
				return false;
			}
			return true;
                }
		catch( const s11n_exception & sex )
		{ // we simply rethrow these, the hope being that they are thrown honoring the current throw_policy()
			throw sex;
		}
		catch( const std::exception & ex )
		{
			potentially_throw( ThrowOnFailedSerialize, ex.what() );
			return false;
		}
                catch(...)
                {
			potentially_throw( ThrowWrapUnknownExceptions | ThrowOnFailedSerialize, "Unknown exception during serialize<>(DataNode,SerializableT)" );
			if(throw_policy() & ThrowRethrowUnknownExceptions) throw;
			return false;
                }
        }

        /**
           Deserializes target from src using the default API marshaling
           mechanism. Returns true on success.

	   Errors and Exceptions:

	   If the underlying deserialize call fails then false is returned
	   unless throw_policy() specifies any of the following:

	   a) all s11n_exceptions are rethrown, as they are assumed to
	   have been thrown with throw_policy() in mind.

	   b) std::exceptions are translated to s11n_exceptions if the
	   policy dictates ThrowOnFailedDeserializer, else they
	   translate to a false return value.

	   c) Unknown exceptions are translated to s11n_exceptions if
	   the policy dictates (ThrowWrapUnknownExceptions |
	   ThrowOnFailedDeserializer). If the policy dicates
	   ThrowRethrowUnknownExceptions the exception is rethrown
	   as-is, otherwise it translates to a false return value.

        */
        template <typename DataNodeType, typename DeserializableT>
        bool deserialize( const DataNodeType & src, DeserializableT & target )
        {
		typedef s11n_traits<DeserializableT> STR;
                try
                {
                        if( ! s11n_api_marshaler<DeserializableT>::deserialize( src, target ) )
			{
				potentially_throw( ThrowOnFailedDeserialize, "deserialize() failed on target of type "+STR::class_name(&target) );
				return false;
			}
			return true;
                }
		catch( const s11n_exception & sex )
		{ // we simply rethrow these, the hope being that they are thrown honoring the current throw_policy()
			throw sex;
		}
		catch( const std::exception & ex )
		{
			potentially_throw( ThrowOnFailedDeserialize, ex.what() );
			return false;
		}
                catch(...)
                {
			potentially_throw( ThrowWrapUnknownExceptions | ThrowOnFailedDeserialize, "Unknown exception during deserialize<>(DataNode,DeserializableT)" );
			if(throw_policy() & ThrowRethrowUnknownExceptions) throw;
			return false;
                }
        }

        /**
           Tries to deserialize a DeserializableT from src, using
           <code>s11n_traits<DeserializableT>::factory_type()(node_traits<DataNodeType>::class_name(src))</code>
           to create a new DeserializableT. Returns 0 on error,
           otherwise returns a pointer to a new object, which the
           caller takes ownership of.

	   This function honors the conventions set out by
	   throw_policy(), throwing when the policy suggests we
	   should. Specifically, it throws if it cannot factory-load
	   the appropriate type or if deserialize() into the
	   newly-created object fails, assuming the global policy says
	   to. If it throws during deserialize(), the new object is
	   destroyed.
        */
        template <typename DataNodeType, typename DeserializableT>
        DeserializableT * deserialize( const DataNodeType & src )
        {
                typedef ::s11n::s11n_traits<DeserializableT> STR;
                typedef ::s11n::node_traits<DataNodeType> NTR;
		typename STR::factory_type fac;
                std::auto_ptr<DeserializableT> obj( fac( NTR::class_name( src ) ) );
                if( ! obj.get() )
                {
			std::ostringstream errmsg;
                        errmsg << "deserialize<>(DataNode): classload failed for class '"
                                << NTR::class_name( src )<<"'."
                                << " It is probably not registered with its base-most classloader.";
			potentially_throw( ThrowOnFactoryError | ThrowOnFailedDeserialize, errmsg.str() );
                        return 0;
                }
                if( deserialize<DataNodeType,DeserializableT>( src, *obj ) ) // might throw
                {
			return obj.release(); // take over ownership.
                }
                return 0;
        }

        /**
           Serializes src to as a subnode of target, named
           nodename. Except for the addition of a subnode, it is
           identical to serialize( target, src ).

	   If serialization into the subnode throws the subnode is not
	   added to target and any exception is propagated back to the
	   caller.

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename SerializableT >
        bool serialize_subnode( DataNodeType & target,
                                const std::string & nodename,
                                const SerializableT & src )
        {
                NODEERR << "serialize_subnode<>(DataNodeType, '"<<nodename<<"', SerializableT)\n";
                std::auto_ptr<DataNodeType> sub(new DataNodeType);
                typedef ::s11n::node_traits<DataNodeType> NT;
                NT::name( *sub, nodename );
                if( serialize<DataNodeType,SerializableT>( *sub, src ) )
                {
			DataNodeType * n = sub.release(); // take over ownership;
                        NT::children(target).push_back( n );
			return true;
                }
                else
                {
			sub.reset(0); // delete content
		}
                return false;
        }


        /**
           If a child named subnodename is found in src then this function
           returns deserialize( child, target ) and returns it's result, otherwise
           it returns 0.

	   The function might throw: it uses the two-arg form of
	   deserialize() and therefor inherits that function's support
	   of throw_policy().

           This is a convenience function: not part of the s11n
           kernel.
        */
        template <typename DataNodeType, typename DeserializableT>
        bool deserialize_subnode( const DataNodeType & src,
                                  const std::string & subnodename,
                                  DeserializableT & target )
        {
                const DataNodeType * ch = find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize<DataNodeType,DeserializableT>( *ch, target );

        }

        /**
           If a child named subnodename is found in src then this function
           returns the result of deserialize(child), otherwise
           it returns 0.

	   The function might throw: it uses the two-arg form of
	   deserialize() and therefor inherits that function's support
	   of throw_policy().

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename DeserializableT>
        DeserializableT * deserialize_subnode( const DataNodeType & src,
                                               const std::string & subnodename )
        {
                const DataNodeType * ch = find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize<DataNodeType,DeserializableT>( *ch );
        }


        /**
           Clones an arbitrary SerializableType using it's
           DataNodeType serialization implementation.

           Returns a clone of tocp, or returns 0 on error.
           The caller owns the returned pointer.

           This copy is polymorphism-safe as long as all participating
           Serializables (re)implement the appropriate de/serialize
           operations, similarly to as they would do for a copy ctor
           or classical Clone() member function.

           Tip: clone() is a convenient way to test new de/serialize
           functions, e.g., for new Serializables, because if it works
           then deserializing from streams/files will also work. This
           function takes SerializableType through the whole
           de/serialize process, including classloading.

	   This function was renamed from clone() in version 1.1.

	   Exceptions and errors:

	   This function may return 0 or throw on an error, as dictated
	   by serialize() and then deserialize() (in that order).
        */
        template <typename DataNodeType, typename SerializableType>
        SerializableType * s11n_clone( const SerializableType & tocp )
        {
                DataNodeType node;
                if( ! serialize( node, tocp ) ) return 0;
                return deserialize<DataNodeType,SerializableType>( node );
        }
	/**
	   DEPRECATED name for s11n_clone().

	   @deprecated s11n_clone()
	*/
        template <typename DataNodeType, typename SerializableType>
        SerializableType * clone( const SerializableType & tocp )
        {
		return s11n_clone<DataNodeType,SerializableType>( tocp );
	}


        /**
           "Casts" t1 to t2 using serialization. This will work
           whenever t1 and t2 are "semantically compatible", whatever
           that really means. It can be used, e.g., to copy a
           list&lt;int&gt; to a vector&lt;double&gt;, provided both
           types have been proxied.

           Note that in the case of containers, the pointerness of the
           contained types is irrelevant: this works on both, thus
           a list&lt;int&gt; can be "cast" to a vector&lt;double*&gt;.

           As usual for a failed deserialization, if it returns false
           then t2 may be in an undefined state. There is no guaranty,
           however, that t2's deserialize operator will ever be
           called, as the serialization of t1 must first succeed
           for that to happen.

	   Exceptions and errors:

	   This function may return 0 or throw on an error, as dictated
	   by serialize() and then deserialize() (in that order).
        */
        template <typename NodeType, typename Type1, typename Type2>
        bool s11n_cast( const Type1 & t1, Type2 & t2 )
        {
                NodeType n;
                return serialize<NodeType,Type1>( n, t1 ) && deserialize<NodeType,Type2>( n, t2 );
        }


} // namespace s11n

#undef NODEERR

#endif // s11n_DATA_SERIALIZE_H_INCLUDED
