#ifndef t5x_CLASS_NAME_H_INCLUDED
#define t5x_CLASS_NAME_H_INCLUDED 1
#include <cassert>
#include <string>
#include <typeinfo>
// #include <iostream> // cerr

#include <map>

namespace { // ns is not strictly necessary: works in global-scope as well.
        // anonymous NS is very helpful in some contexts, though.

        /**
           A utility class to provide human-usable class names,
           available at runtime.

           It MUST be specialized to work.

           Call one of these macros:

<pre>
CLASS_NAME(Type)
CLASS_NAME_ALIAS(Type,AliasForType)
</pre>

	from a class' header file (not an impl file, for complex
	linking reasons). It may only be called one time per Type per
	compilation unit, or you will get specialization collisions at
	compile time.

        Caveats:

           - template types with commas in the names will break the
           macro, requiring hand-specialization for these types.

           - typedef'd names will get their typedef'd name registered,
           not their real name. Maybe a feature, maybe not.

           - classes which want to use class_name on themselves will
           have to do some forward declarations and call CLASS_NAME()
           before using it.

           To get proper names for such cases you must hand-specialize
           this class. See the macros for a sample implementation, or
           the class_names shell script which should have come with
           this class.
        */
        template <class T>
        struct class_name
        {
                typedef T value_type;
                /** returns the class name for class T. */
                static const char * name()
                {
                        static const std::string tid = typeid((value_type *)0).name();
//                         std::cerr << "this class_name<typeid=["
//                                 + tid
//                                 + "]> is unspecialized! See " << __FILE__ << " for the solution!\n";
                        return tid.c_str();
                }
        };

        /**
           A specialization to avoid the necessity to register, e.g.,
           CLASS_NAME(Foo) and CLASS_NAME(Foo *). Note that it returns
           the exact same name as the non-pointer variant, as this
           part is not relevant for the purposes i put this class
           through (classloading/[de]serializing), and in fact the
           extra token requires a lot of extra work down-stream.
        */
        template <class T>
        struct class_name<const T *>
        {
                typedef T base_type;
                static const char * name()
                {
                        return class_name<base_type>::name();
                }
        };

        template <class T>
        struct class_name<T *>
        {
                typedef T base_type;
                static const char * name()
                {
                        return class_name<base_type>::name();
                }
        };
} // namespace

/**
   Shorthand form of class_name<T>::name().
*/
template <typename T>
const char * classname()
{
        return class_name<T>::name();
}


namespace {
////////////////////////////////////////////////////////////////////////////////
// i get internal compiler errors with these: (gcc 3.3.1)
////////////////////////////////////////////////////////////////////////////////
//         template <class K, class M>
//         struct class_name< std::map<K,M> >
//         {
//                 typedef K key_type;
//                 typedef M mapped_type;
//                 typedef std::map<key_type,mapped_type> base_type;
//                 static const char * name()
//                 {
// //                         static const std::string n = 
// // 				std::string("std::map") + "<" +
// // 				::classname< key_type >() + "," + 
// // 				::classname< mapped_type >() + ">"; 
// //                         return n.c_str();
//                         return classname.c_str();
//                 }
//                 static std::string classname;

//         };
//         template <class K, class M>
//         std::string class_name< std::map<K,M> >::classname = 
// 				std::string("std::map") + "<" +
// 				::classname< K >() + "," + 
// 				::classname< M >() + ">"; 


//         template <class F, class S>
//         struct class_name< std::pair<F,S> >
//         {
//                 typedef F first_type;
//                 typedef S second_type;
//                 typedef std::pair<first_type,second_type> base_type;
//                 static const char * name()
//                 {
//                         return classname.c_str();
//                 }
//                 static std::string classname;

//         };
//         template <class F, class S>
//         std::string class_name< std::pair<F,S> >::classname = 
// 				std::string("std::pair") + "<" +
// 				::classname< F >() + "," + 
// 				::classname< S >() + ">"; 


//         template <class V>
//         struct class_name< std::list<V> >
//         {
//                 typedef V value_type;
//                 static const char * name()
//                 {
//                         return classname.c_str();
//                 }
//                 static std::string classname;
//         };
//         template <class V>
//         std::string class_name< std::list<V> >::classname = 
// 				std::string("std::pair") + "<" +
// 				::classname< V >() + ">"; 


}

////////////////////////////////////////////////////////////////////////        
//    Creates a class_name<> specialization for Type, using the class
//    name Alias.
////////////////////////////////////////////////////////////////////////        
#define CLASS_NAME_ALIAS(Type,Alias) \
	namespace { template <> \
	struct class_name< Type > {  static const char * name() { return # Alias;  } };}

#define CLASS_NAME(Type) CLASS_NAME_ALIAS(Type,Type)
////////////////////////////////////////////////////////////////////////        



#endif // t5x_CLASS_NAME_H_INCLUDED
