#ifdef NDEBUG
#  undef NDEBUG // want assert() to always work
#endif

#include <cassert>
#include <utility> // pair
#include <list>
#include <map>
#include <memory> // auto_ptr
#include <typeinfo>

#include <s11n.net/cl/class_loader.hpp>
#include <s11n.net/cl/cl_debuggering_macros.hpp>
#include <s11n.net/s11n/io/serializers.hpp>
#include <s11n.net/s11n/pods_streamable.hpp>
#include <s11n.net/acme/argv_parser.hpp>
#include <s11n.net/acme/algo.hpp> // free_map_entries()
#include <s11n.net/s11n/traits.hpp>
#include <s11n.net/s11n/s11n_node.hpp>
#include <s11n.net/s11n/list.hpp>

// #define NODE_TYPE s11n::data_node
#define NODE_TYPE s11n::s11n_node

#include "LokiTypeInfo.h"

namespace s11n { namespace proxy {

        template <typename NodeT, typename SerBaseT>
        class base_proxy
        {
        public:
                typedef NodeT node_type;
                typedef SerBaseT serializable_type;

                base_proxy()
                {
                }
#define CONST_UNDECIDED
// #define CONST_UNDECIDED const
                virtual ~base_proxy() {}


                bool operator()( const node_type & src, serializable_type & dest ) CONST_UNDECIDED
                {
                        return this->deserialize( dest, src );
                }

                bool operator()( node_type & dest, const serializable_type & src ) CONST_UNDECIDED
                {
                        return this->serialize( dest, src );
                }

        protected:
                virtual bool deserialize( const node_type & src, serializable_type & dest ) CONST_UNDECIDED = 0;

                virtual bool serialize( node_type & dest, const serializable_type & src ) CONST_UNDECIDED = 0;
#undef CONST_UNDECIDED
        };

        /**

        ContainerT must not be a pointer type.

        CleanerFunc must implement:

        WHATEVER operator()( ContainerT & ) const;

        (It may be a member template, and ContainerT may actually be a
        pointer type in this case.)

        The operator should deallocate all elements of the given
        container and empty the container. It is not required to
        recursively clean containers of containers, though that can
        potentially be accomplished by specializing
        acme::object_deleter<Container>.

        */
        template <typename ContainerT, typename CleanerFunc>
        struct container_cleaner
        {
                typedef ContainerT container_t;
                typedef CleanerFunc cleaner_functor_t;
                container_cleaner()
                {
                }

                /**
                   Returns this object's container.
                */
                container_t & container()
                {
                        return this->m_container;
                }

                /**
                   Calls cleaner_functor_t()( container() )
                */
                ~container_cleaner()
                {
                        cleaner_functor_t()( this->container() );
                }
        private:
                container_t m_container;
        };

        /**
           For use with container_cleaner<SomeMapType,map_cleaner>.
        */
        struct map_cleaner
        {
                template <typename MapT>
                void operator()( MapT & m ) const
                {
                        acme::free_map_entries( m );
                }
        };

        /**
           For use with container_cleaner<SomeListType,list_cleaner>.
        */
        struct list_cleaner
        {
                template <typename ListT>
                void operator()( ListT & m ) const
                {
                        acme::free_list_entries( m );
                }
        };

        /**
           A convenience wrapper to avoid having to specify templates types
           so much in client code, by wrapping them up in a single
           proxy_tool<...> typedef.
        */
        template <typename NodeT, typename SerBaseT>
        struct proxy_tool
        {
                typedef NodeT node_type;
                typedef SerBaseT serializable_type;
                typedef base_proxy<node_type,serializable_type> proxy_type;

                /**
                   This function takes over ownership of p. It is
                   deleted during the static destruction phase
                   (post-main()).
                */
                static void register_proxy( const serializable_type * s, proxy_type * p )
                {
                        typename map_t::iterator it = map().find(key(s));
                        if( map().end() != it )
                        {
                                delete (*it).second;
                                map.erase( it );
                        }
                        map()[key(s)] = p;
                }

                /**
                   Caller does NOT own the returned object (which may
                   be 0). It is owned by this class.
                */
                static proxy_type * get_proxy( const serializable_type * s )
                {
                        typename map_t::iterator it = map().find(key(s));
                        return (map().end() == it) ? 0 : (*it).second;
                }

        private:
                typedef Loki::TypeInfo key_type;
                typedef std::map<key_type,proxy_type *> map_t;
                static key_type key( const serializable_type * s )
                {
                        return Loki::TypeInfo(typeid(s));
                }

                static map_t & map()
                {
                        static container_cleaner<map_t,map_cleaner> bob;
                        return bob.container();
                }
        };

} } // namespace s11n::proxy

typedef std::list<int> ListT;


#define S11N_TYPE ListT
#define S11N_TYPE_NAME "intlist"
#define S11N_SERIALIZE_FUNCTOR s11n::list::streamable_list_serializable_proxy
#include <s11n.net/s11n/reg_serializable_traits.hpp>

int test_dproxy()
{
        typedef s11n::proxy::base_proxy<NODE_TYPE,ListT> BP;

        ListT l;
        for( int i = 0; i < 10; i++ )
        {
                l.push_back(i);
        }

        NODE_TYPE node;
        bool worked = s11n::serialize( node, l );
        assert( worked && "serialize<NODE_TYPE,ListT>() failed!" );

        typedef s11n::io::data_node_serializer<NODE_TYPE> IOT;
        typedef std::auto_ptr<IOT> IOTP;
        IOTP io = IOTP( s11n::io::create_serializer<NODE_TYPE>( "parens" ) );
        assert( io.get() && "Loading Serializer failed!" );

        worked = io->serialize( node, std::cerr );
        assert( worked && "io->serialize() failed :(" );

        return 0;
}

acme::argv_parser argv;

int main(int argc, char *_argv[])
{
        argv = acme::argv_parser::args(argc,_argv);
        if (argv.is_set("d"))
        {
                cl::class_loader_debug_level(1);
        }

        int ret = test_dproxy();

        CERR << "Goodbye, world!\n";
        return ret;
}

