// proxied_struct.h
//
// see proxied_struct.cpp for notes
//
//
#ifndef __PROXIED_STRUCT__
#define __PROXIED_STRUCT__

#include <vector>
#include <iostream>
#include <s11n.net/s11n/s11nlite.hpp>    

// an example struct. It has no serialization functions.
struct elem_t {
  int 		index;
  double	value;
  elem_t(void) :  index(-1), value(0.0) {}
  elem_t(int i, double v) : index(i), value(v) {}
};

// add a functor to proxy the serialization of elem_t
//
// Note that this could be done in a separate file, making code management
// and experimentation simple
//

struct elem_t_s11n {
        // serialize
        template <typename NodeType>
        bool operator()( NodeType &dest, const elem_t &src ) const {
                typedef s11n::node_traits<NodeType> NTR;
                NTR::class_name( dest, "elem_t" );
                NTR::set( dest,"i", src.index );
                NTR::set( dest, "v", src.value );
                return true;
        }
        // deserialize
        template <typename NodeType>
        bool operator()( const NodeType &src, elem_t &dest ) const {
                typedef s11n::node_traits<NodeType> NTR;
                dest.index = NTR::get( src, "i", -1 );
                dest.value = NTR::get( src, "v", 0.0 );
                return true;
        }
};

#include <s11n.net/s11n/list.hpp> // list_serializable_proxy


// Now we tell s11n to use this proxy when it needs to serialize an
// elem_t struct
//
// Note that when the serialization functions are written into a class,
// we register the class by using:
//   #include <s11n.net/s11n/reg_serializable.hpp>
// but when we use a proxy, we register the proxy by using
//   #define S11N_SERIALIZE_FUNCTOR functor_name  
// as shown below, to tell s11n the name of the proxy functor
//////////////////////////////////////////////////////////////////////
// Register the proxy with s11n. We match the TYPE we'd like to
// proxy with the TYPE of the proxy, and supply a class name
// for the proxIED type with it's classloader:
#define S11N_TYPE elem_t                     // TYPE to proxy
#define S11N_TYPE_NAME "elem_t"                   // type NAME
#define S11N_SERIALIZE_FUNCTOR elem_t_s11n   // PROXY type
#include <s11n.net/s11n/reg_proxy.hpp>                  // registration code
//////////////////////////////////////////////////////////////////////
// This registration process can be repeated as many times as needed
// on different types. The reg_proxy.h file undefines the above #defines, 
// so it can be included again and again.
//

// // we'll want a vector of elem_t in our SparseVector class
// typedef std::vector<elem_t> elem_t_vect;
 
// // Tell s11n how to serialize vectors of elem_t
// #define S11N_TYPE elem_t_vect
// #define S11N_TYPE_NAME "elem_t_vect"
// #define S11N_SERIALIZE_FUNCTOR s11n::list::list_serializable_proxy   // use a proxy
// #include <s11n.net/s11n/reg_proxy.hpp>
 


// SparseVector, a class that manages a vector of our structs
//
// Note that this class does not have any serialization functions.
// We'll also add them via a proxy functor
//
// The only change we need to make is to allow our functor to have
// access to the private variables. We do that be declaring it to 
// be a friend of this class.
class SparseVector
{
friend struct SparseVector_s11n;

public:
  // default constructor, destructor, and copier

  void  add_element(int i, double v) { elements.push_back(elem_t(i, v)); }
  void  print(void) { 
            for (std::vector< elem_t >::iterator i=elements.begin(); 
                             i!=elements.end(); ++i) {
               std::cout << (*i).index << " " << (*i).value << std::endl;
            }         
        }

private:
  std::vector< elem_t >	elements;
};


// serialization proxy functor for SparseVector
struct SparseVector_s11n 
{
  // serialization
  bool operator()( s11nlite::node_type &dest, const SparseVector &src ) const {
    bool worked = s11nlite::serialize( s11n::create_child(dest, "vect"),  src.elements);
    return worked;
  }
  // deserialization
  bool operator()( const s11nlite::node_type &src, SparseVector &dest ) const {
    const s11nlite::node_type* n = s11nlite::find_child(src, "vect");
    if (!n) { return false; }
    bool worked = s11nlite::deserialize(*n, dest.elements);
    return worked;
  }
};




////////////////////////////////////////////////////////////////////////
// Register the SparseVector serialization proxy with s11n
#define S11N_TYPE SparseVector
#define S11N_TYPE_NAME "SparseVector"
#define S11N_SERIALIZE_FUNCTOR SparseVector_s11n
#include <s11n.net/s11n/reg_proxy.hpp>
                                                                                           




#endif // __PROXIED_STRUCT__

