////////////////////////////////////////////////////////////////////////////////
// main_dn.cpp
// Implmentation for s11nconvert utility, to convert data between the various
// Serializers.
// Author: stephan@s11n.net
// License: Public Domain
////////////////////////////////////////////////////////////////////////////////


#include <iostream>
#include <string>
#include <memory> // auto_ptr
#include <sys/time.h> // gettimeofday()
#include <unistd.h> // isatty()

////////////////////////////////////////////////////////////////////////
// We do this to cut down compile times, as s11nconvert doesn't actually
// deal with Serializables (but with Serializers).
#define s11n_NO_DEFAULT_POD_REGISTRATIONS 1
////////////////////////////////////////////////////////////////////////

#include <s11n.net/acme/pointer_cleaner.hpp> // pointer_cleaner class
#include <s11n.net/acme/algo.hpp> // free_list_entries<>()
#include <s11n.net/acme/argv_parser.hpp> // argv_parser class
#include <s11n.net/zfstream/zfstream.hpp> // get_i/ostream()
#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // CERR
#include <s11n.net/s11n/s11n.hpp> // s11n core framework
#include <s11n.net/s11n/s11nlite.hpp> // s11nlite framework
#include <s11n.net/s11n/io/serializers.hpp> // utility code for s11n::io
#include <s11n.net/cl/dll_util.hpp> // cllite dll utils.

bool verbose = false;
#define VERBOSE if( verbose ) CERR

typedef acme::argv_parser ARGV;


void show_version_info( std::ostream & os )
{
        os << ARGV::args().get( "$0", "??? WTF ???" ) << " version " <<  s11n_S11N_LIBRARY_VERSION << "\n";
        os << "License: " << s11n_PACKAGE_LICENSE << std::endl;
}

/**
   t2 must be newer than or equal to t1.
*/
size_t time_udiff( timeval & t1, timeval & t2 )
{
        size_t d = (t2.tv_sec - t1.tv_sec) * 1000;
        d += (t2.tv_usec - t1.tv_usec);
        return d;
}

int
main( int argc, char **argv )
{
#define ERR std::cerr << "s11nconvert error: "        
        ARGV & args = ARGV::args( argc, argv );


        if( args.is_set( "v" ) || args.is_set( "verbose" ) )
        {
                verbose = true;
        }
        VERBOSE << "Verbose mode on.\n";
        if( args.is_set( "cldebug" ) )
        {
                cl::class_loader_debug_level( 1 );
                VERBOSE << "classloader debugging enabled.\n";
        }

        if( args.is_set( "clex" ) )
        {
                cllite::use_exceptions( true );
                VERBOSE << "classloader exception throwing enabled.\n";
        }


        enum {
        ErrorNoSerializer = 1,
        ErrorFileNotFound,
        ErrorNoInputFile,
        ErrorInputReadFailed,
        ErrorOutputOpenFailed,
        ErrorLoadNodeFailed,
        ErrorSaving,
        ErrorDLL,
        ErrorForcedSerializer,
        ErrorUsageError,
        ErrorNoSerializers
        };


        /////////////////////////////////////////////////////////////////////
        // load requested DLLs
        typedef std::list<cllite::dll::dlentry> DList;
        std::string dlarg;
        DList dlllist;
        if( 0 != cllite::dll::parse_dlentry_flags( argc -1 , argv+1, dlllist ) )
        {
                DList::const_iterator it = dlllist.begin(), et = dlllist.end();
                std::string dl;
                bool dltolerant;
                bool clexceptions = cllite::use_exceptions();
                for( ; et != it; ++it )
                {
                        dl = "";
                        dlarg = (*it).name;
                        dltolerant = (*it).tolerant;
                        try
                        {
                                cllite::use_exceptions(false);
                                VERBOSE << "Trying to load dll '"<<dlarg<<"'...\n";
                                dl = cllite::open_dll( dlarg );
                                cllite::use_exceptions(clexceptions);
                                if( dl.empty() )
                                { // try harder...
                                        dl = cllite::open_dll( dlarg + "_serializer" );
                                }
                                if( (!dltolerant) && dl.empty() )
                                {
                                        throw cllite::cl_exception( "DLL '"+dlarg+ "' not found." );
                                }
                        }
                        catch( const cllite::cl_exception & ex )
                        {
                                if( !dltolerant )
                                {
                                        ERR << "["<<dlarg<<"] DLL EXCEPTION: " << ex.what() << "\n";
                                        return ErrorDLL;
                                }
                                else
                                {
                                        VERBOSE << "["<<dlarg<<"] DLL EXCEPTION: " << ex.what() << "\n";
                                }
                        }
                        if( dl.empty() )
                        {
                                VERBOSE << "Failed to open dll for '"<<dlarg<<"'.\n";
                        }
                        else
                        {
                                VERBOSE << "Opened DLL ["<<dl<<"]"<<std::endl;
                        }
                }
        }
        // end DLL loading.
        ////////////////////////////////////////////////////////////////////////

        ////////////////////////////////////////////////////////////////////////
        // Load list of known/registered Serializers.
        // Do this AFTER loading DLLs so that DLLs may install additional Serializers...
        std::string known_ser;
        typedef std::list< std::string > SerList;
        SerList serlist;
        s11n::io::serializer_list<s11nlite::node_type>( serlist );
        SerList::const_iterator cit = serlist.begin(),
                cet = serlist.end();
        for( ; cet != cit; ++cit )
        {
                known_ser += (*cit) + " ";
        }
        if( known_ser.empty() )
        {
                CERR << "ERROR: no data formats loaded! Try the -dl CLASSNAME option!\n";
                return ErrorNoSerializers;
        }


        if( args.is_set( "known-serializers" ) || args.is_set( "K" ) )
        {
                std::cout << known_ser << std::endl;
                return 0;
        }
        ////////////////////////////////////////////////////////////////////////


        args.set_help( "?, help",
                       "print help text" );

        args.set_help( "dl dllname",
                       std::string("Opens the given DLL. ")
                       +"May be used multiple times. All DLLs are loaded before the Serializer (-s) is loaded, "
                       +"DLL names may be absolute file names or base filenames if those files are in the lib path. "
                       +"This is useful when, e.g., the Serializer treats filenames as db keys or something similar."
                       +" Use -v to see information about what DLLs get loaded this way."
                       );

        args.set_help( "DL dllname",
                       std::string("Exacly like -dl")+
                       ", but does not abort s11nconvert if the dll is not found or if there is an error opening it. "+
                       "In s11n slang this is called a \"tolerant dll\"." );


        args.set_help( "f filename, file=filename",
                       "specify input filename. Same as --file. Filename of - means stdin." );
        args.set_help( "z",
                       "Compress output with zlib compression (works only for files)." );
        args.set_help( "bz",
                       "Compress output with bz2 compression (works only for files)." );
	args.set_help( "o filename, output=filenmae",
                       "specify output filename. Filename of - means stdout." );

        args.set_help( "cldebug",
                       "toggles on classloader debug output." );


        args.set_help( "K, known-serializers",
                       std::string("Dumps a list of all registered Serializers, including any loaded via -dl or -DL.") );


        args.set_help( "s SERIALIZER, format SERIALIZER",
                       std::string("output format (i.e., a Serializer's name). ") +
                       "Built-in serializers:\n\t" + known_ser + 
                       "\n\t(Note: some of those different names might actually be the same Serializer.)" +
                       "\n\tAny name which is valid for s11nlite::create_serializer() is accepted. " +
                       "\n\tPlease read the library manual and be aware of any caveats/limitations for any given Serializer."
                       );

        args.set_help( "S INPUT_SERIALIZER",
                       std::string()
                       +"FORCE a given Serializer for input. Use this in conjunction with -dl DLLNAME to load a Serializer "
                       +"from a DLL and force it to act as the input handler."
                       );


        args.set_help( "v, verbose",
                       "Verbose mode (more output), sent to stderr so it won't interfere with serialized output." );

        args.set_help( "version, V",
                       "print app and libs11n version info and quit." );


	if ( args.is_help_set() )
	{
                std::ostream & os = std::cout; 
                show_version_info( os );
                os << "Command-line arguments:\n"
                   << args.dump_help()
                   << "\nThe -gz and -bz options only work if your "
                   << "libzfstream was compiled with the corresponding compression support.\n"
		   << "\n";
                   ;
                return 0;
	}

        if( args.is_set( "version" ) || args.is_set( "V" ) )
        {
                show_version_info( std::cout );
                return 0;
        }



        if( args.is_set( "z" ) && args.is_set( "bz" ) )
        {
                ERR << "-z and -bz may not be used together." << std::endl;
                return ErrorUsageError;
        }


        std::string fmt = args.get( "s", args.get( "format", "" ) );
        if( fmt.empty() )
        {
                ERR << "No output format specified. "
                    << "Try using [-s or --format] with one of: "
                    << known_ser << "\n";
                return ErrorNoSerializer;
        }

        

        typedef std::auto_ptr<s11nlite::serializer_base_type> APSER;
        APSER ser = APSER( s11nlite::create_serializer( fmt ) );
        if( ! ser.get() )
        {
                ERR << "No Serializer found for name '"<<fmt<<"'. Try one of: "
                    << known_ser << "\n";
                return ErrorNoSerializer;
        }

 
        zfstream::CompressionPolicy comp = zfstream::compression_policy();


        bool usegz = false;
        bool usebz = false;
        if( args.is_set( "z" ) )
        {
                usegz = true;
                comp = zfstream::GZipCompression;
        }
        else if( args.is_set( "bz" ) )
        {
                usebz = true;
                comp = zfstream::BZipCompression;
        }
        zfstream::compression_policy( comp );



        std::string ifname = args.get( "f", args.get( "file", "" ) );
        typedef acme::pointer_cleaner<std::istream> IStreamClean;
        IStreamClean icleaner;
        std::istream * is = 0;
        bool use_infile = false;
        if( ( "-" == ifname ) || ! isatty(STDIN_FILENO) )
        {
                is = &std::cin;
        }
        else if( ifname.empty() )
        {
                ERR << "No input file specified.\n";
                return ErrorNoInputFile;
        }
        else
        {
                use_infile = true;
        }



        std::ostream * os = 0;

        std::string ofname = args.get( "o", args.get( "output","" ) );
        bool use_ofile = false;
        if( ofname.empty() || ("-" == ofname) )
        {
                if( usegz || usebz )
                {
                        ERR << "Compression (-z and -bz) works only for files, not cout." << std::endl;
                        return ErrorUsageError;
                }
                os = &std::cout;
        }
        else
        {
                use_ofile = true;
        }

        timeval time1 = timeval(); // to measure load/save times.
	timeval time2 = timeval();

        typedef std::auto_ptr<s11nlite::node_type> NAP;
        NAP innode;
        std::string forcedinser = args.get( "S", std::string() );
        if( ! forcedinser.empty() )
        {
                VERBOSE << "Forcing input serializer: " << forcedinser << "\n";
                s11nlite::serializer_base_type * forceser = s11nlite::create_serializer( forcedinser );
                if( 0 == forceser )
                {
                        ERR << "Could not load forced input serializer '"<<forcedinser<<"'!\n";
                        return ErrorForcedSerializer;
                }
                gettimeofday( &time1, 0 );
                innode = NAP( use_infile
                              ? forceser->deserialize( ifname )
                              : forceser->deserialize( *is ) );
                gettimeofday( &time2, 0 );
        }
        else
        {
                gettimeofday( &time1, 0 );
                innode = NAP( use_infile ? s11nlite::load_node( ifname ) : s11nlite::load_node( *is ) );
                gettimeofday( &time2, 0 );
        }
        if( ! innode.get() )
        {
                ERR << "Error loading node tree from input stream!\n";
                return ErrorLoadNodeFailed;
        }
        VERBOSE << "Node load time: " << time_udiff( time1, time2 ) << " useconds.\n";

        // And FINALLY... we get down to the real work...
        // these couple lines almost now don't seem worth all the hassle ;)
        
        gettimeofday( &time1, 0 );
        bool workie = use_ofile ? ser->serialize( *innode, ofname ) : ser->serialize( *innode, *os );
        gettimeofday( &time2, 0 );
        if( ! workie )
        {
                ERR << "Error saving output!\n";
                return ErrorSaving;
        }
        VERBOSE << "Node save time: " << time_udiff( time1, time2 ) << " useconds.\n";

	return 0;
#undef ERR
};



