/**
   Demonstration client app for s11nlite.

   Generates player characters for the Enondas roleplaying system.

   Demonstrates:

   - Serializing app-wide state.

   - A serializable client-side class.

   - Serialization of rather complex container combinations,
     e.g. map<string,map<string,double>>


Sample usage:

./enondas -d enondas.s11n -s compact -name Bob -kindred dwarf -dice 4 -droplow


Args:

-d DATAFILE (default=enondas.s11n)

-s SERIALIZER (for s11nlite)

-name "PC's NAME"

-kindred pc_race (must be defined in DATAFILE)

-dice X (number of dice to roll for attributes)

-droplow (if set, lowest attribute die is dropped).

-ds ("dump state", dumps out the DATAFILE data, so we can see if it
    was all read in.)

-seed UNSIGNED_LONG (RNG seed). If you don't use this, running the app
   several times in rapid succession may generate the same PC info!

-dump (human-readable dump of generated PC).

-o OUTFILE. Save PC to given file, defaulting to stdout. If
   -dump is used and -o is not used, the data is NOT sent
   to stdout.


*/

#include <s11n.net/s11n/s11nlite.hpp>
#include <s11n.net/s11n/node_traits_wrapper.hpp>
#include <s11n.net/s11n/pods_streamable.hpp>
#include <s11n.net/s11n/list.hpp>
#include <s11n.net/s11n/map.hpp>
#include <s11n.net/acme/argv_parser.hpp>
#include <s11n.net/acme/random.hpp>


#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // CERR

#include <stdlib.h> // random()
#include <time.h> // time()


typedef std::list<std::string> StringList;

/** Global list of available PC attributes. STR, DEX, etc. */
StringList attribute_list;

/** List of available kindred (races). */
StringList kindred_list;

/** map[attribute] = modifier */
typedef std::map<std::string,double> AttrModMap;

/** map[kindred][attribute] = modifier */
typedef std::map<std::string,AttrModMap> KindredModMap;

/**
   map[kindred][attribute] = modifier
*/
KindredModMap kindred_mods_map;

/**
   A player-character (PC) for the Enondas system.
*/
class enondas_pc
{
public:
        enondas_pc() : m_name("Unnamed"), m_kindred("human")
        {
                this->init();
        }

        explicit enondas_pc( const std::string & name, const std::string & race ) : m_name(name), m_kindred(race)
        {
                this->init();
        }

        ~enondas_pc() {}

        typedef std::map<std::string,int> attr_map;

        bool operator()( s11nlite::node_type & dest ) const
        {
                s11n::node_traits_wrapper<s11nlite::node_type> tr(dest);
                tr.class_name( "enondas_pc" );
                tr.set( "gender", this->m_gender );
                tr.set( "name", this->m_name );
                tr.set( "kindred", this->m_kindred );
                s11n::map::serialize_map( dest, "attributes", this->m_attr );
                return true;
        }

        bool operator()( const s11nlite::node_type & src )
        {
                s11n::node_traits_wrapper<s11nlite::node_type> tr(src);
                this->m_name = tr.get( "name", this->m_name );
                this->m_kindred = tr.get( "kindred", this->m_kindred );
                s11n::map::deserialize_map( src, "attributes", this->m_attr );
                this->m_gender = tr.get( "gender", this->m_gender );
                return true;
        }

        attr_map & attr() { return this->m_attr; }
        const attr_map & attr() const { return this->m_attr; }

        std::string name() const { return this->m_name; }
        void name( const std::string & n ) { this->m_name = n; }

        std::string kindred() const { return this->m_kindred; }
        void kindred( const std::string & n ) { this->m_kindred = n; }

        int gender() const { return this->m_gender; }
        void gender( int g ) { this->m_gender = g; }


        int adds() const
        {

                int adds = 0;
                int at;
                attr_map bogo = this->m_attr; // to avoid having to use iterators
                // when doing the attr lookups (map[] is non-const).
#define ADDS(A) at = bogo[A]; \
                adds += (at < 9 ? ((9-at)*-1) : (at>12 ? (at-12) : 0));
                ADDS("STR");
                ADDS("LK");
                ADDS("DEX");
                return adds;
#undef ADDS
        }

private:
        void init()
        {
                StringList::const_iterator cit = attribute_list.begin(),
                        cet = attribute_list.end();
                for( ; cet != cit; ++cit )
                {
                        this->m_attr[*cit] = 0;
                }
        }

        std::string m_name;
        std::string m_kindred;
        int m_gender; // 1 == male, 0 == female.
        attr_map m_attr;

};

#define S11N_TYPE enondas_pc
#define S11N_TYPE_NAME "enondas_pc"
#include <s11n.net/s11n/reg_serializable_traits.hpp>


/**
   Class to save/restore app-wide state.
*/
struct enondas_state
{

        // serialize
        template <typename NodeT>
        bool operator()( NodeT & dest ) const
        {
                typedef s11n::node_traits<NodeT> TR;
                TR::class_name( dest, "enondas_state" );
                s11n::list::serialize_list( dest, "attributes", attribute_list );
                s11n::list::serialize_list( dest, "kindred", kindred_list );
                s11n::map::serialize_map( dest, "kindred_modifiers", kindred_mods_map );
                return true;
        }

        // deserialize
        template <typename NodeT>
        bool operator()( const NodeT & src ) const
        {
                typedef s11n::node_traits<NodeT> TR;
                attribute_list.clear();
                s11n::list::deserialize_list( src, "attributes", attribute_list );
                s11n::list::deserialize_list( src, "kindred", kindred_list );
                s11n::map::deserialize_map( src, "kindred_modifiers", kindred_mods_map );
                return true;
        }
};

#define S11N_TYPE enondas_state
#define S11N_TYPE_NAME "enondas_state"
#include <s11n.net/s11n/reg_serializable_traits.hpp>

/**
   A human-friendly dump of a PC's state.
*/
void dump_pc( enondas_pc & pc, std::ostream & os )
{

        os << "Name:\t" << pc.name() << "\n";
        os << "Kindred: " << (pc.gender() ? "male" : "female") << " " << pc.kindred() << "\n";

        StringList::const_iterator cit = attribute_list.begin(),
                cet = attribute_list.end();
        std::string attr;
        for( ; cet != cit; ++cit )
        {
                attr = (*cit);
                os << attr << ":\t" << pc.attr()[attr] << "\n";
        }
        os << "Adds:\t" << pc.adds() << "\n";

//         s11nlite::save( pc, std::cout );

}


/**
   Rolls num d6. If drop_low is true then the lowest roll is discarded
   (used for 4d6/drop low attribute rolling system).
*/
size_t roll_d6( size_t num, bool drop_low = false )
{
        size_t accum = 0;
        size_t low = 6;
        size_t roll;
        for( size_t i = 0; i < num; i++ )
        {
                roll = acme::random( 1, 6 );
                low = (roll < low ? roll : low);
                accum += roll;
        }
        if( drop_low )
        {
                accum -= low;
                //CERR << "Dropping lowest die: " << low <<". Roll="<<accum<<"\n";
        }
        return accum;
}

/**
   Rolls attributes for the given pc, using roll_d6(dice,drop_low).
*/
void roll_attributes( enondas_pc & pc, size_t dice, bool drop_low )
{

        StringList::const_iterator cit = attribute_list.begin(),
                cet = attribute_list.end();
        double val;
        std::string attr;
        double mod;
        for( ; cet != cit; ++cit )
        {
                attr = *cit;
                val = roll_d6( dice, drop_low );
                mod = kindred_mods_map[pc.kindred()][attr];
                //CERR << "Attribute: " << attr << " = "<<val<<" (mod="<<mod<<")\n";
                val = val * mod;
                pc.attr()[attr] = static_cast<int>( val );
        }
        //pc.attr()["HP"] = pc.attr()["CON"];
}


int main( int argc, char ** argv )
{
        acme::argv_parser & args = acme::argv_parser::args( argc, argv );


        s11nlite::serializer_class( args.get( "s", "expat" ) );


        std::string datafile = args.get( "d", "enondas.s11n" );

        enondas_state * state = s11nlite::load_serializable<enondas_state>( datafile );
        if( ! state )
        {
                CERR << "Error loading game data from '"<<datafile<<"'!\n";
                return 1;
        }

        if( args.is_set( "ds" ) ) // dump state
        {
                CERR << "App state:\n";
                s11nlite::save( *state, std::cout );
        }

        delete state;

        acme::random(1,2); // kludge. that funk seeds RNG on first run,
        // but i wanna force a specific seed below...
        unsigned long seed = args.get<ulong>( "seed", ::time(NULL) );
        ::srandom( seed );

        enondas_pc pc;
        pc.name( args.get( "name", "unnamed" ) );
        pc.kindred( args.get( "kindred", "human" ) );
        pc.gender( args.get( "gender", acme::random(0,1) ) );
        roll_attributes( pc, args.get( "dice", 3 ), args.get_bool( "droplow", false ) );

//         COUT << "Character:\n";

        std::string ofile = args.get( "o", "" );
        if( ! ofile.empty() )
        {
                bool b = s11nlite::save( pc, ofile );
                CERR << (b ? "Saved" : "Error saving")
                     << " PC to file [" << ofile << "].\n";
        }
        else if( args.is_set( "dump" ) )
        {
                dump_pc( pc, std::cout );
        }
        else
        {
                s11nlite::save( pc, std::cout );
        }

        //CERR << "Adds = " << pc.adds() << "\n";
        
        return 0;

}
