////////////////////////////////////////////////////////////////////////
// s11nlite.cpp: impl file for s11nlite.h
// author: stephan@s11n.net
// license: public domain
////////////////////////////////////////////////////////////////////////
#include <list>
#include <string>

#include "s11nlite.hpp"
#include <s11n.net/s11n/s11n_config.hpp>
#include <s11n.net/cl/cllite.hpp>
#include <s11n.net/s11n/s11n_debuggering_macros.hpp>
#include <s11n.net/acme/environment.hpp>
#include <s11n.net/stringutil/stdstring_tokenizer.hpp>
#include <s11n.net/s11n/io/serializers.hpp> // create_serializer()

#define S11NLITE_CONFIG_FILE std::string("${HOME}/.s11nlite")
#define S11NLITE_SERIALIZER_KEY std::string("serializer_class")
namespace s11nlite {

        namespace sharing {
                struct sharing_context {};
        }


        /**
           Load some normally-available Serializers...
        */
        void s11nlite_init()
        {
                cllite::class_path().add_path( s11n_CONFIG_LIB_DIR );
                typedef std::list<std::string> SList;
                SList li;
#define SER(X) li.push_back( std::string(# X)+"_serializer" );
                SER(funtxt);
                SER(funxml);
                SER(compact);
                SER(parens);
                SER(simplexml);
                SER(wesnoth);
#if s11n_HAVE_LIBEXPAT
                SER(expat);
#endif
#undef SER
                const char * aload = ::getenv( "S11NLITE_AUTOLOAD_DLLS" );
                if( aload )
                {
                        stringutil::stdstring_tokenizer tok( aload,":" );
                        std::string t;
                        while( tok.has_tokens() )
                        {
                                li.push_back( tok.next_token() );
                        }
                }

                //CERR << "classpath: " << cllite::class_path().path_string() << "\n";
                SList::iterator it = li.begin(), et = li.end();
                for( ; et != it; ++it )
                {
                        std::string found;
                        try
                        {
                                found = cllite::open_dll( *it );
                        }
                        catch( const cllite::cl_exception & ex )
                        {
                                CERR << "Autoload of Serializer '"<<(*it)<<"' failed. :(\n"
                                     << ex.what() << "\n";
                                continue;
                        }
                        //CERR << "autoload '" <<(*it)<<"': " << found<<"\n";
                }

        }


        int dll_init_placeholder = (s11nlite_init(),0);

        /**
           Class name of current Serializer.
        */
// 	std::string m_serializer = S11NLITE_DEFAULT_SERIALIZER_TYPE_NAME;
        std::string m_configfile;

        node_type * m_confignode = 0;

        struct config_saver
        {
                // when this is dtor'd config() will be saved.
                ~config_saver()
                {
                        //CERR << "Saving s11nlite config file ["<<m_configfile<<"]...\n";
                        save( *m_confignode, m_configfile );
                        delete m_confignode;
                        m_confignode = 0;
                }
        };


//         int config_init = (config(),0);

        node_type & config()
        {
                static config_saver m_confsaver;
                if( ! m_confignode )
                {
                        m_confignode = node_traits_type::create( "s11nlite_config" );
                        acme::environment & env = acme::environment::env();
                        m_configfile = env.expand_vars( S11NLITE_CONFIG_FILE );
                        //CERR << "Loading s11nlite config file ["<<m_configfile<<"]\n";
                        node_type * tmp = load_node( m_configfile );
                        if( tmp )
                        {
                                *m_confignode = *tmp;
                                delete tmp;
                        }
                        else
                        {
                                CERR << "s11nlite config file ["<<m_configfile<<"] not found or loading failed. Creating it...\n";
                                if( ! save( *m_confignode, m_configfile ) )
                                {
                                        CERR << "Could not create ["<<m_configfile<<"]! You may want to create one to avoid these silly error messages!\n";
                                        return *m_confignode;
                                }
                        }
                }
                return *m_confignode;
        }


        void serializer_class( const std::string & c )
        {
                node_traits::set( config(), S11NLITE_SERIALIZER_KEY, c );
        }

        std::string serializer_class()
        {
                return node_traits::get( config(), S11NLITE_SERIALIZER_KEY, S11NLITE_DEFAULT_SERIALIZER_TYPE_NAME );
        }

        serializer_base_type *
        create_serializer( const std::string & classname )
        {
                return ::s11n::io::create_serializer<node_type>( classname );
        }

        serializer_base_type * create_serializer()
        {
                return create_serializer( serializer_class() );
        }


        node_type *
        find_child( node_type & parent,
                    const std::string subnodename )
        {
                return s11n::find_child_by_name( parent, subnodename );
        }

        const node_type *
        find_child( const node_type & parent,
                    const std::string subnodename )
        {
                return s11n::find_child_by_name( parent, subnodename );
        }

        bool save( const node_type & src, std::ostream & dest )
        {
                typedef std::auto_ptr<serializer_base_type> AP;
                AP s = AP(create_serializer());
                if( ! s.get() ) return false;
                return s.get()->serialize( src, dest );
        }


        bool save( const node_type & src, const std::string & filename )
        {
                typedef std::auto_ptr<serializer_base_type> AP;
                AP s = AP(create_serializer());
                if( ! s.get() ) return false;
                return s.get()->serialize( src, filename );
        }

        node_type * load_node( const std::string & src )
        {
                return s11n::io::load_node<node_type>( src );
        }

        node_type * load_node( std::istream & src )
        {
                return s11n::io::load_node<node_type>( src );
        }


} // namespace

#undef S11NLITE_CONFIG_FILE
