#ifndef S11N_LITE_H_INCLUDED
#define S11N_LITE_H_INCLUDED 1

////////////////////////////////////////////////////////////////////////
// headers which want to check for s11nlite's inclusion should check
// for:
#define s11n_S11NLITE_INCLUDED 1
// That's "the official way" to know if s11nlite is avaliable.
////////////////////////////////////////////////////////////////////////

/**
   s11nlite is a simplified subset of the s11n interface. It refines
   the s11n interface to it's bare minimum (or pretty close), while
   still leaving the full power of the underlying framework at the
   clients' disposals if they need it. It also provides a binding to
   the default i/o implementation (the s11n::io-related Serializers),
   meaning that clients don't have to deal with file formats at all,
   and almost never even need to deal with the Serializer interface.

   It is completely compatible with the core s11n framework, but
   directly provides only a subset of the interface: those operations
   common to most client-side use-cases, and those functions where
   s11nlite can eliminate one of the required template parameters
   (s11n's conventional NodeType argument, which s11nlite hard-codes
   in it's note_type typedef).

    Many of the functors and proxies in the s11n namespace are directly
    usable with s11nlite without having to fight with template parameters.

   This implementation can easily be used as a basis for writing
   custom client-side, s11n-based serialization frameworks.

   Suggestions for including specific features into this interface are
   of course welcomed.

   Common conventions concerning this API:

   - node_type must conform to s11n::data_node conventions.

   - node_traits is equivalent to s11n::node_traits<node_type>.

   - Serializers usable by this framework must be classloadable via
     the serializer_base_type classloader (including all that this
     implies).


     License: Do As You Damned Well Please

     Author: stephan@s11n.net
*/
namespace s11nlite { /** here to please doxygen :( */ }

#include <memory> // auto_ptr
#include <iterator> // insert_interator<>
#include <functional> // for_each

#include <s11n.net/s11n/s11n_config.hpp>

#include <s11n.net/s11n/s11n.hpp> // the whole s11n framework
#include <s11n.net/s11n/io/data_node_io.hpp> // s11n::io::data_node_serializer class
#include <s11n.net/zfstream/zfstream.hpp> // get_[o,i]stream()


////////////////////////////////////////////////////////////////////////
// Maintenance notes: the following macros are defined in
// s11n_config.hpp:
//
////////////////////////////////////////////////////////////////////////////////

namespace s11nlite {



        /**
           The sharing namespace contains "sharing context" classes.
        */
        namespace sharing {
                /**
                   Internal marker class.
                 */
                struct sharing_context;
        }

        /**
           node_type is the type used to store/load a Serializable
           object's data.

	   FYI: node_type might change to s11n::s11n_node in 1.1,
	   because that implementation is a tiny bit more lightweight
	   than s11n::data_node, and was designed specifically with
	   node_traits in mind (data_node came much earlier than
	   either of them).  If you only use node_traits to access
	   nodes' data then your code will be oblivious to this change
	   (but will need a recompile).
        */
        typedef s11n::data_node node_type;

        /** Deprecated (it's too long!): use node_traits instead. */
        typedef s11n::node_traits<node_type> node_traits_type;

        /** The s11n::node_traits type for node_type. */
        typedef s11n::node_traits<node_type> node_traits;


        /**
           This is the base-most type of the serializers used by s11nlite.
        */
        typedef s11n::io::data_node_serializer<node_type> serializer_base_type;


        /**
           Returns a reference to s11nlite's configuration object. It
           is read from $HOME/.s11nlite at library startup and saved
           at library shutdown. s11n clients are encouraged to use their
           own configuration storage, but this one "may" be used by
           s11n-related clients (like s11nbrowser).
        */
        node_type & config();


        /**
           Returns a new instance of the default serializer class.

           The caller owns the returned pointer.

         */
        serializer_base_type * create_serializer();

        /**
           Returns a new instance of the given serializer class, or 0
           if one could not be loaded. classname must represent a subtype
           of serializer_base_type.

           The caller owns the returned pointer.

           You can also pass a serializer's cookie here, and that
           should return the same thing as it's class name would.

           The internally-supported serializes all support a "friendly
           form" of the name, an alias registered with their
           classloader. Passing either this name or the cookie of the
           Serializer should return the same thing as the classname
           itself would.

           Short-form names of internally-supported Serializers:

           - compact
           - expat (IF s11n was built with expat support)
           - funtxt, text, txt
           - funxml, xml
           - parens
           - simplexml
           - wesnoth (added in 0.9.14)

           Note that the short-form name is always the same as that of
           the underlying Serializer class, minus the _serializer
           suffix. This is purely a convention, and not a rule.

         */
        serializer_base_type *
        create_serializer( const std::string & classname );


        /**
           Sets the current Serializer class used by s11nlite's
           create_serializer(). Pass it a class name, or one of
           the convenience names:

           compact, funtxt, funxml, simplexml, parens
        */
        void serializer_class( const std::string & );

        /**
           Gets the name of the current Serializer type.
        */
        std::string serializer_class();


        /**
           A non-const overload. Ownership is not modified by calling
           this function: normally the parent node owns it.
        */
        node_type *
        find_child( node_type & parent,
                    const std::string subnodename );

        /**
           Equivalent to s11n::find_child_by_name( parent, subnodename ).
        */
        const node_type *
        find_child( const node_type & parent,
                    const std::string subnodename );

        /**
           See s11n::serialize().
        */
        template <typename SerializableType>
        bool serialize( node_type & dest,
                        const SerializableType & src )
        {
                return s11n::serialize<node_type,SerializableType>( dest, src );
        }

        /**
           See s11n::serialize().
        */
        template <typename SerializableType>
        bool serialize_subnode( node_type & dest,
                                const std::string & subnodename,
                                const SerializableType & src )
        {
                node_type * n = new node_type;
                n->name( subnodename );
                if( serialize<SerializableType>( *n, src ) )
                {
                        dest.children().push_back( n );
                }
                else
                {
                        delete( n );
                        n = 0;
                }
                return n != 0;
        }

       
        /**
        Saves the given node to the given ostream using the default
        serializer type.

        Returns true on success, false on error.

        ONLY use this for saving root nodes!
        */
        bool save( const node_type & src, std::ostream & dest );

        /**
        Saves the given node to the given filename using the default
        serializer type.

        Returns true on success, false on error.

        ONLY use this for saving root nodes!
        */
        bool save( const node_type & src, const std::string & filename );

        /**
        Saves the given Serializable to the given ostream using the default
        serializer type.

        Returns true on success, false on error.

        ONLY use this for saving root nodes!
        */
        template <typename SerializableType>
        bool save( const SerializableType & src, std::ostream & dest )
        {
                node_type n;
                if( ! serialize( n, src ) ) return false;
                return save( n, dest );
        }
        /**
        Saves the given Serializable to the given filename using the default
        serializer type.
        
        Returns true on success, false on error.

        ONLY use this for saving root nodes!
        */
        template <typename SerializableType>
        bool save( const SerializableType & src, const std::string & dest )
        {
                typedef std::auto_ptr<std::ostream> AP;
                AP os = AP( zfstream::get_ostream( dest ) );
                if( ! os.get() ) return 0;
                return save( src, *os );
        }

        /**
           Tries to load a node from the given filename.

           The caller owns the returned pointer.
         */        
        node_type * load_node( const std::string & src );

        /**
           Tries to load a node from the given input stream.

           The caller owns the returned pointer.

           Only usable for loading ROOT nodes.
         */        
        node_type * load_node( std::istream & src );





        /**
           deserializer is a functor for deserializing
           SerializableType objects from node_type objects.

           Sometimes this may be more convenient to use than
           deserialize(), e.g., when deserializing multiple
           objects of the same type.

           Sample usage:
<pre>
typedef deserializer<MyBaseType> Deser;
Deser d;
MyBaseType * obj = d( my_data_node );
</pre>

        */
        template <typename SerializableType>
        struct node_deserializer
        {
                typedef SerializableType serializable_type;
                serializable_type * operator()( const node_type & src )
                {
                        return s11n::deserialize<node_type,serializable_type>( src );
                }
        };

        /**
           A free-function equivalent to deserializer<>.

           It is implemented in terms of deserializer<>,
           so that specializations of that class will
           be picked up by calls to this function, allowing
           clients to selectively replace certain ones.

           ACHTUNG: if you are using both s11n and s11nlite
           namespaces this function will be ambiguous with one provided
           in the namespace s11n. You must then qualify it
           with the namespace of the one you wish to use.
        */
        template <typename SerializableType>
        SerializableType * deserialize( const node_type & src )
        {
                return s11n::deserialize<node_type,SerializableType>( src );
        }



        /**
           DEPRECATED, and will be removed.

           Please use deserialize_subnode(), which does the same thing.

           Tries to find a subnode of src named subnodename. If it finds
           it, it tries to deserialize<SerializableT>( thenode
           ). Returns the result of deserialize( thenode ), or 0 if no
           such child could be found.
        */
        template <typename SerializableType>
        SerializableType * deserialize( const node_type & src, const std::string & subnodename )
        {
                const node_type * ch = s11n::find_child_by_name( src, subnodename );
                if( ! ch ) return 0;
                return deserialize<SerializableType>( *ch );
        }

        /**
           Tries to deserialize src into target. Returns true on
           success. If false is returned then target is not guaranteed
           to be in a useful state: this depends entirely on the
           object (but, it could be argued, it it was in a useful
           state it's deserialize operator would have returned true!).
        */
        template <typename DeserializableT>
        bool deserialize( const node_type & src, DeserializableT & target )
        {
                return s11n::deserialize<node_type,DeserializableT>( src, target );
        }


        /**
           Exactly like deserialize(), but operates on a subnode of
           src named subnodename. Returns false if no such file is
           found.
         */
        template <typename DeserializableT>
        bool deserialize_subnode( const node_type & src,
                                  const std::string & subnodename,
                                  DeserializableT & target )
        {
                return s11n::deserialize_subnode<
                        node_type,
                        DeserializableT>( src, subnodename, target );
        }

        /**
           Exactly like deserialize(), but operates on a subnode of
           src named subnodename. Returns 0 if no such file is
           found.
         */
        template <typename DeserializableT>
        DeserializableT * deserialize_subnode( const node_type & src,
                                               const std::string & subnodename )
        {
                return s11n::deserialize_subnode<
                        node_type,
                        DeserializableT>( src, subnodename );
        }


        /**
           Tries to load a data_node from src, then deserialize that
           to a SerializableType.
        */
        template <typename SerializableType>
        SerializableType * load_serializable( std::istream & src )
        {
                typedef std::auto_ptr<node_type> AP;
                AP n = AP( load_node( src ) );
                if( ! n.get() ) return 0;
                return deserialize<SerializableType>( *n );
        }

        /**
           Overloaded form which takes a file name.

           See get_istream() for a description of the AsFile
           parameter.
        */
        template <typename SerializableType>
        SerializableType * load_serializable( const std::string & src, bool AsFile = true )
        {
                typedef std::auto_ptr<std::istream> AP;
                AP is = AP( zfstream::get_istream( src, AsFile ) );
                if( ! is.get() ) return 0;
                return load_serializable<SerializableType>( *is );
        }

        /**
           See s11n::clone().
        */
        template <typename SerializableType>
        SerializableType * clone( const SerializableType & tocp )
        {
                node_type node;
                if( ! serialize( node, tocp ) ) return 0;
                return deserialize<SerializableType>( node );
        }

        /**
           See s11n::s11n_cast().
        */
        template <typename Type1, typename Type2>
        bool s11n_cast( const Type1 & t1, Type2 & t2 )
        {
                node_type n;
                return serialize( n, t1 ) && deserialize( n, t2 );
        }



} // namespace s11nlite

#endif // S11N_LITE_H_INCLUDED
