#ifndef s11n_cl_S11N_CLASSLOAD_HPP_INCLUDED
#define s11n_cl_S11N_CLASSLOAD_HPP_INCLUDED 1

#include <s11n.net/cl/cllite.hpp> // default classloader/factory implementation.


namespace s11n {

        /**
           The s11n::cl namespace encapsulates the classloader-related
           API for libs11n.

           Note that the registration functions in this namespace
           register with the cllite framework. Clients wishing to use
           their own factories must:

           - register with their appropriate classloader.

           - specialize the object_factory class template so that it
           forwards calls to their classloader.


           This layer is used for classloading anything which s11n
           needs to dynamically load, including:

           - Serializables

           - Serializers

           - FlexLexers

           It supports loading types via DLLs if the types meet
           cllite's default DLL requirements.
        */
        namespace cl {

                /**
                   A default object factory implementation for use
                   with the s11n::cl::classload()-related functions.

                   Clients may specialize this to return objects from
                   their own factories. By default it uses the cllite
                   framework and thus can load any type registered with
                   cllite's BaseType classloader.
                */
                template <typename BaseType>
                struct object_factory
                {
                        /** Same as BaseType. */
                        typedef BaseType base_type;

                        /**
                           Takes a class name as an argument. Returns
                           a pointer to a new BaseType, or 0 on
                           error. The caller owns the returned
                           pointer.
                        */
                        base_type * operator()( const std::string & classname ) const
                        {
                                try
                                {
                                        return ::cllite::classload<BaseType>( classname );
                                }
                                catch(...)
                                {
                                        return 0;
                                }
                        }
                };


                /**
                   Returns BaseT's classloader.

                   See cllite::classloader() for important info.

                   Before mucking with this object, please be aware of
                   the cllite relationship between it and the default
                   object_factory<BaseT> implementation (e.g., by looking
                   at the s11n::cl source code).

                   Using this classloader for types which have
                   customized, non-cllite-based object_factory<BaseT>
                   specializations is essentially useless.
                */
                template <typename BaseT>
                ::cl::class_loader<BaseT> &
                classloader()
                {
                        return ::cllite::classloader<BaseT>();
                }

                /**
                   Returns object_factory<BaseType>()( key ).


                   Note that calling this function is NOT the same as
                   calling classloader().load(key). In many cases they
                   will provide the same result, but the latter does
                   not provide DLL support.
                */
                template <typename BaseType>
                BaseType * classload( const std::string key )
                {
                        return object_factory<BaseType>()( key );
                }

                /**
                   Registers SubType with BaseType's classloader, using
                   a default object factory.

                   SubType must be a subtype of BaseType, or may be
                   the same as BaseType. If BaseType is abstract it
                   must first be registered as an abstract base, e.g.,
                   using classloader_register_abstract().  In that
                   case BaseType should not be registered with this
                   function.

                */
                template <typename BaseType, typename SubType>
                void classloader_register( const std::string & classname )
                {
                        classloader<BaseType>().register_factory(
                                                                 classname,
                                                                  ::cl::object_factory<BaseType,SubType>::new_instance
                                                                 );
                }

                /**
                   Registers BaseType with it's own classloader.
                */
                template <typename BaseType>
                void classloader_register_base( const std::string & classname )
                {
                        classloader_register<BaseType,BaseType>( classname );
                }

                /**
                   Registers BaseType as an abstract type. That is, it
                   assigns it a factory which always returns 0.

                   If types you register with classloader_register()
                   (and friends) have an abstract BaseType then that
                   BaseType should be registered via this function (or
                   equivalent).
                */
                template <typename BaseType>
                void classloader_register_abstract( const std::string & classname )
                {
                        classloader<BaseType>().register_factory( classname,
                                                                  ::cl::object_factory<BaseType,BaseType>::no_instance
                                                                  );
                }



        } // namespace cl
} // namespace s11n


#endif // s11n_cl_S11N_CLASSLOAD_HPP_INCLUDED
