
#include <algorithm> // for_each
#include <iterator>

#include <s11n.net/acme/functor.hpp> // object_deleter, etc.
#include <s11n.net/tostring/to_string.hpp>
#include <s11n.net/phoenix/phoenix.hpp>

#include "data_node.hpp"
#include "data_node_functor.hpp" // functors to serialize data_nodes.


namespace s11n {
#define NODE_IMPL_CLASS "s11n::data_node"
// note that the default implclass is not useable for purposes of classloading
// serializables, but:
//  a) the parsers need some text to read.
//  b) "raw" nodes are often useful for de/ser'ing maps, lists, etc.
//  c) Serializables must set their implclass *anyway*, so they will
//     (or should) always overwrite the default class.


	data_node::data_node( const std::string & name ) : m_name(name),m_iname(NODE_IMPL_CLASS)
	{
	}
	data_node::data_node( const std::string & name, const std::string implclass ) : m_name( name ), m_iname( implclass )
	{
	}

	data_node::data_node() : m_name("data_node"), m_iname(NODE_IMPL_CLASS)
	{
	}

        data_node & data_node::operator=( const data_node & rhs )
        {
                if( &rhs == this ) return *this;
                this->copy( rhs );
                return *this;
        }
        data_node::data_node( const data_node & rhs )
        {
                if( &rhs == this ) return;
                this->copy( rhs );
        }
	data_node::~data_node()
	{
                //CERR << "~data_node() " << std::hex << this << "\n";
                this->clear_children();
	}

        void data_node::copy( const data_node & rhs )
        {
		if ( &rhs == this ) return;

                this->clear();

                this->name( rhs.name() );
                this->impl_class( rhs.impl_class() );

                std::copy( rhs.begin(), rhs.end(),
                           std::insert_iterator<map_type>( this->m_map, this->m_map.begin() )
                           );

                std::for_each( rhs.children().begin(),
                               rhs.children().end(),
                               acme::child_pointer_deep_copier<child_list_type>( this->children() )
                               );
        }

//         const data_node *
//         data_node::child( const std::string & nodename ) const
//         {
//                 data_node *ch = 0;
//                 typedef data_node::child_list_type::const_iterator CIT;
//                 CIT cit = this->children().begin();
//                 CIT cet = this->children().end();
//                 for ( ; cit != cet; ++cit )
//                 {
//                         if ( ( *cit )->name() == nodename )
//                         {
//                                 ch = ( *cit );
//                                 break;
//                         }
//                 }
//                 return ch;
//         }
        
        void
        data_node::set_bool( const std::string & key, bool val )
        {
                this->set_string( key, tostring::to_string<bool>(val) );
        }




	data_node::child_list_type & data_node::children()
	{
                return this->m_children;
	}

	const data_node::child_list_type & data_node::children() const
	{
                return this->m_children;
	}

	void data_node::clear()
	{
                this->clear_children();
		this->clear_properties();
	}

	void data_node::impl_class( const std::string & n )
	{
		this->m_iname = n;
	}

	std::string
        data_node::impl_class()const
	{
		return this->m_iname;
	}


	void
        data_node::name( const std::string & n )
	{
		this->m_name = n;
	}

	std::string
        data_node::name() const
	{
		return this->m_name;
	}



	using namespace std;


	const std::string data_node::operator[] ( const string & key ) const
	{
		return this->get_string( key );
	}

	std::string & data_node::operator[] ( const string & key )
	{
		return this->m_map[key];
	}

        void
        data_node::insert( const value_type & v )
        {
                this->set_string( v.first, v.second );
        }

	std::string data_node::get_string( const std::string & key, const std::string & defaultVal ) const
	{
		data_node::map_type::const_iterator citer = m_map.find( key );
                if( m_map.end() == citer ) return defaultVal;
                return ( *citer ).second;
	}

	void data_node::set_string( const std::string & key, const std::string & val )
	{
		m_map[key] = val;
		return;
	}

	bool data_node::is_set( const std::string & mkey ) const
	{
		std::string key = mkey;	// experiment to try to work around a segfault.
		if ( key.empty() )
			return false;
		data_node::map_type::const_iterator iter;
		iter = this->m_map.find( key );
		return ( iter != m_map.end() ) ? true : false;
	}

	void data_node::unset( const std::string & key )
	{
		data_node::map_type::iterator iter;
		iter = m_map.find( key );
		if ( iter == m_map.end() ) return;
		m_map.erase( iter );
		return;
	}

	void data_node::clear_properties()
	{
		if ( m_map.empty() ) return;
		m_map.erase( m_map.begin(), m_map.end() );
	}

	void data_node::clear_children()
	{
                std::for_each( this->children().begin(),
                               this->children().end(),
                               acme::object_deleter() );
                this->children().clear();
        }


	data_node::iterator data_node::begin()
	{
		return this->m_map.begin();
	}

	data_node::const_iterator data_node::begin()const
	{
		return this->m_map.begin();
	}

	data_node::iterator data_node::end()
	{
		return this->m_map.end();
	}

	data_node::const_iterator data_node::end()const
	{
		return this->m_map.end();
	}

	data_node::iterator data_node::find( const std::string & key )
	{
		return m_map.find( key );
	}

	data_node::map_type & data_node::map()
	{
		return m_map;
	}
	const data_node::map_type & data_node::map() const
	{
		return m_map;
	}



	bool data_node::get_bool( const std::string & key, bool defaultVal ) const
	{
		if ( !this->is_set( key ) )
			return defaultVal;
		return data_node::bool_val( this->get_string( key, "???" ) );
	}

        /** phoenix<> initializer functor. */
        struct data_node_truesmap_initializer
        {
                typedef std::map<std::string,bool> trues_map;
                void operator()( trues_map & map )
                {
			map["true"] = true;
			map["TRUE"] = true;
			map["True"] = true;
			map["yes"] = true;
			map["YES"] = true;
			map["Yes"] = true;
			map["y"] = true;
			map["Y"] = true;
			map["1"] = true;
                }
        };

	bool // static 
        data_node::bool_val( const std::string & key )
	{
		typedef std::map < std::string, bool > TrueMap;
                typedef phoenix::phoenix<TrueMap,data_node,data_node_truesmap_initializer> PHX;
                TrueMap & trues = PHX::instance();
		return trues.end() != trues.find( key );
	}

        size_t
        data_node::size() const
        {
                return this->m_map.size();
        }




// frivilous:
//         bool data_node::delete_child( const std::string & nodename )
//         {
//                 data_node * ch = this->take_child( nodename );
//                 if( ! ch ) return false;
//                 delete( ch );
//                 return true;
//         }
//         bool data_node::delete_child( data_node * sub )
//         {
//                 data_node * ch = this->take_child( sub );
//                 if( ! ch ) return false;
//                 delete( ch );
//                 return true;
//         }


//         bool
//         data_node::empty() const
//         {
//                 return (0 == this->m_map().size())
//                         && 
//                         (this->children().empty());
//         }

                


} // namespace s11n
