#ifndef s11n_DATA_ALGO_H_INCLUDED
#define s11n_DATA_ALGO_H_INCLUDED
////////////////////////////////////////////////////////////////////////
// data_node_algo.hpp:
//
// Defines some helper algorithms for working with Serializables and
// Data Nodes.
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#include <string>
#include <list>
#include <iterator>
#include <algorithm> // for_each
#include <cassert>

#include "s11n_debuggering_macros.hpp" // COUT/CERR
// #include <s11n.net/tostring/to_string.hpp> // to/from_string()
#include <s11n.net/acme/functor.hpp> // same_name, et al

#include "abstract_creator.hpp" // abstract_creator class


////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.hpp ALLOWED!
////////////////////////////////////////////////////////////////////////

namespace s11n {



        /**
           Adds ch as a child of parent. Parent takes over ownership
           of ch.

           NodeType must have a node_traits<> specialization.
        */
        template <typename NodeType, typename ChildType>
        void add_child( NodeType & parent, ChildType * ch )
        {
                typedef ::s11n::node_traits<NodeType> NTR;
                NTR::children( parent ).push_back( ch );
        }

 
 

        /**
           Creates a new node, named nodename, as a child of parent.

           Returns a reference to the new child, which parent now
           owns.

           NodeType must have a node_traits<> specialization or work
           using the default
        */
        template <typename NodeType>
        NodeType & create_child( NodeType & parent,
                                  const std::string nodename )
        {
                typedef ::s11n::node_traits<NodeType> NTR;
                NodeType * n = NTR::create( nodename );
                NTR::children( parent ).push_back( n );
                return *n;
        }



        /**
           Each child in parent.children() which has the given name is
           copied into the target container.

           Returns the number of items added to target.

           DestContainerT must support an insert iterator which will
           insert the pointer type contained in the list returned by
           parent.children(). i.e., it must hold (const
           SomeSerializableType *).

           Ownership of the child does not change by calling this
           function. Normally they are owned by the parent node
           (unless the client explicitely does something to change
           that).
        */
        template <typename NodeT, typename DestContainerT>
        size_t find_children_by_name( const NodeT & parent,
                               const std::string & name,
                               DestContainerT & target )
        {
		typedef node_traits<NodeT> TR;
                size_t c = target.size();
                acme::copy_if( TR::children(parent).begin(),
			       TR::children(parent).end(),
			       std::insert_iterator<DestContainerT>( target,
								     target.begin() ),
			       acme::same_name<NodeT>( name )
			       // ^^^^ BUG: this only works when NodeT::name() is public,
			       // as it avoids node_traits<>!
			       // In 1.1 s11n_node::name() might be made private, and node_traits<>
			       // will be his only friend. Then this bug will show up.
			       // s11n::data_node is frozen: he won't change, but s11nlite
			       // might switch to using s11n_node in 1.1!
                                         );
                // you gotta love the STL, man.
                return target.size() - c;
        }

        /**
           Finds the FIRST child in parent with the given name and
           returns a pointer to it, or 0 if no such child is found.

           Ownership of the children does not change by calling this
           function: parent still owns it.
        */
        template <typename NodeT>
        const NodeT *
        find_child_by_name( const NodeT & parent, const std::string & name )
        {
		typedef node_traits<NodeT> TR;
                typedef typename NodeT::child_list_type::const_iterator CIT;
                CIT it = std::find_if( TR::children(parent).begin(),
                                       TR::children(parent).end(),
                                       acme::same_name<NodeT>( name )
                                       );
                return (TR::children(parent).end() == it) ? 0 : *it;
        }

        /**
           A non-const overload of find_child_by_name(). Functionally
           identical to the const form, except for the constness of
           the parent argument and return value.

           Ownership of the returned pointer is not changed by calling
           this function (normally parent owns it, but clients may
           change that without affecting this function). When in
           doubt, i.e. during "normal usage", do NOT delete the returned
           pointer, because the parent node owns it.
        */
        template <typename NodeT>
        NodeT *
        find_child_by_name( NodeT & parent, const std::string & name )
        {
		typedef node_traits<NodeT> TR;
                typedef typename NodeT::child_list_type::iterator IT;
                IT it = std::find_if( TR::children(parent).begin(),
				      TR::children(parent).end(),
                                       acme::same_name<NodeT>( name )
                                       );
                return (TR::children(parent).end() == it) ? 0 : *it;
        }


} // namespace s11n

#endif // s11n_DATA_ALGO_H_INCLUDED
