#ifndef s11n_DATA_NODE_FUNCTOR_H_INCLUDED
#define s11n_DATA_NODE_FUNCTOR_H_INCLUDED

////////////////////////////////////////////////////////////////////////
// data_node_functor.hpp
// Some functors/algorithms for working with data_node-compliant types.
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#include <string>
#include <list>
#include <map>
#include <typeinfo> // aid debugging
#include <utility> // std::pair/make_pair
#include <iterator>
#include <iostream>

// #include <s11n.net/phoenix/phoenix.hpp> // phoenix class
#include <s11n.net/tostring/to_string.hpp> // to/from_string()
#include <s11n.net/stringutil/string_util.hpp> // translate_entities()

#include <s11n.net/acme/pointer_stripper.hpp> // pointer_stripper class
#include <s11n.net/name_type/class_name.hpp> // unfortunate dependency.

#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // COUT/CERR
#include <s11n.net/s11n/abstract_creator.hpp> // abstract_creator class
#include <s11n.net/s11n/data_node_serialize.hpp> // core de/serialize interface
#include <s11n.net/s11n/data_node_algo.hpp> // useful data_node algos
#include <s11n.net/s11n/traits.hpp> // node_traits

////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.hpp ALLOWED!
////////////////////////////////////////////////////////////////////////

namespace s11n {

        /**
           Dumps some info about n to the given ostream, mostly
           useful for debugging failed deserialization.
        */
        template <typename NodeType>
        void dump_node_debug( const NodeType & n, std::ostream & os )
        {
                typedef node_traits<NodeType> NTR;
                os << "node dump: ["<<NTR::name(n)<<"]["<<NTR::class_name(n)<<"]@"<<std::hex<<&n<<"\n";
                typedef typename NTR::const_iterator PT;
                PT b = NTR::begin(n), e = NTR::end(n);
                os << "==================== properties:\n";
                for( ; e != b; ++b )
                {
                        os << (*b).first << " = " << (*b).second << "\n";
                }
                os << "[end node]\n";
        }

        /**
           A helper functor to loop over serializable
           children.

           Designed for use with std::for_each().

           NodeType must be compatible with
           s11n::data_node.

           Please see the operator() docs for important usage
           information, especially if you want to use this
           object outside the context of for_each().
        */
        template <typename NodeType>
        struct data_node_child_serializer
        {
                typedef NodeType node_type;
                /**
                   Creates an object for serializing

                   Preconditions:

                   - dest must outlive this object.
                   More correctly, this object's
                   operator() must not be called after
                   dest is destroyed.

                */
                data_node_child_serializer( node_type & dest, const std::string & subname )
                        : result(true), m_name(subname), m_root(&dest)
                {
                }
 
                /**
                   Serializes src into a subnode of dest using the
                   name given in this object's ctor.

                   Note that during an, e.g., for_each() this object
                   will return false on a failed serialize, and will
                   CONTINUE to return false on additional serializations.
                   This is to avoid the possibility that for_each()
                   fails on the first item of a list, handles 3000 items,
                   and then the whole thing fails because of the first one.
                   Thus, this operator will never process another request
                   once it has returned false ONCE.
                */
                template <typename SerializableT>
                bool operator()( const SerializableT * src )
                {
                        if( ! this->result ) return false;
                        return this->result = 
                                serialize_subnode( *this->m_root,
                                                   this->m_name,
                                                   *src );
                }
                template <typename SerializableT>
                bool operator()( const SerializableT & src )
                {
                        return this->operator()( &src );
                }

                /**
                   For use as a "return value catcher" for std::for_each().
                   See operator() for how it is set. The starting value
                   is true, which means that looping over an empty list
                   with this object will return a true result (which is
                   the convention in s11n).
                */
                bool result;
        private:
                std::string m_name;
                node_type * m_root;
        };

        /**
           A helper functor deserialize a set of data_nodes.

           Designed for use with std::for_each().

           ListType must be compatible with
           <code>std::list<SomeSerializableType *></code>.
        */
        template <typename ListType>
        struct data_node_child_deserializer
        {
                /**
                   The ListType templatized type. Must support:

                   push_back(value_type *);
                 */
                typedef ListType list_type;
                /**
                   value_type is equivalent to a SerializableType.
                */
                typedef typename acme::pointer_stripper<
                        	typename list_type::value_type
 			>::value_type  value_type;

                /**
                   Creates an object for deserializing Serializables
                   into a target list.

                   If tolerant is true then this object will always
                   try to deserialize new items passed to it's
                   operator(). If it is false (the default) it will
                   stop processing after a single failure.

                   Preconditions:

                   - dest must outlive this object. More correctly,
                   this object's operator() must not be called after
                   either dest is destroyed.

                */
                data_node_child_deserializer( list_type & dest,
                                              bool tolerant = false )
                        : result(true), m_tolerant(tolerant), m_list(&dest)
                {
                }

                /**
                   Tries to deserialize a (value_type *) from src. On error it returns 0,
                   else it inserts the new child into this object's destination container
                   and returns true.

                   Postconditions:

                   - If successful, this object's destination container owns the returned
                   pointers.
                */
                template <typename NodeType>
                bool operator()( const NodeType * src )
                {
                        if( ! result && ! this->m_tolerant ) return false; // once an item fails, stop processing them.
                        if( ! src ) return result = false;
                        value_type * ch = 0;
                        ch = s11n::template deserialize<NodeType,value_type>( *src );
                        //CERR << "data_node_child_deserializer<>("<<src->name()<<") = "
                        //     << std::hex << ch <<"\n";
                        if( ! ch ) return result = false;
                        this->m_list->push_back( ch );
                        return result = true;
                }

                template <typename NodeType>
                bool operator()( const NodeType & src )
                {
                        return this->operator()( &src );
                }

                /**
                   For use as a "return value catcher" for std::for_each().

                   If tolerant is true then this will only reveal the value of
                   the most-recently-processed node.
                */
                bool result;
        private:
                bool m_tolerant;
                list_type * m_list;
        };




        /**
           A Serializable Proxy for streamable types. It "should" work
           with any type which meets these conditions:

           - i/ostream operators are implemented (as member or free
           functions).

           - supports a copy ctor (for deserialization).

           - has a class_name&lt;&gt; specialization. Such
           specializations come pre-installed for PODs, std::string,
           and maybe others. Use the CLASS_NAME(T) macro to create these.


           It's output is significantly bigger than using, e.g., node
           properties to store them, but with this proxy any
           streamable can be treated as a full-fledged Serializable,
           which allows some generic container-based serialization to
           be done regardless of the underlying types (see
           list_serializable_proxy for an example).


         ACHTUNG: never pass the same Serializable to the
         operators more than once or you will get duplicate and/or
         incorrect data.
        */
        struct streamable_type_serialization_proxy
        {
                /**
                   Sets the subnode name used for this object's
                   serialize operator.
                */
                explicit streamable_type_serialization_proxy( const std::string & subnodename )
                        : m_subname(subnodename)
                {}
                /**
                   Uses a default, unspecified, name for subnode
                   insertions.
                */
                streamable_type_serialization_proxy() : m_subname("")
                {}

                /**
                   Creates a property in dest, called 'v', and sets
                   its value to src. Each node gets the name assigned
                   via this object's ctor or an undefined dummy name.
                   Note that the deserialize operator pays no attention
                   to the subnode names.

                   Always returns true.
                */
                template <typename NodeType, typename SerType>
                bool operator()( NodeType & dest, const SerType & src ) const
                {
                        typedef node_traits<NodeType> NTR;
                        NTR::class_name( dest, ::classname<SerType>() );
                        if( ! this->m_subname.empty() ) NTR::name( dest, this->m_subname );
                        NTR::set( dest, "v", src );
                        return true;
                }

                /**
                   Looks for a property in src called 'v' and sets
                   dest to it's value. The default for dest, in the case
                   of a missing property or nonconvertable value is
                   dest itself.

                   If the 'v' property is missing this function does
                   nothing and returns false, otherwise dest gets
                   assigned the property's value and true is
                   returned. This function cannot catch a case of
                   inability to convert 'v' into a SerType: client
                   code interested in doing so should compare dest's
                   value to a known error value after this
                   function returns.
                */
                template <typename NodeType, typename SerType>
                bool operator()( const NodeType & src, SerType & dest ) const
                {
                        typedef node_traits<NodeType> NTR;
                        if( ! NTR::is_set( src, "v" ) )
                        {
                                CERR << "streamable_serializable_proxy: deser failed: property 'v' missing!\n";
                                dump_node_debug( src, std::cerr );
                                return false;
                        }
                        dest = NTR::get( src, "v", dest );
                        return true;
                }
        private:
                std::string m_subname;
        };





        namespace io {



                /**
                   A helper for serializing properties to a
                   stream. Intended for use by Serializers, not
                   Serializables.

                   NodeType is the container type used for data
                   serialization (e.g. s11n::data_node).

                */
                template <typename NodeType>
                class key_value_serializer
                {
                public:
                        typedef NodeType node_type;
                        typedef typename node_type::value_type pair_type;

                        typedef std::map<std::string,std::string> entity_translation_map;

                        /**
                           map: needed so we can do entity translation
                           in a unified manner here. It must outlive
                           this object. Pass it 0 for no translation.
                           Translations are only applied on VALUES,
                           not KEYS.

                           prefix: inserted before each property.

                           separator: inserted between the key and value.

                           suffix: appended after each entry.


                        */
                        key_value_serializer( const entity_translation_map * map,
                                              std::ostream & dest,
                                              const std::string & prefix,
                                              const std::string & separator,
                                              const std::string & suffix
                                              )
                                : m_pre(prefix), m_sep(separator), m_suf(suffix), m_os(dest), m_map(map)
                        {
                        }

                        /**
                           Sends the following formatted string to os:

                           {prefix}{src.first}{separator}{src.second}{suffix}

                        */
                        void operator()( const pair_type & src ) const
                        {
                                static const std::string errval = "";
                                std::string key = tostring::to_string( src.first );
                                std::string val = tostring::to_string( src.second );
                                // should we xlate the key as well?
                                if( this->m_map )
                                {
                                       stringutil::translate_entities( val, *(this->m_map) );
                                }
                                this->m_os << this->m_pre;
                                this->m_os << key;
                                this->m_os << this->m_sep;
                                this->m_os << val;
                                this->m_os << this->m_suf;
                        }
                private:
                        std::string m_pre;
                        std::string m_sep;
                        std::string m_suf;
                        std::ostream & m_os;
                        const entity_translation_map * m_map;
                };





                /**
                   A helper functor to loop over serializable children
                   of a node from within a Serializer implementation.

                   Designed for use with std::for_each().

                   SerializerT must be compatible with
                   <code>data_node_serializer<></code>.

                */
                template <typename SerializerT>
                struct node_child_simple_formatter
                {
                        typedef SerializerT serializer_type;
//                         typedef typename SerializerT::node_type node_type;
                        /**
                           Preconditions:

                           - Ser must be valid references.

                           - Both ser and os must outlive this
                           object. More correctly, this object's
                           operator() must not be called after either
                           ser or os are destroyed.

                        */
                        node_child_simple_formatter( serializer_type & ser, std::ostream & os,
                                               const std::string & prefix = "", const std::string & suffix = "\n" )
                                : m_ser(ser), m_os(&os), m_pre(prefix), m_suf(suffix)
                        {
                        }

                        /**
                           Serializes src into this object's target
                           container, using this object's serializer.
                        */
                        template <typename NodeType>
                        bool operator()( const NodeType * src ) const
                        {
                                if( ! src ) return false;
                                if( ! this->m_pre.empty() ) *(this->m_os) << this->m_pre;
                                bool b = this->m_ser.serialize( *src, *(this->m_os) );
                                if( ! this->m_suf.empty() ) *(this->m_os) << this->m_suf;
                                return b;
                        }

                private:
                        serializer_type & m_ser;
                        std::ostream * m_os;
                        std::string m_pre;
                        std::string m_suf;
                };


        } // namespace io

} // namespace s11n


#endif // s11n_DATA_NODE_FUNCTOR_H_INCLUDED
