#ifndef s11n_DATA_SERIALIZE_H_INCLUDED
#define s11n_DATA_SERIALIZE_H_INCLUDED
////////////////////////////////////////////////////////////////////////
// data_node_serialize.hpp:
//
// Defines the core de/serialize() functions (and close friends).
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////

#include <string>
#include <list>
#include <map>
#include <deque>

#include <cassert>
#include <typeinfo>

#include <s11n.net/s11n/s11n_debuggering_macros.hpp> // COUT/CERR
#include <s11n.net/name_type/class_name.hpp> // ::classname<>()
#include <s11n.net/s11n/classload.hpp> // classload()
#include <s11n.net/s11n/traits.hpp> // s11n_traits


////////////////////////////////////////////////////////////////////////////////
// NO DEPS ON data_node.hpp/s11n_node.hpp ALLOWED!
////////////////////////////////////////////////////////////////////////////////

#define NODEERR if(0) CERR
// ^^^ a debugging macro

namespace { // anon namespace important for linking reasons, to avoid ODR violations.

        /***
            s11n_api_marshaler is an internal API marshaller for s11n.
	    See the lib manual for full details.

            For the default implementation s11n_traits<SerializableT>
            is used to marshal the de/serialize() calls.
            is used to marshal the de/serialize() calls.
        */
        template <typename SerializableT>
        struct s11n_api_marshaler
        {
                /**
                   Same as SerializableT.
                */
                typedef SerializableT serializable_type;

                /**
                   Returns s11n_traits<serializable_type>::serialize_functor()( dest, src ).

                   It also sets dest's class_name() to ::classname<serializable_type>(),
                   but this is only suitable for monomorphic serializable_types.
                */
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type & src )
                {
                        typedef ::s11n::node_traits<NodeType> NTR;
                        typedef ::s11n::s11n_traits<serializable_type> STR;
                        NTR::class_name( dest, ::classname<serializable_type>() ); // only good for monomorphs
                        return STR::serialize_functor()( dest, src );
                }

                /**
                   Returns s11n_traits<SerializableT>::deserialize_functor()( src, dest ).
                */
                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type & dest )
                {
                        typedef ::s11n::s11n_traits<serializable_type> STR;
                        return STR::deserialize_functor()( src, dest );
                }
        };

        /**
           A specialization to handle pointer types the same as
           reference/value types. It simply translates the pointers
           into references.
        */
        template <typename SerializableT>
        struct s11n_api_marshaler<SerializableT *>
        {
                /**
                   The SerializableT templatized type, minus any
                   pointer part.
                */
                typedef SerializableT serializable_type;

                /**
                   Convenience typedef: this class' parent type.
                 */
                typedef s11n_api_marshaler<serializable_type> parent_type;

                /**
                   Returns parent_type::serialize( dest, *src );

                   src must be a valid pointer.
                */
                template <typename NodeType>
                static bool serialize( NodeType &dest, const serializable_type * src )
                {
                        if( ! src ) return false;
                        return parent_type::serialize( dest, *src );
                }

                /**
                   Returns parent_type::deserialize( src, *dest );

                   dest must be a valid pointer.
                */
                template <typename NodeType>
                static bool deserialize( const NodeType & src, serializable_type * dest )
                {
                        if( ! dest ) return false;
                        return parent_type::deserialize( src, *dest );
                }
        };

} // anon namespace


/**********************************************************************

General Conventions:

	NodeType should conform to the conventions laid out
	by s11n::data_node.


    SerializableTypes/BaseTypes:

	- BaseT must have the following in it's interface:

	- bool SerializeFunction( NodeType & dest ) const;

	- bool DeserializeFunction( const NodeType & dest );

	SerializeFunction/DeserializeFunction need not be virtual,
	though they may be.

    Proxy functors:

	Serialization functor must have:

	bool operator()( NodeType & dest, const BaseT & src ) const;

	Deserialization functor must have:

	bool operator()( const NodeType & src, BaseT & dest ) const;

	They may be the same functor type - const resolution will
	determine which s11n uses. Sometimes this causes ambiguity,
	and may require 2 functors.

	These signatures apply for all functors designed to work as
	de/serializers.



**********************************************************************/


namespace s11n {



        /**
	   DEPRECATED - will go away in s11n 1.1.

           Returns func( target, src ). Note that using this function
           bypasses internal API marshalling for src. This is,
           however, useful in some cases. Note that if src initiates
           recursive serialization on objects of it's own type the
           framework will revert to using it's API marshalling,
           bypassing SerializerFunctorT. A solution is sought for this
           problem, but none exists yet.
        */
        template <typename DataNodeType,typename SerializableT, typename SerializerFunctorT>
        bool serialize( DataNodeType & target, const SerializableT & src, const SerializerFunctorT & func )
        {               
                try
                {
                        return func( target, src );
                }
                catch(...)
                {
                        CERR << "Unknown exception during serialize<>(DataNode,SerializableT,SerializerFunctorT)!\n";
                        return false;
                }
        }

        /**
	   DEPRECATED - will go away in s11n 1.1.

           Returns func( src, target ).  Note that using this function
           bypasses internal API marshalling for the target. This is,
           however, useful in some cases.

           See the complementary form of serialize() for important
           information regarding recursive deserialization.
        */
        template <typename DataNodeType, typename DeserializableT, typename DeserializerFunctorT>
        bool deserialize( const DataNodeType & src, DeserializableT & target, const DeserializerFunctorT & func )
        {
                try
                {
                        return func( src, target );
                }
                catch(...)
                {
                        CERR << "Unknown exception during deserialize<>(DataNode,DeserializableT,DeserializerFunctorT)!\n";
                        return false;
                }
        }

        /**
           Serializes src to target using the default API marshaling
           mechanism.
        */
        template <typename DataNodeType, typename SerializableT>
        bool serialize( DataNodeType & target, const SerializableT & src )
        {
                try
                {
//                         typedef typename s11n_traits<SerializableT>::serialize_functor Proxy;
//                         return serialize<DataNodeType,SerializableT,Proxy>( target, src, Proxy() );
//                         return Proxy()( target, src );
                        return s11n_api_marshaler<SerializableT>::serialize( target, src );
                }
                catch(...)
                {
                        CERR << "Unknown exception during serialize<>(DataNode,SerializableT)!\n";
                        return false;
                }
        }

        /**
           Deserializes target from src using the default API marshaling
           mechanism.
        */
        template <typename DataNodeType, typename DeserializableT>
        bool deserialize( const DataNodeType & src, DeserializableT & target )
        {
                try
                {
//                         typedef typename s11n_traits<DeserializableT>::deserialize_functor Proxy;
//                         return deserialize<DataNodeType,DeserializableT,Proxy>( src, target, Proxy() );
//                        return Proxy()( src, target );
                        return s11n_api_marshaler<DeserializableT>::deserialize( src, target );
                }
                catch(...)
                {
                        CERR << "Unknown exception during deserialize<>(DataNode,DeserializableT)!\n";
                        return false;
                }
        }

        /**
           Tries to deserialize a DeserializableT from src, using
           <code>s11n_traits<DeserializableT>::factory_type()(node_traits<DataNodeType>::class_name(src))</code>
           to create a new DeserializableT. Returns 0 on error,
           otherwise returns a pointer to a new object, which the
           caller takes ownership of.
        */
        template <typename DataNodeType, typename DeserializableT>
        DeserializableT * deserialize( const DataNodeType & src )
        {
                typedef ::s11n::s11n_traits<DeserializableT> STR;
                typedef ::s11n::node_traits<DataNodeType> NTR;
                DeserializableT * obj = STR::factory_type()( NTR::class_name( src ) );
                if( ! obj )
                {
                        NODEERR << "deserialize<>(DataNode): impl class '"
                                << NTR::class_name( src )<<"' classload failed.\n"
                                << "It is probably not registered with it's base classloader.\n";
                        return 0;
                }
                if( ! deserialize<DataNodeType,DeserializableT>( src, *obj ) )
                {
                        NODEERR << "deserialize<>(DataNode): failed for unknown reason.\n";
                        delete( obj );  // must not throw!
                        obj = 0;
                }
                return obj;
        }

        /**
           Serializes src to as a subnode of target, named
           nodename. Except for the addition of a subnode, it is
           identical to serialize( target, src ).

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename SerializableT >
        bool serialize_subnode( DataNodeType & target,
                                const std::string & nodename,
                                const SerializableT & src )
        {
                NODEERR << "serialize_subnode<>(DataNodeType, '"<<nodename<<"', SerializableT)\n";
                DataNodeType * sub = new DataNodeType;
                typedef ::s11n::node_traits<DataNodeType> NT;
                NT::name( *sub, nodename );
                if( ! serialize<DataNodeType,SerializableT>( *sub, src ) )
                {
                        delete( sub );
                        sub = 0;
                }
                else
                {
                        NT::children(target).push_back( sub );
                }
                return sub != 0;
        }


        /**
           If a child named subnodename is found in src then this function
           returns deserialize( child, target ) and returns it's result, otherwise
           it returns 0.

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename DeserializableT>
        bool deserialize_subnode( const DataNodeType & src,
                                  const std::string & subnodename,
                                  DeserializableT & target )
        {
                const DataNodeType * ch = find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize<DataNodeType,DeserializableT>( *ch, target );

        }

        /**
           If a child named subnodename is found in src then this function
           returns the result of deserialize(child), otherwise
           it returns 0.

           This is a convenience function: not part of the s11n kernel.
        */
        template <typename DataNodeType, typename DeserializableT>
        DeserializableT * deserialize_subnode( const DataNodeType & src,
                                               const std::string & subnodename )
        {
                const DataNodeType * ch = find_child_by_name( src, subnodename );
                if( ! ch ) return false;
                return deserialize<DataNodeType,DeserializableT>( *ch );
        }


        /**
           Clones an arbitrary SerializableType using it's
           DataNodeType serialization implementation.

           Returns a clone of tocp, or returns 0 on error.
           The caller owns the returned pointer.

           This copy is polymorphism-safe as long as all participating
           Serializables (re)implement the appropriate de/serialize
           operations, similarly to as they would do for a copy ctor
           or classical Clone() member function.

           Tip: clone() is a convenient way to test new de/serialize
           functions, e.g., for new Serializables, because if it works
           then deserializing from streams/files will also work. This
           function takes SerializableType through the whole
           de/serialize process, including classloading.
        */
        template <typename DataNodeType, typename SerializableType>
        SerializableType * clone( const SerializableType & tocp )
        {
                DataNodeType node;
                if( ! serialize( node, tocp ) ) return 0;
                return deserialize<DataNodeType,SerializableType>( node );
        }



        /**
           "Casts" t1 to t2 using serialization. This will work
           whenever t1 and t2 are "semantically compatible", whatever
           that really means. It can be used, e.g., to copy a
           list&lt;int&gt; to a vector&lt;double&gt;, provided both
           types have been proxied.

           Note that in the case of containers, the pointerness of the
           contained types is irrelevant: this works on both, thus
           a list&lt;int&gt; can be "cast" to a vector&lt;double*&gt;.

           As usual for a failed deserialization, if it returns false
           then t2 may be in an undefined state. There is no guaranty,
           however, that t2's deserialize operator will ever be
           called, as the serialization of t1 must first succeed
           for that to happen.
        */

        template <typename NodeType, typename Type1, typename Type2>
        bool s11n_cast( const Type1 & t1, Type2 & t2 )
        {
                NodeType n;
                return serialize<NodeType,Type1>( n, t1 ) && deserialize<NodeType,Type2>( n, t2 );
        }


} // namespace s11n

#undef NODEERR

#endif // s11n_DATA_SERIALIZE_H_INCLUDED
