////////////////////////////////////////////////////////////////////////
// reg_serializable_traits.hpp
//
// Supermacro for registering s11n proxies for the s11n API, using
// the s11n_traits<> technique introduced in 0.9.3.
//
// Vars used by this file:
//
// S11N_TYPE: type to be registered
//
// S11N_TYPE_NAME: stringified name of S11N_TYPE
//
// S11N_BASE_TYPE: if S11N_TYPE subclasses a regisitered S11N_TYPE,
// pass it here. If this is set then the rest of the following
// variables need not be set, as they are inherited.
//
// S11N_SERIALIZE_FUNCTOR: functor type responsible for handling calls
// to serialize<S11N_TYPE>(). Default is
// s11n::default_serialize_functor, which simply calls
// S11N_TYPE::operator()( NodeT & ).
//
// S11N_DESERIALIZE_FUNCTOR: functor type responsible for handling calls
// to serialize(). Defaults to S11N_SERIALIZE_FUNCTOR.
//
// S11N_FACTORY_TYPE: functor which is responsible for instantiating
// objects of type S11N_TYPE. Default behaviour is to use
// s11n::cl::object_factory< S11N_BASE_TYPE >.
//
// THINGS TO KNOW:
//
// - If the default factory type is NOT used, the caller is
// responsible for registering S11N_TYPE with the appropriate factory,
// using the key S11N_TYPE_NAME.
//
////////////////////////////////////////////////////////////////////////
// The s11n header files are expected to have been included by the
// time this file is ever included.
//
//
// Sample usage:
//
// #define S11N_TYPE std::map<Foo,Bar>
// #define S11N_TYPE_NAME "std::map<Foo,Bar>"
// #include [this file]
//
// // repeat for each type:
// #define S11N_TYPE std::map<Foo *,Bar *>
// #define S11N_TYPE_NAME "std::map<Foo*,Bar*>"
// #define S11N_SERIALIZE_FUNCTOR some_functor_name
// #include [this file]
//
// If S11N_TYPE is a subtype of another Serializable,
// and should be available via that base-type's classloader
// (Yes, you normally WANT this) then define this macro:
//
//   S11N_BASE_TYPE
//
// to the type name of S11N_TYPE's BaseType.
//
// If S11N_BASE_TYPE is not set then the following macros
// are also used:
//
// #define S11N_SERIALIZE_FUNCTOR some_serializer_functor
// #define S11N_DESERIALIZE_FUNCTOR some_deserializer_functor
//
// If S11N_DESERIALIZE_FUNCTOR is defined, that is used
// as the Deserialization proxy, otherwise S11N_SERIALIZE_FUNCTOR
// functor is used for both cases.
//
// If neither are set then ::s11n::default_serialize_functor
// is installed, and S11N_TYPE must conform to that proxy's
// expectations.
//
// After each include all of these macros are unset so that they may
// be immediately re-used for another registration.
////////////////////////////////////////////////////////////////////////


#ifndef S11N_TYPE
#  error "S11N_TYPE must be set before including this file."
#endif

#ifndef S11N_TYPE_NAME
#  ifdef S11N_NAME // older convention
#    define S11N_TYPE_NAME S11N_NAME
#    undef S11N_NAME
#  else
#    error "S11N_TYPE_NAME must be set before including this file. Set it to the stringified form of S11N_TYPE."
#  endif
#endif


// S11N_BASE_TYPE only needs to be set when registering subtypes of an
// already-registered base type...
#ifdef S11N_BASE_TYPE
#  define S11N_INHERIT 1
#else
#  define S11N_BASE_TYPE S11N_TYPE
#  define S11N_INHERIT 0
#  ifndef S11N_SERIALIZE_FUNCTOR
#    define S11N_SERIALIZE_FUNCTOR ::s11n::default_serialize_functor
#  endif
#  ifndef S11N_DESERIALIZE_FUNCTOR
#    define S11N_DESERIALIZE_FUNCTOR S11N_SERIALIZE_FUNCTOR
#  endif
#endif // S11N_BASE_TYPE


#ifndef S11N_FACTORY_TYPE
     // then register a factory of our own...
#    define S11N_REG_CLLITE 1
#    define S11N_FACTORY_TYPE s11n::cl::object_factory< S11N_BASE_TYPE >
#else
#    define S11N_REG_CLLITE 0
#endif // S11N_FACTORY_TYPE


////////////////////////////////////////////////////////////////////////
// Set up s11n_traits<> specialization...
namespace s11n {

        template <>
        struct s11n_traits < S11N_TYPE >
#if S11N_INHERIT
                : public s11n_traits < S11N_BASE_TYPE >
#endif
        {
#if ! S11N_INHERIT
                typedef S11N_TYPE serializable_type;
                typedef S11N_SERIALIZE_FUNCTOR serialize_functor;
                typedef S11N_DESERIALIZE_FUNCTOR deserialize_functor;
                typedef S11N_FACTORY_TYPE factory_type;
#endif // S11N_INHERIT
// #if S11N_REG_CLLITE
//                 static bool cl_reg_placeholder;
// #endif // S11N_REG_CLLITE

        };
// #if S11N_REG_CLLITE
//         bool s11n_traits< S11N_TYPE >::cl_reg_placeholder =
//                 (s11n::cl::classloader_register< S11N_BASE_TYPE, S11N_TYPE >( S11N_TYPE_NAME ),
//                  (CERR << "registering " << S11N_TYPE_NAME << "\n"),
//                  true);
// #endif // S11N_REG_CLLITE

} // namespace s11n
// end s11n_traits<>
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
// Register the type's name with ::classname<S11N_TYPE>().
// This is highly unfortunate, but currently required.
#define NAME_TYPE S11N_TYPE
#define TYPE_NAME S11N_TYPE_NAME
#include <s11n.net/name_type/name_type.hpp>
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////
#if S11N_REG_CLLITE
// register with default classloader...
#  define CL_TYPE S11N_TYPE
#  define CL_TYPE_NAME S11N_TYPE_NAME
#  define CL_BASE_TYPE S11N_BASE_TYPE
#    ifdef S11N_ABSTRACT_BASE
#        define CL_ABSTRACT_BASE 1
#    endif
#  include <s11n.net/cl/cllite_reg.hpp>
#endif // S11N_REG_CLLITE
////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////
// clean up for the next #include ...
#undef S11N_TYPE
#undef S11N_TYPE_NAME
#undef S11N_BASE_TYPE
#undef S11N_SERIALIZE_FUNCTOR
#undef S11N_DESERIALIZE_FUNCTOR
#undef S11N_FACTORY_TYPE
#undef S11N_INHERIT
#undef S11N_REG_CLLITE
#ifdef S11N_ABSTRACT_BASE
#  undef S11N_ABSTRACT_BASE
#endif
