#ifndef s11n_S11N_TRAITS_HPP_INCLUDED
#define s11n_S11N_TRAITS_HPP_INCLUDED 1


#include <vector>
#include <map>

#include <s11n.net/s11n/classload.hpp> // default classloader/factory implementation.

namespace s11n {

        /**
           A default serialization proxy, which simply
           forwards de/serialize calls to an interface
           implemented as two overloaded member functions
           SerializableType::operator()( NodeT ).
        */
        struct default_serialize_functor
        {
                /**
                   Serialize src to dest using src.operator()( dest ).

                   The serialize operator must look like:

                   bool operator()( NodeT & ) const;

                   It may be virtual or a function template.

                */
                template <typename NodeT, typename SerializableType>
                bool operator()( NodeT & dest, const SerializableType & src ) const
                {
                        return src.operator()( dest );
                }

                /**
                   Deserialize dest from src using dest.operator()( src ).

                   The deserialize operator must look like:

                   bool operator()( const NodeT & );

                   It may be virtual or a function template.
                */
                template <typename NodeT, typename DeserializableType>
                bool operator()( const NodeT & src, DeserializableType & dest ) const
                {
                        return dest.operator()( src );
                }
        };


        /**
           node_traits encapsulates information relevant to Data
           Nodes, much in the same way that std::char_traits
           encapsulates character type information.

           The default implementation works with
           s11n::s11n_node or API-compatible
           types. Specializations may be defined to work with
           other node types.

           By using node_traits, instead of directly accessing a
           Node's API, client code may remain blissfully ignorant of
           the underlying node type.
        */
        template <typename NodeT>
        struct node_traits
        {
                /**
                   The same as NodeT.
                */
                typedef NodeT node_type;

                /**
                   The type uses to store properties for node_type
                   objects.
                */
                typedef typename node_type::map_type property_map_type;

                /**
                   An iterator for properties.
                */
                typedef typename property_map_type::iterator iterator;

                /**
                   A const iterator for properties.
                */
                typedef typename property_map_type::const_iterator const_iterator;

                /**
                   The type used to store children of node_type
                   objects.
                */
                typedef typename node_type::child_list_type child_list_type;

                /**
                   An iterator for node children.
                */
                typedef typename child_list_type::iterator child_iterator;

                /**
                   A const iterator for node children.
                */
                typedef typename child_list_type::const_iterator child_const_iterator;

                /** Internal typedef. */
                typedef node_traits< node_type > this_type;


                /**
                   Returns a new node_type. The caller owns the
                   returned pointer.
                */
                static node_type * create()
                {
                        return new node_type;
                }

                /**
                   Returns a new node_type with the given name. The
                   caller owns the returned pointer.
                */
                static node_type * create( const std::string & name )
                {
                        node_type * n = this_type::create();
                        this_type::name( *n, name );
                        return n;
                }

                /**
                   Returns an iterator pointing to the first
                   property in node.
                */
                static iterator begin( node_type & node )
                {
                        return node.begin();
                }

                /**
                   Returns a const iterator pointing to the
                   first property in node.
                */
                static const_iterator begin( const node_type & node )
                {
                        return node.begin();
                }

                /**
                   Returns an iterator pointing to the
                   after-the-end property in node.
                */
                static iterator end( node_type & node )
                {
                        return node.end();
                }

                /**
                   Returns a const iterator pointing to the
                   after-the-end property in node.
                */
                static const_iterator end( const node_type & node )
                {
                        return node.end();
                }

                /**
                   Sets the property key to the given value in
                   the given node.

                   ValueT must support complementary ostream<< and
                   istream>> operators.
                */
                template <typename ValueT>
                static void set( node_type & node,
                                 const std::string & key,
                                 const ValueT & value )
                {
                        node.set( key, value );
                }

                /**
                   Unsets (removes) the given property from
                   node.
                */
                static void unset( node_type & node,
                                   const std::string & key )
                {
                        node.unset( key );
                }

                /**
                   Returns true if node contains a property
                   named key, else returns false.
                */
                static bool is_set( const node_type & node,
                                    const std::string & key )
                {
                        return node.is_set( key );
                }


                /**
                   Returns node's map of properties.
                */
                const property_map_type & properties( const node_type & node )
                {
                        return node.properties();
                }

                /**
                   Returns node's map of properties.
                */
                property_map_type & properties( node_type & node )
                {
                        return node.properties();
                }

                /**
                   Returns the value of the property with the given
                   key, or default_value if that property does not
                   exist or cannot be lexically cast to type ValueT.

                   ValueT must support complementary ostream<< and
                   istream>> operators.
                */
                template <typename ValueT>
                static ValueT
                get( const node_type & node,
                     const std::string & key,
                     const ValueT & default_value )
                {
                        return node.template get<ValueT>( key, default_value );
                }

                /**
                   Returns a mutable list of children
                   belonging to node.
                */
                static child_list_type & children( node_type & node )
                {
                        return node.children();
                }

                /**
                   Returns an immutable list of children
                   belonging to node.
                */
                static const child_list_type & children( const node_type & node )
                {
                        return node.children();
                }

                /**
                   Sets the class name of the type for which
                   node holds serialized data.
                */
                static void class_name( node_type & node, const std::string & classname )
                {
                        node.class_name( classname );
                }


                /**
                   Returns the class name of the type for
                   which node holds serialized data.
                */
                static std::string class_name( const node_type & node )
                {
                        return node.class_name();
                }

                /**
                   Sets node's name.
                */
                static void name( node_type & node, const std::string & name )
                {
                        node.name( name );
                }


                /**
                   Returns node's name.
                */
                static std::string name( const node_type & node )
                {
                        return node.name();
                }

                /**
                   Removes all children and properties from node,
                   freeing up their resources.
                */
                static void clear ( node_type & node )
                {
                        node.clear();
                }

        }; // end node_traits<>

        /**
           s11n_traits encapsulates information about what
           type(s) are responsible for handling de/serialize
           operations for a given type, plus the factory for
           that type. It should be specialized to define
           various aspects of serialization for a given type.

           Client code is not expected to need to use this
           type directly, except for purposes of plugging in
           their types into the s11n framework. More
           specifically, it is not expected that Serializables
           will use this type.

           The interface shown here is the bare minimum which
           s11n_traits specializations must implement.
           Specializations may optionally add to the
           interface, but client code is discouraged from
           relying on any extensions.

           Parameterized on:

           SerializableT: the base-most Serializable
           type. This is the base-most point of reference for
           classloading and "template typing". Subclasses of
           SerializableT are assumed to be handleable via the
           same de/serialize interface as SerializableT.
        */
        template <typename SerializableT>
        struct s11n_traits
        {

                /**
                   The type of object we want to [de]serialize.
                */
                typedef SerializableT serializable_type;

                /**
                   Type which will be used to instantiate new objects
                   of serializable_type. It must implement:

                   serializable_type * operator()( const std::string & classname ) const;

                   It is expected to return, polymorphically if
                   possible, a new serializable_type on success or 0
                   on failure.

                   The default factory_type works with types
                   registered via the s11n::cl::classload()
                   family of functions.
                */
                typedef ::s11n::cl::object_factory<serializable_type> factory_type;

                /**
                   Functor type implementing serialize code.

                   Must implement:

                   bool operator()( SomeNodeType & dest, const base_type & src ) const;

                */
                typedef ::s11n::default_serialize_functor serialize_functor;

                /**
                   Functor type implementing deserialize code.

                   Must implement:

                   bool operator()( const SomeNodeType & src, base_type & dest ) const;
                */
                typedef ::s11n::default_serialize_functor deserialize_functor;
        }; // end s11n_traits<>

//         namespace Private
//         {
//                 /**
//                    An internal helper to enable
//                    s11n_traits&lt;T*&gt;. Wraps up calls to ProxyT's
//                    operator()(NodeT,T&) by providing
//                    operator(NodeT,T*) implementations which
//                    dereference the T pointer and pass it on to
//                    ProxyT's implementation.

//                    Achtung: doesn't compile properly: ambiguity problem.
//                 */
//                 template <typename ProxyT>
//                 struct ser_pointer_type_wrapper
//                 {
//                         typedef ProxyT proxy_type;
//                          proxy_type proxy;

//                         /**
//                            Returns proxy( dest, src ).
//                         */
//                         template <typename NodeT, typename SerializableT>
//                         bool operator()( NodeT & dest, const SerializableT * src ) const
//                         {
//                                 if( ! src ) return false;
//                                 return proxy( dest, *src );
//                         }

//                         /**
//                            Returns proxy( src, dest ).
//                         */
//                         template <typename NodeT, typename DeserializableT>
//                         bool operator()( const NodeT & src, DeserializableT * dest ) const
//                         {
//                                 if( ! dest ) return false;
//                                 return proxy( src, *dest );
//                         }
//                 };
//         } // namespace Private

//         /**
//            A specialization to catch pointer types. With this specialization
//            in place, de/serialize( node, foo ) and de/serialize(node, &foo) will
//            work identically.
//         */
//         template <typename SerializableType>
//         struct s11n_traits < SerializableType * >
//         {
//                 typedef s11n_traits < SerializableType > parent_type;
//                 typedef typename parent_type::serializable_type serializable_type;
//                 typedef typename Private::ser_pointer_type_wrapper<typename parent_type::serialize_functor> serialize_functor;
//                 typedef typename Private::ser_pointer_type_wrapper<typename parent_type::deserialize_functor> deserialize_functor;
//                 typedef typename parent_type::factory_type factory_type;
//         };


}  // namespace s11n


#endif // s11n_S11N_TRAITS_HPP_INCLUDED
