////////////////////////////////////////////////////////////////////////
// valarray.hpp: s11n algos/proxies for handling std::valarray objects.
// This header is not included by default: clients who want to de/serialize
// valarrays must include it themselves.
////////////////////////////////////////////////////////////////////////
#ifndef s11n_VALARRAY_HPP_INCLUDED
#define s11n_VALARRAY_HPP_INCLUDED 1


namespace { // set up class_name<> specializations...

        ////////////////////////////////////////////////////////////
        // std::valarray
        template <typename ValueType>
        struct class_name< std::valarray<ValueType> >
        {
                typedef ValueType value_type;
                typedef std::valarray<ValueType> valarray_type;
                static bool cl_reg_placeholder;
                static const char * name()
                {
                        static std::string n = "valarray";
                        if( !cl_reg_placeholder && (cl_reg_placeholder=true) )
                        {
                                ::s11n::cl::classloader_register<
                                valarray_type,valarray_type
                                        >( n );
                        }
                        return n.c_str();
                }
        };
        template <typename VT>
        bool class_name< std::valarray<VT> >::cl_reg_placeholder = (::classname< std::valarray<VT> >(),true);
} // anonymous ns

namespace s11n {

        /**
           s11n::va encapsulates de/serialization operators for
           std::valarray objects.
        */
        namespace va {

                /**
                   Serializes src to dest. Returns true on success,
                   false on error. VAT must be a std::valarray type
                   with a numeric value_type.
                */
                template <typename NodeT, typename VAT>
                bool serialize_valarray( NodeT & dest, const VAT & src )
                {
                        typedef s11n::node_traits<NodeT> TR;
                        TR::class_name( dest, ::classname<VAT>() );
                        static const int bsize = 10;
                        char num[bsize];
                        char fmt[bsize];
                        size_t sz = src.size();
                        int places = 1; // # of digits to use
                        for( ; sz >= 0x0f; sz = (size_t)(sz/0x0f)) { ++places; }
                        snprintf( fmt, bsize, "x%%0%dx", places );
                        sz = src.size();
                        TR::set( dest, "size", sz );
                        for( size_t i = 0 ; i < sz; i++ )
                        {
                                snprintf( num, bsize, fmt, i );
                                TR::set( dest, num, src[i] );
                        }
                        return true;
                }

                /**
                   Deserializes dest from src. Returns true on
                   success, false on error. VAT must be a
                   std::valarray type with a numeric value_type.
                */
                template <typename NodeT, typename VAT>
                bool deserialize_valarray( const NodeT & src, VAT & dest )
                {
                        typedef node_traits<NodeT> TR;
                        typedef typename VAT::value_type VT;
                        typename TR::const_iterator it = TR::begin(src), et = TR::end(src);
                        const static std::string szkey = "size";
                        size_t size = TR::get( src, szkey, dest.size() );
                        VT defaultval;
                        dest.resize( size, defaultval );
                        size_t i = 0;
                        for( ; et != it; ++it )
                        {
                                if( szkey == (*it).first ) continue;
                                dest[i++] = tostring::from_string<VT>( (*it).second, defaultval );
                        }
                        return true;
                }

                /**
                   A Serializable proxy for valarray types.
                */
                struct valarray_serializable_proxy
                {

                        valarray_serializable_proxy()
                        {}

                        /**
                           see serialize_valarray().

                        */
                        template <typename NodeType, typename SerType>
                        bool operator()( NodeType & dest, const SerType & src ) const
                        {
                                return serialize_valarray( dest, src );
                        }

                        /** see deserialize_valarray(). */
                        template <typename NodeType, typename SerType>
                        bool operator()( const NodeType & src, SerType & dest ) const
                        {
                                return deserialize_valarray( src, dest );
                        }


                };

        } // namespace va



        /**
           s11n_traits<> specialization for std::valarray types.
        */
        template <typename ValueType>
        struct s11n_traits < std::valarray<ValueType> >
        {
                typedef std::valarray<ValueType> serializable_type;
                typedef ::s11n::va::valarray_serializable_proxy serialize_functor;
                typedef serialize_functor deserialize_functor;
                typedef ::s11n::cl::object_factory<serializable_type> factory_type;
        };


} // namespace s11n



#endif // s11n_VALARRAY.HPP_HPP_INCLUDED
