#ifndef s11n_SERIALIZERS_HPP_INCLUDED
#define s11n_SERIALIZERS_HPP_INCLUDED 1

////////////////////////////////////////////////////////////////////////////////
// serializers.hpp: Some utility code for working with
//  s11n::io Serializer types.
//
// License: Public Domain
// Author: stephan@s11n.net
////////////////////////////////////////////////////////////////////////////////

#include <s11n.net/s11n/s11n_config.hpp>

#include <s11n.net/s11n/io/data_node_io.hpp> // data_node_serializer<> class.


namespace s11n {
        namespace io {

                /**
                   Populates target list with the names of registered
                   Serializers. ListT must support push_back( std::string ).

                   If onlyShortNames is true (the default) then only
                   "simple" names are put into target, not the "full"
                   names of the classes. This is to make the data more
                   useful in the context of client applications as,
                   e.g., a list of choices for users.

                   Note that only serializers extending from
                   s11n::io::data_node_serializer<NodeT> are
                   returned. If onlyShortNames is true then only names
                   which contain only alphanumeric or underscore
                   characters are returned. This is used for filtering
                   out all names except Serializer aliases, excluding
                   the full class names and any magic cookie aliases.
                */
                template <typename NodeT, typename ListT>
                void serializer_list( ListT & target, bool onlyShortNames = true )
                {
                        typedef ::s11n::io::data_node_serializer<NodeT> BaseSerT;
                        typedef ::cl::class_loader< BaseSerT > SerCL;
                        typedef typename SerCL::alias_map AMap;
#define SERCL ::s11n::cl::classloader< BaseSerT >()
                        typename AMap::const_iterator cit = SERCL.aliases().begin(),
                                cet = SERCL.aliases().end();
#undef SERCL
                        std::string alias;
                        static const std::string nonspecial = 
                                "_0123456789abcdefghijklmnopqrstuvwqxyzABCDEFGHIJKLMNOPQRSTUVWQXYZ";
                        for( ; cet != cit; ++cit )
                        {
                                alias = (*cit).first;
                                if( onlyShortNames )
                                { // filter out all but "simple" names:
                                        if( std::string::npos !=
                                            alias.find_first_not_of( nonspecial )
                                            )
                                        {
                                                continue;
                                        }
                                }
                                target.push_back( alias );
                        }
                }


                /**
                   Registers a Serializer type with the s11n::io layer. It must:

                   - be templatized on a DataNodeType

                   - subclass s11n::io::data_node_serializer<DataNodeType>

                   - provide a node_type typedef which is the same as DataNodeType

                   i.e., the conventions of all of the Serializers
                   included with libs11n.


                   Registering makes the type available to the
                   data_node_serializer classloader.


                   Arguments:

                   - classname = SerializerT's stringified class name.

                   - alias = a "friendly" name for the SerializerT.


                   SerializeT's magic_cookie() function is called to
                   alias the cookie as an alias for classname. Thus, a
                   SerializerT object is (briefly) instantiated.

                   Node that this functions essentially performs the
                   same operations as the reg_serializer.hpp
                   supermacro, and the two should be equivalent
                   (though this seems easier to use).
                */
                template <typename SerializerT>
                void register_serializer( const std::string & classname, const std::string & alias )
                {
                        static bool inited = false;
                        if( ! inited && (inited=true) )
                        {
                                cllite::class_path().add_extension( "_serializer.so" );
                                cllite::class_path().add_extension( "_serializer.dll" );
                        }
                        //CERR << "register_serializer(" << classname << ","<<alias<<")\n";
                        typedef SerializerT ST;
                        typedef typename ST::node_type NT;
                        typedef s11n::io::data_node_serializer<NT> BaseST;
                        ::s11n::cl::classloader_register< BaseST, ST >( classname );
                        ::s11n::cl::classloader< BaseST >().alias( alias, classname );
                        ::s11n::cl::classloader< BaseST >().alias( ST().magic_cookie(), classname );
                }

                /**
                   Returns a Serializer object, which must have been registered with
                   s11n::cl::classloader< s11n::io::data_node_serializer<NodeT> >().

                   The caller owns the returned pointer, which may be 0.
                */
                template <typename NodeT>
                s11n::io::data_node_serializer<NodeT> *
                create_serializer( const std::string & classname )
                {
                        typedef s11n::io::data_node_serializer<NodeT> BaseSerT;
                        BaseSerT * s = 0;
                        if( ( s = s11n::cl::classload< BaseSerT >( classname ) ) )
                        {
                                return s;
                        }
                        static const char * addon = "_serializer";
                        if( (std::string::npos == classname.find(addon)) ) // try harder!
                        {
                                std::string harder = classname + addon;
                                //CERR << "Trying harder for " << classname << " --> " << harder << "!\n";
                                s = create_serializer<NodeT>( harder );
                        }
                        //CERR << "Harder try= " << std::hex << s << "\n";
                        return s;
                }



        } // namespace io
} // namespace s11n


#endif // s11n_SERIALIZERS_HPP_INCLUDED
