#ifndef acme_ALGO_H_INCLUDED
#define acme_ALGO_H_INCLUDED 1

/////////////////////////////////////////////////////////////////////////
// algo.h: generic algorithms
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
/////////////////////////////////////////////////////////////////////////

#include <functional> // for_each()

#include "functor.hpp" // object_deleter()

namespace acme {


        /**
           For each item in [begin,end) object_deleter()(*item) is called.

           After this call the container from which the iterators come
           still contains the items but they are invalid - deleted
           pointers. The client should call erase(begin,end) to delete
           the entries, or use convenience function
           free_list_entries(), which accepts a list-style
           container.
        */
        template <typename IterT>
        void delete_objects( IterT begin, IterT end )
        {
                std::for_each( begin, end, object_deleter() );
        }



        /**
           For each item in [first,last), copies the item to OutputIt
           if pred(*item) returns true.

           Copied from:

           http://www.bauklimatik-dresden.de/privat/nicolai/html/en/cpp.html
        */

        template <typename InputIt, typename OutputIt, typename UnaryPredicate>
        OutputIt copy_if (InputIt first,
                          InputIt last,
                          OutputIt result,
                          UnaryPredicate pred) 
        {
                while (first != last)
                {
                        if (pred(*first)) *result++ = *first;
                        ++first;
                }
                return result;
        }



        /**
           Deallocates all objects in list. If the entries
           are pointers they are deleted, else this is a no-op.

           After this call, list will be empty.
        */
        template <typename ListType>
        void free_list_entries( ListType & list )
        {
                object_reference_wrapper<ListType> wr(list); // in case ListType is a pointer type
                if( ! wr.good() ) return;
                delete_objects( wr().begin(), wr().end() );
                wr().clear();
        }


       /**
           Deletes any pointers in map, and does nothing for reference
           types, then calls map.clear(). This is used to treat
           arbitrary maps containing pointers or value types
           identically, by applying the same set of deallocation rules
           for both cases, simplifying some template implementations.

           After calling this, map will be empty.
        */
        template <typename MapType>
        void free_map_entries( MapType & map )
        {
                object_reference_wrapper<MapType> wr(map); // in case MapType is a pointer type
                if( ! wr.good() ) return;
                std::for_each( wr().begin(), wr().end(), pair_entry_deallocator() );
                wr().clear();
        }


}

#endif // acme_ALGO_H_INCLUDED
