// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
#ifndef acme_CHILDRENSERVER_H_INCLUDED
#define acme_CHILDRENSERVER_H_INCLUDED 1

#include <string>
#include <list>
#include <map>
#include <vector>
//#include <deque> // causes crashes in some cases where vector does not :/

namespace acme
{
	/**
           EXPERIMENTAL!

           children_server is an experimental object for managing
           parent-child relationships between objects.

           The interface should allow any given class, including
           non-modifiable 3rd-party classes, to have any number of
           child types.

           This code does no automatic clean-up of children. Client
           code may free all child pointers by calling
           cleanup_parent().

           Parameterized on:

           PType = type of the parent objects.

           CType = type of the child objects.

        */
	template < class PType, class CType = PType > class children_server
	{
	      public:
                /**
                   The type of this object's parents in
                   parent-child relationships.
                 */
		typedef PType parent_type;

                /**
                   The type of this object's children in
                   parent-child relationships.
                 */
		typedef CType child_type;

                /**
                   The container type used to store the lists of children.
                 */
		typedef std::list < child_type * >list_type;

                /**
                   A shortcut typedef to help ease the implementation code for this class.
                 */
		typedef children_server < parent_type, child_type > this_type;

                /**
                   iterator which can be dereferenced to a (child_type *).
                 */
		typedef typename list_type::iterator iterator;

                /**
                   iterator which can be dereferenced to a (const child_type *).
                 */
		typedef typename list_type::const_iterator const_iterator;

		/**
                   Returns the child list for the given parent
                   object. If creationPolicy is non-zero then this
                   function will create the child list if it does not
                   yet exist (in that case, this function will never
                   return NULL except on an out-of-memory error).

                   The caller takes ownership of the returned
                   pointer. All children in the list can be deleted at
                   once by calling cleanup_parent( parent ).

                   Different calls to this function will always return
                   the same list (or the same NULL, depending on
                   creationPolicy ;) until either unmap_parent() or
                   cleanup_parent() are called, in which case further
                   calls to this function may return a different
                   pointer the next time it is called.
                */
		static this_type::list_type *
                child_list( const this_type::parent_type * parent, int creationPolicy = 0 )
		{
			if ( !parent ) return NULL;
			static this_type::map_type & cmap = parent_children_map();
			typename map_type::const_iterator it = cmap.find( parent );
			if ( cmap.end() != it ) return ( *it ).second;
			if ( 0 == creationPolicy ) return NULL;
			list_type *cl = new list_type();
                        cmap[parent] = cl;
                        return cl;
		}


		/**
                   Removes parent from this object's internal
                   map. This only removes the pointer to the parent's
                   child list and the mapping which binds the parent
                   to the children, but does not delete() any children
                   nor the child list attached to that parent.

                   Returns true if it unmaps the parent, else
                   false. It will only fail if it doesn't have an
                   entry for parent.

                   This is more of a maintenance detail than anything
                   else, but can be used to take over control of a
                   child list from this class.

                */
		static bool unmap_parent( const this_type::parent_type * parent )
		{
			if ( !parent ) return false;
			static this_type::map_type & cmap = parent_children_map();
			typename this_type::map_type::iterator it = cmap.find( parent );
			if ( it == cmap.end() ) return false;
			cmap.erase( parent );
			return true;
		}

		/**
                   Simlar to unmap_parent(), but also deletes the
                   children in the list and then deletes the list.

                   Subclasses or proxies of this class should call
                   this function from their dtor, in order to avoid
                   leaving any dangling pointers in this class'
                   internal data.
                */
		static bool cleanup_parent( const this_type::parent_type * parent )
		{
			if ( !parent )
				return false;
			static this_type::map_type & cmap = parent_children_map();
			typename this_type::map_type::iterator it = cmap.find( parent );
			if ( it == cmap.end() )
			{
				// we were probably just never registed because children() was never called.
				return false;
			}
			typename this_type::list_type * li = ( *it ).second;
			if ( !unmap_parent( parent ) )
			{
				return false;
			}
			typename this_type::list_type::iterator vit;
			typename this_type::child_type * child = 0;
			for ( vit = li->begin(); li->begin() != li->end(); )
			{
                                // note: i explicitely call begin()/end() on each iteration
                                // because i'm not certain of iterator invalidation here :/
				child = ( *vit );
				li->erase( vit );
				delete( child );
				child = 0;
			}
			delete( li );
			return true;
		}


	      private:
		typedef std::map < const this_type::parent_type *, this_type::list_type * > map_type;
                /**
                   i hate working with static objects in template
                   classes (for syntax reasons), and parent_children_map()
                   is a helper to avoid that.
                 */
		static map_type & parent_children_map()
		{
			static map_type meyers;
			return meyers;
		}

	};

} //  namespace acme
#endif // acme_CHILDRENSERVER_H_INCLUDED
