#ifndef acme_FUNCTOR_H_INCLUDED
#define acme_FUNCTOR_H_INCLUDED 1
/////////////////////////////////////////////////////////////////////////
// functor.h: generic functors
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
/////////////////////////////////////////////////////////////////////////

#include <string>
#include "acme_debuggering_macros.hpp"

namespace acme
{

//         /**
//            Swiped from Alexandrescu:
//            http://www.cuj.com/experts/1810/
//         */
//         template <int v>
//         struct Int2Type
//         {
//                 enum { value = v };
//         };

 

        /**
           Deletes an object passed to it.

           This type accepts non-pointer types for "destruction." This
           is a non-operation, and is supported to allow other
           template code to generically free objects without needing
           to know if they are pointers. This allows some
           formerly-non-consolidatable reference-vs-pointer-type code to
           share a single implementation, as "delete(myobj)" is not
           valid for non-pointer types, but object_deleter()(myobj)
           is.
        */
        struct object_deleter
        {
                /**
                   Deletes t.
                */
                template <typename T>
                void operator()( T * t ) const
                {
                        // ACH!!!! If we use (const T *) for the arguments
                        // then the (const T &) version is called
                        // even when a pointer type is passed in!!!!

                        // i don't fully understand why
                        // delete( const T * ) is legal, 
                        // considering that it triggers a dtor,
                        // and dtors are non-const.
                        //CERR << "object_deleter deleting "<< std::hex<<t<<"\n";
                        delete( t );
                }
                /**
                   Does nothing: is here to allow some
                   reference-vs-pointer-type transparency.
                */
                template <typename T>
                void operator()( const T & t ) const
                {
                        //CERR << "object_deleter no-op\n";
                }
        };



        /**
           object_reference_wrapper is a type for giving access
           to T objects via their dot operator, regardless of whether
           they are pointers or not.

           Intended for use with value_types which come from, e.g.,
           std::list, so objects of those types can be called using
           the same syntax regardless of whether they are pointer
           types or not.

           e.g., assuming MyType might be a pointer or a reference,
           we can ignore that difference for call-syntax purposes
           with:
<pre>           
object_reference_wrapper&lt;MyType&gt; wrap;
wrap(myobj).memberfunc();
</pre>

or:

<pre>
object_reference_wrapper&lt;MyType&gt; wrap(myobj);
wrap().memberfunc();           
</pre>
        */
        template <typename T>
        struct object_reference_wrapper
        {
                typedef T value_type;
                typedef T base_value_type;
                object_reference_wrapper() : m_ptr(0) {}
                object_reference_wrapper( value_type &obj ) : m_ptr(&obj) {};
                /**
                   Sets this object's proxy object to t and returns t.
                 */
                base_value_type & operator()( value_type & t )
                {
                        this->m_ptr = &t;
                        return t;
                }
                /**
                   Returns this object's wrapped object.

                   ACHTUNG: this function WILL Cause Grief if it is called
                   on a default-constructed version of this object: you must
                   set this object's wrapped value using the ctor (or via
                   copying) before calling this.
                 */
                base_value_type & operator()() const { return *(this->m_ptr); }

                /**
                   Returns true if this object is wrapping a non-0 object, else
                   false.
                */
                bool good() const
                {
                        return 0 != this->m_ptr;
                }

        private:
                value_type * m_ptr;
        };

        /**
           A specialization to wrap pointers to (T *) such that they
           can be accessed, via this wrapper, using a dot instead of
           <tt>-&gt;</tt>.
        */
        template <typename T>
        struct object_reference_wrapper<T *>
        {
                typedef T * value_type;
                typedef T base_value_type;
                object_reference_wrapper() : m_ptr(0) {}
                object_reference_wrapper( value_type & obj ) : m_ptr(obj) {};
                /** Sets this object's proxied object to t and Returns t. */
                base_value_type & operator()( value_type & t )
                {
                        this->m_ptr = t;
                        return *t;
                }
                /** Returns this object's wrapped object.

                   ACHTUNG: this function WILL Cause Grief if it is
                   called on a default-constructed version of this
                   object: you must set this object's wrapped value
                   using the ctor, operator(T), or via copying this
                   object from a properly-initialized one before
                   calling this.
                */
                base_value_type & operator()() const { return *(this->m_ptr); }
        private:
                base_value_type * m_ptr;
        };


        /**
           const_object_reference_wrapper is identical in usage to
           object_reference_wrapper, except that it deals with const
           objects. It is a separate functor to avoid ambiguity and
           some impossible overloads.
        */
        template <typename T>
        struct const_object_reference_wrapper
        {
                typedef T value_type;
                typedef T base_value_type;
                const_object_reference_wrapper() : m_ptr(0) {}
                const_object_reference_wrapper( const value_type &obj ) : m_ptr(&obj) {};
                /**
                   Sets this object's proxied obj to t and returns t.
                 */
                const base_value_type & operator()( const value_type & t )
                {
                        this->m_ptr = t;
                        return t;
                }
                /**
                   Returns this object's wrapped object.

                   ACHTUNG: this function WILL Cause Grief if it is
                   called on a default-constructed version of this
                   object: you must set this object's wrapped value
                   using the ctor, operator(T), or via copying this
                   object from a properly-initialized one before
                   calling this.
                 */
                const base_value_type & operator()() const { return *this->m_ptr; }
        private:
                const value_type * m_ptr;
        };

        /**
           A specialization to wrap pointers to (T *) such that they
           can be accessed, via this wrapper, using a dot instead of
           <tt>-&gt;</tt>.
        */
        template <typename T>
        struct const_object_reference_wrapper<T *>
        {
                typedef T * value_type;
                typedef T base_value_type;
                const_object_reference_wrapper() : m_ptr(0) {}
                const_object_reference_wrapper( const value_type & obj ) : m_ptr(obj) {};
                /** Returns (*t). */
                const base_value_type & operator()( value_type & t )
                { 
                        this->m_ptr = t;
                        return *t;
                }
                /** Returns this object's wrapped object. */
                const base_value_type & operator()() const { return *(this->m_ptr); }
        private:
                const base_value_type * m_ptr;
        };


        /**
           A functor allowing pairs of PODs and pointers to be
           mixed together in any combination and be deallocated in a
           uniform way. Admitedly, it's a waste of time for value types,
           but the point is a single functor which can work for all
           pair types.

           sample:

           std::for_each( map.begin(), map.end(), pair_entry_deallocator() );
           
        */
        struct pair_entry_deallocator
        {
                /**
                   Deallocates p.first and p.second. PairType must not
                   be a pointer type.
                */
                template <typename PairType>
                void operator()( PairType & p ) const
                {
                        object_deleter od;
                        //CERR << "pair_entry_deallocator wiping out something...\n";
                        od( p.first );
                        od( p.second );
                }
        };


        /**
           child_pointer_deep_copier is a functor to deep-copy
           a list of pointers into another list. Designed for
           use with std::for_each and the like.

	   Assuming T is the type contained in ListType, stripped
	   of any pointer part, then the following must hold:


           - List must support <code>push_back( T * )</code>.

	   - This must be a valid expression:

	   <code>T * t = new T( *p );</code>
	   
	   Where p is a passed to this type's operator().

           ACHTUNG: This is only useful for non-polymorphic
           copying.

           It might be interesting to note that copying
           monomorphic s11n::data_node objects this way is
           "pseudo-polymorphic" - the copy itself is
           monomorphic but the data needed to deserialize the
           proper type from the node is maintained.
        */
        template <typename ListType>
        class child_pointer_deep_copier
        {
        private:
                template <typename T> struct PtrStrip { typedef T type; };
                template <typename T> struct PtrStrip<T *> { typedef T type; };
        public:
                typedef ListType list_type;
                typedef typename ListType::value_type full_value_type;
                typedef typename PtrStrip<full_value_type>::type value_type; // that is, list_type::value_type, minus any pointer part.
                /**
                   Target list must outlive this object.
                */
                child_pointer_deep_copier( list_type & target ) : m_childs(&target)
                {}

                /**
                   Inserts a copy of p into this object's list and returns true.

                   Returns true if p is successfully copied.

                   If an exception thrown while copying this function has
                   no effect, and the target node is not modified.

                   The target node takes ownership of the new copy of p.
                */
                bool operator()( const value_type * p )
                {
                        if( ! this->m_childs || ! p ) return false;
                        value_type * cp = 0;
                        try
                        {
                                cp = new value_type( *p ); // todo: polymorphic cloning!
                                if( ! cp ) return false;
                        }
                        catch(...) {
                                delete( cp ); // i honestly don't know if i need this here :/
                                return false;
                        }
                        this->m_childs->push_back( cp );

                        return true;
                }
        private:
                list_type * m_childs;
        };


        /**
           Helper to avoid using bind1st/bind2nd.
        */
        template <typename ComparedType>
        struct equal_to
        {
                typedef ComparedType compared_type;
                explicit equal_to( const compared_type & d ) : m_data(d)
                {}

                /**
                   Returns true if p compares equal to the value
                   passed to this object's ctor.
                */
                void operator()( const compared_type & p )
                {
                        return p == this->m_data;
                }
        private:
                compared_type m_data;
        };

        /**
           Functor to return true if given NameableT objects match a
           certain name. NameableT must support:

           std::string name() const;

           This class is admittedly to avoid the use of bind1st/bind2nd
           :/.
        */
        template <typename NameableT>
        struct same_name
        {
                typedef NameableT nameable_type;
                explicit same_name( const std::string & n ) : m_name(n)
                {}

                bool operator()( const nameable_type * x ) const
                {
                        if( ! x  ) return false;
                        return x->name() == this->m_name;
                }
        private:
                std::string m_name;
        };


} // namespace acme

#endif // acme_FUNCTOR_H_INCLUDED
