#ifndef acme_POINTER_CLEANER_H_INCLUDED
#define acme_POINTER_CLEANER_H_INCLUDED
////////////////////////////////////////////////////////////////////////////////
// pointer_cleaner.h
// a poor-man's multi-pointer cleaner-upper.
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
////////////////////////////////////////////////////////////////////////////////

#include <algorithm> // for_each()
#include <list>

#include "functor.hpp"  // object_deleter

namespace acme
{

        /**
           A poor-man's garbage collector. It destroys any pointers
           added to it when it is destroyed.

           This type models Non-Copyable. The only thing of use
           you can do with them is create them and then use their
           add() function.
        */
        template <typename T>
        class pointer_cleaner
        {
        public:
                typedef T value_type;

                /**
                   This is the only way to construct a new cleaner:
                   they may not be copied.
                */
                pointer_cleaner() {}

                /**
                   Deletes any objects added via o.
                */
                ~pointer_cleaner()
                {
                        std::for_each( this->m_list.begin(),
                                       this->m_list.end(),
                                       object_deleter()
                                       );
                }

                /**
                   Adds o to be deleted when this object goes out of scope.
                   Once you do this, there is no way to remove o from the
                   this object's Path of Destruction.

                   Returns the ojbect it is passed so that it may be used
                   in call chains, like:
                   <code>int * foo = cleaner.add( new int );</code>
                */
                value_type * add( value_type * o )
                {
                        this->m_list.push_back( o );
                        return o;
                }

        private:
                struct object_deleter
                {
                        template <typename X>
                        void operator()( const X * t )
                        { // i don't understand why delete( const * ) is legal.
                                delete( t );
                        }
                };

                pointer_cleaner( const pointer_cleaner & ); // unimpl
                pointer_cleaner & operator=( const pointer_cleaner & ); // unimpl
                typedef std::list<value_type *> list_type;
                list_type m_list;
        };

} // namespace

#endif // acme_POINTER_CLEANER_H_INCLUDED
