#ifndef acme_PROPERTY_SERVER_H_INCLUDED
#define acme_PROPERTY_SERVER_H_INCLUDED 1
#include <string>

#include <s11n.net/phoenix/phoenix.hpp> // phoenix class

#include <s11n.net/tostring/to_string.hpp> // to/from_string()

namespace acme {

        /**
           property_server is an experimental class for tying
           arbitrary properties to arbitrary objects. It supports, as
           values, any data types which are i/ostreamable using their
           <code> >>istream</code> or <code>ostream<< </code>
           operators.

           License: Do As You Damned Well Please

           Author: stephan@s11n.net
        */

        template <typename ObjIDType>
        class property_server
        {
        public:

                /** The type used to key objects to their properties. */
                typedef ObjIDType object_id_type;

                /** The key type used for properties. */
                typedef std::string key_type;

                /** The underlying data store for properties. */
                typedef std::map<key_type,std::string> property_map_type;


                typedef property_server< ObjIDType > this_type;
                typedef std::map<object_id_type,property_map_type> map_type;

                map_type & object_map()
                {
                        return this->m_map;
                }
                const map_type & object_map() const
                {
                        return this->m_map;
                }


                /**
                   Sets the property key to val for objid.
                */
                template <typename ValueType>
                void set( const object_id_type objid, const key_type & key, const ValueType & val )
                {
                        object_map()[objid][key] = tostring::to_string( val );
                }

                /**
                   Returns the property key for objid. If there is an error
                   converting the value to ValueType, or if the property does
                   not exist, defaultval is returned. This can be used to
                   implement custom error handling by passing a known-invalid
                   value as defaultval and interpretting it as an error
                   value. If there is *no* known-invalid combinations then you
                   can check twice with two different defaulvals. If
                   defaultval is returned both times then there is definately
                   an error, because valid data would only be interpretted one
                   way and you've checked against two different defaultvals.
                */
                template <typename ValueType>
                ValueType get( const object_id_type objid, const key_type & key, const ValueType & defaultval ) const
                {
                        typename map_type::const_iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return defaultval;
                        const property_map_type & pmap = it.second;
                        typename property_map_type::const_iterator pit = pmap.find( key );
                        if( pmap.end() == pit ) return defaultval;
                        return tostring::from_string<ValueType>( pit.second, defaultval );
                }

                /**
                   Removes the given property from the given object.
                */
                void unset( const object_id_type objid, const key_type & key )
                {
                        typename map_type::iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return;
                        property_map_type & pmap = (*it).second;
                        typename property_map_type::iterator pit = pmap.find( key );
                        if( pmap.end() == pit ) return;
                        pmap.erase( pit );
                        if( pmap.empty() ) object_map().erase( it );
                }

                /**
                   Removes the property map for objid, freeing up any
                   resources it consumed.
                */
                void clear( const object_id_type objid )
                {
                        typename map_type::iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return;
                        object_map().erase( it );
                }

                /** Removes all properties for all objects. */
                void clear()
                {
                        object_map().clear();
                }

                /**
                   Returns a pointer to the given object's property map, or 0
                   if no properties have been set for the object (or if
                   clear(objid) has been called).

                   The caller does not own the returned pointer and (ideally)
                   should not modify the map directly (prefer get() and
                   set()). It is provided here primarily for serialization
                   purposes: so the map can be saved and loaded using
                   arbitrary serialization techniques.
                */
                property_map_type * property_map( const object_id_type objid )
                {
                        typename map_type::iterator it = object_map().find( objid );
                        if( object_map().end() == it ) return 0;
                        property_map_type * pm = & ((*it).second);
                        return pm;
                }

                /**
                   Calls set( objid, p.first, p.second ).
                */
                template <typename PairT>
                void insert( object_id_type objid, const PairT & p )
                {
                        this->set( objid, p.first, p.second );
                }

        private:
                map_type m_map;

        };

} // namespace acme

#endif // acme_PROPERTY_SERVER_H_INCLUDED
