#ifndef S11N_NET_ACME_UID_HPP_INCLUDED
#define S11N_NET_ACME_UID_HPP_INCLUDED 1

#include <stddef.h> // size_t
#include <map>

////////////////////////////////////////////////////////////////////////
// uidgen.hpp - classes for generating and working with unique
// identifiers (UIDs).
//
// Author: stephan@s11n.net
// License: Public Domain
// CVS Revision: $Revision: 1.5 $
////////////////////////////////////////////////////////////////////////

namespace acme
{

        /**
           A simple default UID generator which simply returns
           sequential numbers. The numbers are sequential only
           within like ContextTs.

           Intended for use with the uidgen class.
        */
        template <typename ContextT>
        struct context_uid_generator
        {
                /** The type returned by operator(). */
                typedef size_t value_type;

                /** This type's "sharing context". */
                typedef ContextT context_type;

                /**
                   Returns the next sequential ID for this type's
                   context. Guaranteed to never return the value 0, so
                   that value can be reserved for client-internal
                   purposes.
                */
                size_t operator()() const
                {
                        static value_type m_id;
                        ++m_id;
                        return (0 == m_id) ? ++m_id : m_id;

                }
        };

        /**
           A simple default UID generator which simply returns
           sequential numbers.

           Intended for use with the uidgen class.
        */
        struct default_uid_generator
        {
                /** The type returned by operator(). */
                typedef size_t value_type;
                /**
                   Returns the next sequential ID for this type's
                   context. Guaranteed to never return the value 0, so
                   that value can be reserved for client-internal
                   purposes.
                */
                size_t operator()() const
                {
                        static value_type m_id;
                        ++m_id;
                        return (0 == m_id) ? ++m_id : m_id;
                }
        private:

        };


        /**
           EXPERIMENTAL.

           A front-end for doling out unique IDs.

           Parameterized on:

           UIDGeneratorT: a functor which can generate unique IDs. It
           must export a value_type typedef which describes it's
           operator() return type. The default UIDGeneratorT returns
           sequential numbers irrespective of any context.

           UIDGeneratorT should have one specific UID which is never
           returned, such that it can be used for client-specific
           purposes as a market UID. For example, the default
           implementations reserve the value 0, and will never return
           such a UID, so that client code may use that as a "has no
           UID yet" marker. A generator which returns std::string keys
           might reserve, e.g., the empty string. A useful guideline
           is probably to reserve the value of a default-constructed
           value_type object.
        */
        template <typename UIDGeneratorT = default_uid_generator>
        struct uidgen
        {
                /** The UID generator type. */
                typedef UIDGeneratorT generator_type;

                /** The type returned by generator_type's operator(). */
                typedef typename UIDGeneratorT::value_type value_type;

                /**
                   Returns the result of generator_type's operator().
                */
                static value_type next_uid()
                {
                        static generator_type bob;
                        return bob();
                }

                /**
                   Guaranteed to be equivalent to calling next_uid().
                */
                value_type operator()() const
                {
                        return next_uid();
                }
        };

        /**
           A base type for objects which should get a UID upon
           construction.
        */
        template <typename UIDGeneratorT = default_uid_generator>
        class uidable
        {
        public:
                typedef UIDGeneratorT uid_generator_type;
                uidable()
                {
                        this->m_uid = uidgen<uid_generator_type>::next_uid();
                }
                virtual ~uidable() {}

                uid_t uid() const { return this->m_uid; }
                void uid( uid_t uid ) { this->m_uid = uid; }

        private:
                uid_t m_uid;

        }; // uidable


        /**
           EXPERIMENTAL.

           A type for mapping objects to UIDs.

           This type takes no ownership of pointers, but they are
           non-const so that they can be used in wider scope.

           Parameterized on:

           T: type of object POINTER stored here.

           UIDType: the type of UI to be mapped to objects of type (T*).
        */
        template <typename T, typename UIDType = size_t>
        class uidmap
        {
        public:
                /**
                   This type's UID generator type. See uidgen for
                    details.
                */
                typedef UIDType value_type;

                /** The lookup key type for this class. */
                typedef T * key_type;

                /**
                   The map type used to map key_types to value_types.
                */
                typedef std::map<key_type,value_type> map_type;

                /**
                   The map type used to map value_types to key_types.
                */
                typedef std::map<value_type,key_type> reverse_map_type;
                
                /**
                   Maps key to the given uid. Failing to unmap(key)
                   the object when it is destroyed (or no longer
                   needed) effectively results in a memory leak.

                   Returns the key's uid.
                 */
                static value_type map( key_type key, value_type uid )
                {
                        if( value_type() == uid )
                        {
                                uid = generator_type()();
                        }
                        map()[key] = uid;
                        reverse_map()[uid] = key;
                        return uid;
                }

                /**
                   Removes key from this type's map.
                 */
                static void unmap( key_type key )
                {
                        typename map_type::const_iterator cit = map().find( key );
                        if( map().end() != cit )
                        {
                                typename reverse_map_type::const_iterator rit = reverse_map().find( (*cit).second );
                                map().erase( cit );
                                if( reverse_map().end() != rit )
                                {
                                        reverse_map().erase( rit );
                                }
                        }
                        return;
                }

                /**
                   Returns the UID for the given key.
                 */
                value_type uid( key_type key ) const
                {
                        typename map_type::const_iterator cit = map().find( key );
                        return ( map().end() == cit ) ? 0 : (*cit).second;
                }

                /**
                   Returns the key associated with val.
                 */
                key_type key( value_type val ) const
                {
                        typename reverse_map_type::const_iterator rit = reverse_map().find( val );
                        return ( reverse_map().end() == rit ) ? 0 : (*rit).second;
                }

        private:
                /**
                   Returns this type's map of key_type-to-value_type.
                 */
                static map_type & map()
                {
                        static map_type bob;
                        return bob;
                }

                /**
                   Returns this type's map of value_type-to-key_type.
                 */
                static reverse_map_type & reverse_map()
                {
                        static reverse_map_type bob;
                        return bob;
                }

        };
                
} // namespace uid


#endif // S11N_NET_ACME_UID_HPP_INCLUDED
