////////////////////////////////////////////////////////////////////////
// class_loader.cpp:
// some implementation code for class_loader.h and cllite.h
// 
// Author: stephan@s11n.net
// License: Public Domain
////////////////////////////////////////////////////////////////////////

#include <stdlib.h> // getenv()
#include <sstream>

#include "cl_config.hpp" // expecting: cl_CLASS_LOADER_DEFAULT_PATH
#include "path_finder.hpp" // path_finder class
#include "class_loader.hpp" // core cl framework
#include "cllite.hpp" // "lite" framework

#if ! WIN32
#ifdef HAVE_LTDL // prefer libltdl, but if it's not available...
#  include <ltdl.h>
#else           // use libdl instead:
#  include <dlfcn.h>
#endif // HAVE_LTDL
#else // WIN32:
#  include <windows.h> // LoadModule(). All WIN32 code is 100% untested.
#endif // !WIN32


#ifndef cl_CLASSLOADER_DEBUG_DEFAULT
    // lib's startup default debugging level for classloader_debug_level()
#  define cl_CLASSLOADER_DEBUG_DEFAULT 0
#endif

////////////////////////////////////////////////////////////////////////////////
// class_loader stuff:
namespace cl {

        int m_CL_debug_level = cl_CLASSLOADER_DEBUG_DEFAULT;
        int class_loader_debug_level() { return m_CL_debug_level; }
        void class_loader_debug_level( int dlevel ) { m_CL_debug_level = dlevel; }

}
// end class_loader
////////////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// cllite:

#define LITECERR if(cl::class_loader_debug_level() > 0) CERR

namespace cllite {

        bool cl_throws = false;
        void use_exceptions( bool b )
        {
                cl_throws = b;
        }
        bool use_exceptions()
        {
                return cl_throws;
        }

        std::string open_dll( const std::string & key ) throw(cl_exception)
        {
                LITECERR << "open_dll("<<key<<"): path="<<class_path().path_string() << "\n";

                std::string where = class_path().find( key );
                if( where.empty() )
                {
                        std::ostringstream os;
                        os << "No dll found for key '"<<key<<"' "
                           << "in path ["<<class_path().path_string()<<"]";
                        LITECERR << os.str() << std::endl;
                        if( use_exceptions() ) throw cl_exception( os.str() );
                        return where;
                }

                LITECERR << "found="<<where<<"\n";

                static bool donethat = false;
                if( !donethat && (donethat=true) )
                {
                        // About these dlopen(0) calls:
                        // They are here for client-side convenience, to open the main() app.
                        // In theory, that really isn't necessary any more, but that needs 
                        // to be tested.
#if ! WIN32
#  if HAVE_LTDL
                        lt_dlinit();
                        lt_dlopen( 0 );
#  else
                        dlopen( 0, RTLD_NOW | RTLD_GLOBAL );
#  endif
#else
                        // ??? anything?
#endif // WIN32
                }
                void * soh = 0;
#if ! WIN32
#  if HAVE_LTDL // libltdl:
                soh = lt_dlopen( where.c_str() );
#  else // libdl:
                soh = dlopen( where.c_str(), RTLD_NOW | RTLD_GLOBAL );
#  endif
                LITECERR << "so handle="<<std::hex<<soh<<". where="<<where<<"\n";
                if( 0 == soh )
                {
                        std::ostringstream os;
                        const char * dlerr = dll_error();
                        os << "Error opening DLL: " << (dlerr ? dlerr : "unknown error");
                        LITECERR << os.str() << std::endl;
                        if( use_exceptions() ) throw cl_exception( os.str() );
                        return "";
                }
                return where;
#else // WIN32:
                LoadModule( where.c_str(), NULL ); // WTF does LoadModule() really return?
                return where;
#endif // WIN32
        }

        const char * dll_error()
        {
#if ! WIN32
#  if HAVE_LTDL
                return lt_dlerror();
#  else
                // dlerror() return a non-const char *, and it's does
                // do not specify if the caller owns it, so this may
                // actually cause a leak. The moral of the story is:
                // DOCUMENT YOUR FUCKING CODE!!!
                return dlerror();
#  endif
#else
                return 0;
                // win32 equivalent (???)
#endif // !WIN32
        }


        classname_to_dllname::classname_to_dllname() : m_nssep("_")
        {}

        classname_to_dllname::classname_to_dllname( const std::string & ns_xlate )
                : m_nssep(ns_xlate)
        {}

        std::string
        classname_to_dllname::operator()( const std::string & key ) const
        {
                std::string xlated = key;
                std::string::size_type colon, pos = 0;
                if( std::string::npos != (colon = key.find( "<"))  )
                {
                        xlated = xlated.substr( 0, colon );
                }
                while( std::string::npos != (colon = xlated.find( ":", pos ))  )
                {
                        xlated.replace( colon, 2, this->m_nssep );
                        pos = colon + 2;
                }
                return xlated;
        }
        

        std::string no_namespace_name_transformer::operator()( const std::string & key ) const
        {
                std::string::size_type pos = key.find_last_of( ":" );
                if( std::string::npos == pos || pos == key.size()-1 ) return key;
                return key.substr( pos+1 );
        }


        /**
           Internal helper class for setting the shared classloader
           path. Initializes the  given path_finder  using the global
           classloader path. Intended for use with phoenix<>.
         */
        struct class_path_initer
        {
                void operator()( cl::path_finder & path )
                {
#if WIN32
                        path.add_extension( ".dll" );
#else
                        path.add_extension( ".so" );
#endif
                        path.add_path( cl_CLASS_LOADER_DEFAULT_PATH );
                        const char * p = 0;
                        if( (p = getenv( "CLLITE_CLASSPATH" )) )
                        {
                                path.add_path( p );
                        }
                        if( (p = getenv( "LD_LIBRARY_PATH" )) )
                        {
                                path.add_path( p );
                        }
                }
        };



        /**
           Returns this library's class search-path. It is up to specific
           classloaders to use or ignore this path: it is a
           suggestion, and not a rule. Clients are free to modify
           the returned object.
        */
        cl::path_finder & class_path()
        {
                return phoenix::phoenix<
                        cl::path_finder,
                        cllite::sharing_context,
                        class_path_initer
                        >::instance();
        }



} // namespace cllite

