#ifndef cl_CLASSLOADER_H_INCLUDED
#define cl_CLASSLOADER_H_INCLUDED 1
////////////////////////////////////////////////////////////////////////////////
// class_loader.h:
// a factory/template-based classloader framework
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
////////////////////////////////////////////////////////////////////////////////

#include <string>
#include <iostream>
#include <sstream>
#include <fstream>
#include <cassert>
#include <map>
#include <algorithm>


#include <s11n.net/phoenix/phoenix.hpp> // shared object creator/manager
#include "instantiator.hpp" // the underlying static classloader
#include "cl_debuggering_macros.hpp" // COUT/CERR


////////////////////////////////////////////////////////////////////////////////
#ifndef cl_CLASSLOADER_DEBUG
   // If this is 0 then NO debugging output is emitted, regardless
   // of any preference set in class_loader_debug_level().
   // Setting it to one only means that debugging is enabled as long
   // as a preference has been set via class_loader_debug_level().
#  define cl_CLASSLOADER_DEBUG 1
#endif
#ifndef cl_CLASSLOADER_UNCONDITIONAL_DEBUG
#  define cl_CLASSLOADER_UNCONDITIONAL_DEBUG 0
    // ^^^^^ if 1 it will bypass the class_loader_debug_level() check
#endif

////////////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////////////
#if cl_CLASSLOADER_DEBUG || cl_CLASSLOADER_UNCONDITIONAL_DEBUG
#  include <typeinfo>
#  define cl_CL_TYPENAME typeid((ThisType *)NULL).name()
#  define cl_CL_DEBUG if( cl_CLASSLOADER_UNCONDITIONAL_DEBUG || cl_CLASSLOADER_DEBUG ) if ( cl_CLASSLOADER_UNCONDITIONAL_DEBUG || (0 != cl::class_loader_debug_level()) ) CERR << cl_CL_TYPENAME <<" "
#else
#  define cl_CL_TYPENAME ""
#  define cl_CL_DEBUG if( 0 ) CERR
#endif
////////////////////////////////////////////////////////////////////////////////



/**
   namespace cl is home to s11n's class_loader sub-project, which
   provides a non-intrusive classloader.
*/
namespace cl
{
        /**
           Deletes an object passed to it.

           This type accepts non-pointer types for "destruction." This
           is a non-operation, and is supported to allow other
           template code to generically free objects without needing
           to know if they are pointers. This allows some
           formerly-non-consolidatable reference-vs-pointer-type code to
           share a single implementation, as "delete(myobj)" is not
           valid for non-pointer types, but object_deleter()(myobj)
           is.
        */
        struct object_deleter
        {
                /**
                   Deletes t.
                */
                template <typename T>
                void operator()( T * t ) const
                {
                        // ^^^ achtung: if we use (const T *) then
                        // the reference overload is always called!

                        // i don't fully understand why
                        // delete( const T * ) is legal,
                        // considering that it triggers a dtor,
                        // and dtors are non-const.
                        delete( t );
                }
                /**
                   Does nothing: is here to allow some
                   reference-vs-pointer-type transparency.
                */
                template <typename T>
                void operator()( const T & t ) const
                {}
        };

        /**
           A functor allowing pairs of PODs and pointers to be
           mixed together in any combination and be deallocated in a
           uniform way. Admitedly, it's a waste of time for value types,
           but the point is a single functor which can work for all
           pair types.

           sample:

           std::for_each( map.begin(), map.end(), pair_entry_deallocator() );

        */

        struct pair_entry_deallocator
        {
                /**
                   Deallocates p.first and p.second. PairType must not
                   be a pointer type.
                */
                template <typename PairType>
                void operator()( PairType & p ) const
                {
                        object_deleter od;
                        od( p.first );
                        od( p.second );
                }
        };


        /**
           Utility function to help keep an instance count of T.  The
           returned reference should ONLY be modified from ctors and
           dtors of T.
        */
        template <typename T>
        size_t & inst_count() { static size_t c = 0; return c; }


        /**
           Sets the framework-wide debug level. This is shared by
           all class_loaders, regardless of their templatized types.

           Currently only two values are supported: zero (no
           debugging output) and non-zero (all debugging output).

           Note that this uses a compile-time default, which means
           that if compiled without debugging, some debug messages
           will not show up because they happen pre-main(),
           before a client can set the debug level.
        */
        void class_loader_debug_level( int dlevel );

        /**
           Returns the framework-wide debug level.
        */
        int class_loader_debug_level();

	/**
           A basic templates-based classloader implementation,
           intended to work in conjunction with cl::instantiator
           for loading classes via registered object factories.

           Templatized on:

           BaseType - this is the type to which all loaded classes
           should be cast. This is analogous to libfun's
           fun::LoadableClass.

           KeyType - the key type to be used to search for
           classes. Historically this is a string, but any comparable
           type should work. It is important that KeyType support
           less-than ordering.

           UseSharedInstances - determines the ownership policy for
           pointers returned from load(). See load() and the
           library manual for full details. This parameter should
           be left as-is for most use-cases. Note that all
           factories are shared amongst class_loader<BaseType>'s, regardless
           of their sharing policy, so it is a bad idea to mix
           classloaders of the same BaseType with different
           sharing policies, as pointer ownership then becomes confusing.

           To make any given class classloadable by this class, call
           this macro one time (preferably from the class' header file):

<pre>
cl_CLASSLOADER_REGISTER(BaseType,SubType);
cl_CLASSLOADER_REGISTER(SubType,SubType); // optional, and sometimes useful
</pre>

	BaseType need not be ClassLoadable, but SubType must derive
	from (or be) BaseType.  The exact meanings of the parameters
	are discussed at length in the documentation for that macro, in
	the class_loader.h header file and in the libs11n manual.

        As a side-effect of it's template-ness, usage of this class is
        compile-time type-safe without the use of any casts.


        See cllite.h for a simplified client-side interface for this
        class, and lite.cpp for demonstration code.

        */
        template < class BaseType, class KeyType = std::string, bool UseSharedInstances = false >
        class class_loader
	{
        public:
                /**
                   If this is true then calls to load(key) will always
                   return the same instance for the same key. This
                   *might* not apply to calls made post-main(), i.e.,
                   it might return different objects after main()
                   exits, but such calls are dubious, anyway.
                */
                static const bool use_shared_instances = UseSharedInstances;

                /**
                   value_type is the base-most class type to be used
                   for loading.

                   Contrary to usage in some std::-namespace classes,
                   value_type is not a pointer type, even though this
                   class deals only with pointers to value_type
                   objects. This is intended to ease the use of
                   value_type in many common contexts.
                */
		typedef BaseType value_type;

                /**
                   The key type used for class lookups. The default is
                   std::string.
                */
		typedef KeyType key_type;
                
                
                /** Convenience typedef. It's only public for documentation reasons. */
		typedef class_loader < value_type, key_type, use_shared_instances > ThisType;
                /**
                   factory_type is a function pointer which takes no
                   arguments and returns a value_type pointer.
                   
                   todo: add proper functor support.
                */
  		typedef value_type *( *factory_type ) ();	// todo: investigate utility of: BaseType *(*factory_type)( key_type )
// 		typedef std::unary_function<void,value_type *> factory_type;

                class_loader()
		{
                         ++inst_count<ThisType>();
                         //cl_CL_DEBUG << "instance #" << inst_count<ThisType>()<<std::endl;
		}


		virtual ~class_loader()
		{
                        if( use_shared_instances )
                        { // might be optimized away by the compiler :)
                                if( 0 == --inst_count<ThisType>() )
                                {
                                        cl_CL_DEBUG << "Cleaning up shared objects..." << std::endl;
                                        shared_inst_map & bob = shared_objects();
                                        std::for_each( bob.begin(), bob.end(), pair_entry_deallocator() );
                                }
                        }
		}

                /**
                   Functionally identical to load(key).

                   This is for use i making this class usable as a
                   functor, to clients can easily replace classloaders
                   in contexts which use functors to load classes.
                */
                value_type * operator()( const key_type & key )
                {
                        return this->load( key );
                }

                /**
                   Returns a reference to a shared instance of this
                   classloader type.
                */
                static ThisType & shared()
                {
                        return phoenix::phoenix<ThisType>::instance();
                }

		/**
                   Tries to instantiate an object using a factory
                   which has been mapped to the given key. Returns a
                   non-NULL pointer on success and NULL on error. The
                   caller takes ownership of the returned
                   pointer... UNLESS:

                   If ThisType::use_shared_instances is true then this
                   function will always return the same object for any
                   given key, and the client <b>DOES NOT</b> take
                   ownership of the returned pointer! The objects
                   will be deleted when the last class_loader of this type
                   destructs, almost certainly post-main().


                   Subclasses should override this function to
                   customize classloading behaviour.

                   The default implementation loads classes which
                   have been registered via:

                   - ThisType::register_factory()

                   - ThisType::register_subtype()

                   (Remember, the CLASSLOADER_REGISTER macro calls
                   those for you.)
                */
		virtual value_type * load( const key_type & key ) const
		{
                        typename ThisType::alias_map::const_iterator alit = aliases().find( key );

                        key_type realkey = (aliases().end() == alit ) ? key : (*alit).second;
                        cl_CL_DEBUG << ": load("<<realkey<<")"<<std::endl;

                        if( ThisType::use_shared_instances )
                        { // the compiler can hopefully optimize this out in most cases.
                                value_type * ch = NULL;
                                shared_inst_map & bob = shared_objects();
                                typename shared_inst_map::iterator it = bob.find( realkey );
                                if( bob.end() == it )
                                {
                                        if( (ch = ThisType::instantiator_type::instantiate( realkey )) )
                                        { // ^^^^ yes, that's an assignment, not comparison
                                                bob[realkey] = ch;
                                                // don't insert NULLs because it breaks some use-cases.
                                        }
                                }
                                else
                                {
                                        ch = (*it).second;
                                }
                                return ch;
                        }

                        return ThisType::instantiator_type::instantiate( realkey );
		}

                /**
                   Convenience static version of load( key ), which
                   uses the class_loader returned by shared().
                   
                   Avoid using this, it will directly use
                   class_loader&lt;&gt;, bypassing any subclass-implemented
                   loading improvements (like DLL loading).
                **/
                static value_type * load_class( const key_type & key )
                {

                        return shared().load( key );
                }

                /**
                   Returns true if the given key is registered. This is sometimes useful
                   for checking whether a factory needs to be re-registered, which
                   is sometimes necessary post-main(), when the internal map gets hosed
                   before clients are done using it.
                */
                static bool is_registered( const key_type & key )
                {
                        return instantiator_type::is_registered( key );
                }

		/**

                register_factory() associates key with a factory
                function.  If fp == 0 then a default factory is
                used (normally this behaviour is just fine).

                register_factory() probably should not be static, but
                i'm not fully convinced that a non-static
                implementation is useful, and a static implementation
                eases use of this class. Since classloaders
                must(?) all use the same definitions for all like-types,
                anyway, it seems like a moot point. In any case,
                static appears to ease client usage, so static it is.
                */
		static void register_factory( const key_type & key, factory_type fp = 0 )
		{
                        cl_CL_DEBUG << "register_factory("<<key<<")"<<std::endl;
			instantiator_type::register_factory( key, fp );
		}
                

                /**
                   Convenience function which registers a factory
                   creating SubT pointers but returning value_type
                   pointers. A factory of 0 means to use a default
                   factory, generated by this library.

                   ACHTUNG: gcc 3.3.1 often gives parse errors when compiling
                   calls to this function, for reasons i don't understand:

                   e.g.:
<pre>
                classloader<BaseType,KeyType>().register_subtype<SubType>( classname,
                                                                  object_factory<BaseType,SubType>::new_instance
                                                                  );
</pre>

Gives me:
<pre>
   s11n::classloader_register(const std::string&)':
../../include/s11n/s11n_core.h:112: error: parse error before `>' token

</pre>

IMO it's a compiler bug, but i don't know for sure. :/


                */
                template <typename SubT>
		static void register_subtype( const key_type & key, factory_type fp = 0 )
		{
                        cl_CL_DEBUG << "register_subtype("<<key<<")" <<std::endl;
			register_factory( key, fp  ? fp : object_factory<value_type,SubT>::new_instance  );
		}

                
                /**
                  Aliases alias as equivalent to classname.
                  Calls to load(alias) will work exactly as if
                  load(classname) was called.

                **/
                static void alias( const key_type & alias, const key_type & classname )
                {
                        cl_CL_DEBUG << " alias("<<alias<<","<<classname<<")" <<std::endl;
                        ThisType::aliases()[alias] = classname;
                }



                /**
                   shared_inst_map is only used when
                   <code>use_shared_instances</code> is true. It holds
                   the shared classloaded objects.
                */
                typedef std::map<key_type,value_type *> shared_inst_map;

                /**
                   If <tt>use_shared_instances</tt> is true this
                   stores the classloaded shared objects, otherwise it
                   holds nothing.

                   When called post-main() this function may return an
                   empty map.

                   This list is provided in the public interface
                   primarily so clients can do things like serialize
                   the list at application shutdown, provided
                   value_type supports such a feature.
                */
                static shared_inst_map & shared_objects()
                {
                        return phoenix::phoenix<shared_inst_map,ThisType>::instance();
                }


                /**
                   Populates target list with the names of all
                   registered classes. ListT must support:

                   push_back( key_type )

                   Note that all class_loader instances with the same
                   value_type and key_type share a common list.
                */
                template <typename ListT>
                static void class_list( ListT & target )
                {
                        typedef typename instantiator_type::object_factory_map MT;
                        MT & m = instantiator_type::factory_map();
                        typename MT::const_iterator cit = m.begin(), cet = m.end();
                        for( ; cet != cit; ++cit )
                        {
                                target.push_back( (*cit).first );
                        }
                }


                /** Used to store alias-to-classname mappings. */
                typedef std::map<key_type,key_type> alias_map;

                /**
                   Returns the map of classname aliases, keyed on aliases,
                   with their expansions as their values.

                   The returned map is guaranteed to be post-main()
                   safe in the sense that it will never return a dead
                   reference. It is not post-main() safe in the sense
                   that the aliases map will be depopulated at some
                   arbitrary point post-main() and the returned map is
                   not guaranteed to be populated after that point.
                 */
                static alias_map & aliases()
                {
                        return phoenix::phoenix<alias_map,ThisType>::instance();
                }


        private:

                /**
                   Factories registered with this instantiator_type are
                   inherently loadable by this class.

                   The tasks of storing, looking up and calling
                   factories is delegated to this type.
                */
		typedef cl::instantiator < BaseType, KeyType > instantiator_type;

              

	}; // class_loader





} // namespace


#ifndef CLASSLOADER_REGISTERER_INCLUDED
#define CLASSLOADER_REGISTERER_INCLUDED 1
// this guard is necessary when compiling different-namespaced
// versions of this code in the same compilation unit.
namespace { // anonymous ns, important for linking reasons.
        /**
           Internal helper class used for registering classes. The
           whole point of this class is to create a unique (per
           compilation unit) variable which we can assign a
           meaningless value to. We accomplish uniqueness via the
           anonymous namespace and template parameters.
        */
        template <class BaseT,class SubT = BaseT, class KeyT = std::string >
        struct classloader_reg_placeholder
        {
                /**
                   A no-meaning placeholder.
                 */
                static bool reg;
                /**
                   Maintenance note: reg is initialized via
                   calls to the CLASSLOADER_REGISTER macro.
                */
        };
} // namespace
#endif //CLASSLOADER_REGISTERER_INCLUDED

/**
   Call this macro one time to make your classes loadable by all
   class_loader<BaseT> instances.

   Parameters:

   - BaseT, the base-most class type. i.e., that which will be used by
   clients when they call:

<pre>
BaseType * foo = class_loader<BaseType>( "mynamespace::SubtypeOfBaseT" );
</pre>

  The library documentation goes into much more detail about
  this.

   - SubT, an implementation class derived from BaseT.

   Example:

   <pre>
   cl_CLASSLOADER_REGISTER(mynamespace::MyDocumentBase,mynamespace::MyDocument);
   </pre>

   i'd like to call CLASS_NAME(SubT) from this macro, but doing so
   causes template specialization collisions when CLASSLOADER_REGISTER
   is called, e.g., like so:

<pre>
   cl_CLASSLOADER_REGISTER(BaseT,SubT);
   cl_CLASSLOADER_REGISTER(SubT,SubT);
</pre>

   This is valid usage, however, and necessary in some client cases,
   so i can't kill it's usage with an automatic CLASS_NAME() :(.

   See cl_CLASSLOADER_REGISTER[1-5] if you want to customize the
   registration behaviour or register one class with multiple KeyT's.
*/


////////////////////////////////////////////////////////////////////////
// cl_CLASSLOADER_REGISTER[1-5]:
// Pneumonic for remembering which macro to call: the suffix is the
// number of args to pass.
// Variants 1-3 are only known to work for (KeyT = std::string).
//
// Example:
// 	cl_CLASSLOADER_REGISTER4(BaseType,SubType,double,42.42)
// registers SubType via:
//      class_loader<BaseType,double>::register_subtype<SubType>( 42.42 ).
//
////////////////////////////////////////////////////////////////////////

#define cl_CLASSLOADER_REGISTER(BaseT,SubT) cl_CLASSLOADER_REGISTER2(BaseT,SubT)
#define cl_CLASSLOADER_REGISTER1(BaseT) cl_CLASSLOADER_REGISTER2(BaseT,BaseT)
#define cl_CLASSLOADER_REGISTER2(BaseT,SubT) cl_CLASSLOADER_REGISTER3(BaseT,SubT,std::string)
// #3 is only useful for KeyT == std::string. It is here only for consistency.
#define cl_CLASSLOADER_REGISTER3(BaseT,SubT,KeyT)  cl_CLASSLOADER_REGISTER4(BaseT,SubT,KeyT,# SubT) 
#define cl_CLASSLOADER_REGISTER4(BaseT,SubT,KeyT,ThekeY)  \
	namespace { \
		bool classloader_reg_placeholder< BaseT, SubT, KeyT >::reg = \
 		(cl::class_loader< BaseT, KeyT >::register_subtype< SubT >( ThekeY ),true); \
	}

/************************************************************************
Documentation for the cl_CLASSLOADER_ABSTRACT_BASE()
family of macros.

These are a way to registering abstract classes as base types. The root
of the problem is that default factory object uses 'new BaseT', which
will not work if BaseT is abstract (it fails at compile time).

The current implementation has a maintenance hazard: as soon as
class_loader stops using object_factory<> as it's default factory then
this code will need to adjust to suit.

Important notes:

- generated helper code does not use an anonymous namespace, and is
therfore more subject to ODR collisions than the CLASSSLOADER_REGISTER()
macros are.

- These macros may NOT be used in conjunction with the CLASSSLOADER_REGISTER()
macros: these do effectively the same thing.

************************************************************************/

#define cl_CLASSLOADER_ABSTRACT_BASE(BaseT) \
	cl_CLASSLOADER_ABSTRACT_BASE1(BaseT)

#define cl_CLASSLOADER_ABSTRACT_BASE1(BaseT) \
	cl_CLASSLOADER_ABSTRACT_BASE2(BaseT,std::string)

#define cl_CLASSLOADER_ABSTRACT_BASE2(BaseT,KeyT) \
	cl_CLASSLOADER_ABSTRACT_BASE3(BaseT,KeyT,# BaseT)

#define cl_CLASSLOADER_ABSTRACT_BASE3(BaseT,KeyT,ThekeY)  \
	namespace cl { \
		template <> struct object_factory< BaseT > { \
		static BaseT * new_instance() { return 0; } \
        }; } \
	cl_CLASSLOADER_REGISTER4(BaseT,BaseT,KeyT,ThekeY);



/************************************************************************
cl_CLASSLOADER_ALIAS(BaseT,String,Alias,UniqueID)


Note that using this macro is significantly more wasteful than simply
calling class_loader<BaseT>::alias(Alias,String) [achtung: note reversed
order of the args in the function and macro. Sorry.].

BaseT must be registered with the classloader already.

Alias is mapped as being functionally identical to String for purposes
of class_loader<BaseT>::load().

UniqueID is a kludge to get around ODR violations when applying
multiple aliases for a single BaseT. Pass a unique number for each
alias of BaseT.

************************************************************************/

#ifndef CLASSLOADER_ALIASER_INCLUDED
#define CLASSLOADER_ALIASER_INCLUDED 1
// this guard is necessary when compiling different-namespaced
// versions of this code in the same compilation unit.
namespace {

        /**
           Kludge to avoid ODR violations while aliasing classes via
           class_loader at initialization.
        */
        template <typename T, int ID>
        struct classloader_alias_registerer
        {
                static bool reg;
        };
}
#endif // CLASSLOADER_ALIASER_INCLUDED

#define cl_CLASSLOADER_ALIAS(BaseT,OriginalString,Alias,UniqueID) \
	namespace { \
		bool classloader_alias_registerer< BaseT, UniqueID >::reg = (\
			cl::class_loader< BaseT >::alias( Alias, OriginalString  ),\
			UniqueID \
		     ); \
	}

////////////////////////////////////////////////////////////////////////
// set up some shortcuts:
#ifndef cl_REGISTER
#  define cl_REGISTER(BaseT,SubT) \
	cl_CLASSLOADER_REGISTER2(BaseT,SubT)
#  define cl_REGISTER1(BaseT) \
	cl_CLASSLOADER_REGISTER1(BaseT)
#  define cl_REGISTER2(BaseT,SubT) \
	cl_CLASSLOADER_REGISTER2(BaseT,SubT)
#  define cl_REGISTER3(BaseT,SubT,KeyT) \
	cl_CLASSLOADER_REGISTER3(BaseT,SubT,KeyT) 
#  define cl_REGISTER4(BaseT,SubT,KeyT,ThekeY) \
	cl_CLASSLOADER_REGISTER4(BaseT,SubT,KeyT,ThekeY)
#  define cl_ALIAS(BaseT,OriginalString,Alias,UniqueID) \
        cl_CLASSLOADER_ALIAS(BaseT,OriginalString,Alias,UniqueID)

# define cl_ABSTRACT_BASE(Base) \
        cl_CLASSLOADER_ABSTRACT_BASE(Base)
# define cl_ABSTRACT_BASE1(Base) \
        cl_CLASSLOADER_ABSTRACT_BASE1(Base)
# define cl_ABSTRACT_BASE2(Base,KeyT)             \
        cl_CLASSLOADER_ABSTRACT_BASE2(Base,KeyT)
# define cl_ABSTRACT_BASE3(Base,KeyT,TheKey)     \
        cl_CLASSLOADER_ABSTRACT_BASE3(Base,KeyT,TheKey)
#endif
////////////////////////////////////////////////////////////////////////


#endif // cl_CLASSLOADER_H_INCLUDED
