#ifndef cllite_CLLITE_H_INCLUDED
#define cllite_CLLITE_H_INCLUDED 1

////////////////////////////////////////////////////////////////////////
// cllite.hpp:
//
// Defines a simplified front-end for the class_loader interfaces,
// encapsulating the most-commonly-used classloading features.
// It only supports string-keyed classloaders.
//
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain
////////////////////////////////////////////////////////////////////////

#include <string>

#include "cl_config.hpp"

#include "class_loader.hpp" // core classloader framework.
#include "cl_debuggering_macros.hpp" // CERR
#include "path_finder.hpp" // path_finder class

////////////////////////////////////////////////////////////////////////
// internal debuggering macro. Not for use by client code:
#define LITECERR if(cl::class_loader_debug_level() > 0) CERR
////////////////////////////////////////////////////////////////////////

////////////////////////////////////////////////////////////////////////////////
// Shorthand forms of the usual class-to-loader registration macros,
// for use by client code.
// These are all documented at length in the lib manual and class_loader.hpp.
#define CLLITE_BASE(BaseType) \
	cl_CLASSLOADER_REGISTER1(BaseType)
#define CLLITE_SUBTYPE(BaseType,SubType) \
	cl_CLASSLOADER_REGISTER2(BaseType,SubType)
#define CLLITE_ABSTRACT(BaseType) \
	cl_CLASSLOADER_ABSTRACT_BASE(BaseType)
////////////////////////////////////////////////////////////////////////////////



/**
   cllite is a 'lite' front-end to the class_loader framework.
   It's intention is to provide the mostly-used class_loader<> functionality
   under one roof, and simplify it's interface.

   As of the introduction of cllite, the former dll_loader class is
   deprecated in favour of the functions in this namespace. The brief
   reason is that class_loader and dll_loader are not 100%
   usage-compatible, often causing some additional work in client
   code. The functions in this namespace provide more flexible DLL
   support than dll_loader without having to implement a class_loader
   subtype to do it. This code also demonstrates how class_loader can
   transparently work with arbitrary client-side DLL-opening code (reminder:
   class_loader requires NO special symbols in a DLL, as conventional
   classloaders do).
*/
namespace cllite
{

        /**
           An exception type, optionally used by cllite.
        */
        struct cl_exception
        {
                cl_exception( const std::string & what ) : m_what(what) {}
                std::string what() const { return this->m_what; }
        private:
                std::string m_what;
        };

        /**
           Sets cllite's exception-throwing policy. By default it is
           set to false.
        */
        void use_exceptions( bool );

        /**
           Returns cllite's exception-throwing policy. If true then
           open_dll() will throw a cl_exception() on an error, otherwise
           cllite will throw no exceptions.
        */
        bool use_exceptions();


        /**
           The lookup key type used by classloaders in cllite.  Note
           that non-string types do not generically make sense in this
           context because DLL-awareness only works for strings. Why? 
           Consider: how do we load a DLL using an arbitrary
           non-string lookup key without imposing some type of
           translation conventions on clients?
        */
        typedef std::string key_type;


        /**
           Internal marker class for use with phoenix::phoenix<>.
         */
        struct sharing_context {};

        /**
           Returns this library's class search-path. It is up to specific
           classloaders to use or ignore this path: it is a
           suggestion, and not a rule. Clients are free to modify
           the returned object.
        */
        cl::path_finder & class_path();

        /**
           A basic classname-to-DLL-name transformer.

           Intended to be used to transform namespaced/templated names
           into filename-friendly names.


        */
        struct classname_to_dllname
        {
                typedef std::string key_type;
                classname_to_dllname();

                /**
                   ns_xlate sets the namespace separator (::) replacement which
                   will happen when this functor is activated. 


                   e.g., passing "/" will cause
                   foo::bar::MyType to translate to foo/bar/MyType
                */
                explicit classname_to_dllname( const key_type & ns_xlate );

                /**
                   General conventions:

                   Should return a DLL-friendly name of the given key,
                   minus any extension: a set of platform-specific
                   extensions is set by class_path(). See the
                   classname_to_dllname class for a sample
                   implementation.

                   This implementation transforms class names such
                   that:

                   - By default namespace separators (::)
                   are replaced by, "_", but this can be changed via the ctor

                   - Any template parameters are simply removed.

                   e.g.:

                   std::list<foo> = std_list

                   MyType = MyType

                   foo::bar::MyType = foo_bar_MyType


                */
                key_type operator()( const key_type & key ) const;

        private:
                std::string m_nssep; // namespace separator replacement
        };


        /**
           A simple classname-to-dll-name transformer which simply
           strips off any namespace part of the class name. e.g.,
           foo::FooClass becomes FooClass.

        */
        struct no_namespace_name_transformer
        {
                /**
                   Returns key, stripped of anything up to and
                   including the last ":" character. If key contains
                   no ":" then it is returned as-is.
                */
                std::string operator()( const std::string & key ) const;

        };

        /**
           Tries to open a DLL named basename, which should be
           either an absolute path/filename or a partial filename,
           without path and extension parts.
           
           class_path() will be used to resolve the full file name.

           Returns the path to the DLL, if one was found and opened,
           or an empty string on error.

           If use_exceptions() is true then this function will never
           return an empty string: instead it will throw a
           cl_exception when there is an error. This support is not
           yet working on WIN32 compilers.

           Reminder: the classloader architecture is such that simply
           opening a DLL is enough to trigger any classloader
           registrations which might live in it.
        */
        std::string open_dll( const std::string & basename ) throw(cl_exception);

        /**
           Returns open_dll( trans(key) ).
        */
        template <typename NameTransformer>
        std::string open_dll( const std::string & key,
                              const NameTransformer & trans ) throw(cl_exception)
        {
                return open_dll( trans(key) );
        }


        /**
           Returns the same thing as the underlying dlerror()
           implementation. Just as dlerror() does, this
           function will return NULL if called twice in a
           row. That is, it's error code is only valid once,
           and then it will return NULL until another error
           happens.

           Note that the documentation for dlerror() does not specify
           whether the caller owns the returned pointer, thus this
           function may actually cause a leak. The pointer returned
           from *this* function is const, but that returned from
           dlerror() is non-const (presumably for historical reasons).
           GNU's lt_dlerror() also returns a const, and it's interface
           is API-compatible with dlXXXX() functions, so i must assume
           that this function behaves properly and that a call to
           dlerror() does not cause a leaking (char *).  If you are
           using the lt_dlXXXX()-based back-end then this is not an
           issue.
        */
        const char * dll_error();



        /**
           Returns the BaseType classloader, as selected by the
           classloader_selector&lt;BaseType&gt; class.

           Note that the returned object itself is not DLL-aware: use
           classload() to enable DLL lookups for missing classes.
        */
        template <typename BaseType>
        cl::class_loader<BaseType> &
        classloader()
        {
                return phoenix::phoenix<
                        cl::class_loader<BaseType>,
                        sharing_context >::instance();
        }


        /**
           Tries to load the class named 'key' from BaseType's classloader.
           If no such class is registered a DLL search is initiated to
           find the class, using a default classname-to-DLL-name translator.

           The caller owns the returned pointer, which may be 0
           (indicating the classloader could not find the DLL).

           See open_dll() for information about the exception this
           function may throw.
        */
        template <typename BaseType, typename NameXFormerT>
        BaseType * classload( const key_type & key, const NameXFormerT & xf ) throw(cl_exception)
        {

                cl::class_loader<BaseType> & cl =
                        classloader<BaseType>();
                BaseType * ret = cl.load( key );
                if( ret ) return ret;
                LITECERR << "cllite looking for a DLL for '"<<key<<"'\n";
                std::string xl =  xf(key);
                std::string where = open_dll( xl );
                if( where.empty() )
                {
                        LITECERR << "DLL lookup for '"<<xl<<"' failed :(\n";
                        return 0;
                }
                LITECERR << "Found DLL for '"<<xl<<"': " << where << "\n";
                return cl.load( key ); // try again!
        }                

        /**
           Calls classload( key, classname_to_dllname() ).
        */
        template <typename BaseType>
        BaseType * classload( const key_type & key ) throw(cl_exception)
        {
                return classload<BaseType>( key, classname_to_dllname() );
        }

        /**
           Similar to classloader(), but the object it returns serves
           shared instances of objects. For details on object sharing
           see the docs for the UseSharedInstances template parameter
           for the class_loader<> type.
        */
        template <typename BaseType>
        cl::class_loader<BaseType,key_type,true> &
        classloader_shared()
        {
                return phoenix::phoenix<
                        cl::class_loader<BaseType,key_type,true>,
                        sharing_context >::instance();
        }



        /**
           Similar to classload(), but objects returned by this function
           are NOT owned by the caller: the classloader cleans them up
           when it is destroyed (post-main()).

           See open_dll() for information about the exception this
           function may throw.
        */
        template <typename BaseType, typename NameXFormerT>
        BaseType * classload_shared( const key_type & key, const NameXFormerT & xf ) throw(cl_exception)
        {
                typedef cl::class_loader<BaseType,key_type,true> SharedCL;
                SharedCL & cl = classloader_shared<BaseType>();
                BaseType * ret = cl.load( key );
                if( ret ) return ret;
                LITECERR << "cllite looking for a DLL for '"<<key<<"'\n";
                if( open_dll( xf(key) ).empty() )
                {
                        return 0;
                }
                return cl.load( key ); // try again!
        }

        /**
           Calls classload_shared<BaseType>( key, classname_to_dllname() ).

           See open_dll() for information about the exception this
           function may throw.

        */
        template <typename BaseType>
        BaseType * classload_shared( const key_type & key ) throw(cl_exception)
        {
                return classload_shared<BaseType>( key, classname_to_dllname() );
        }

        /**
           Registers key with a factory returning (BaseType *).
        */
        template <typename BaseType>
        void register_factory( const key_type & key, BaseType * (*factory)() )
        {
                classloader<BaseType>().register_factory( key, factory );
        }

        /**
           Registers a default factory for creating (SubType *) returned
           as (BaseType *).

           The default factory requires that SubType have a default
           ctor.
         */
        template <typename BaseType, typename SubType>
        void register_factory( const key_type & key )
        {
                register_factory<BaseType>( key,
                                            cl::object_factory<BaseType,SubType>::new_instance
                                            );
        }
        
        /**
           Registers a default factory for BaseType.
        */
        template <typename BaseType>
        void register_base( const key_type & key )
        {
                register_factory<BaseType,BaseType>( key );
        }

        /**
           Registers BaseType as an abstract base class. Use this for
           abstract base types, so that they get a working factory
           (i.e., one which always returns 0, since 'new BaseType'
           will not work for abstract types).
        */
        template <typename BaseType>
        void register_abstract_base( const key_type & key )
        {
                register_factory( key,
                                  cl::object_factory<BaseType>::no_instance
                                  );
        }

        /**
           Aliases thealias to be functionally equivalent to
           isthesameas for BaseType's classloader.

           This feature is useful for, e.g., assigning "friendly" or
           configurable names for a given application element. e.g.,
           consider the option of registering "DefaultDocumentClass"
           as an alias for some subtype of Document. This allows
           down-stream code to use symbolic names for loading
           application-specific types, as opposed to having to know
           the types real names.
        */
        template <typename BaseType>
        void alias( const key_type & thealias, const key_type & isthesameas )
        {
                classloader<BaseType>().alias( thealias, isthesameas );
        }


        /**
           A functor which uses cllite::classload() to 
           create new objects.

           Parameterized on:

           BaseType: base-most type of classloadable type.

           UseSharedInstances: if true then classload_shared() is used
           to create new objects, otherwise classload() is used. This
           determines whether the caller owns objects returned by this
           class or not - see the docs of operator() for the details.

        */
        template <typename BaseType, bool UseSharedInstances = false>
        struct cllite_factory
        {
                /** Equivalent to BaseType. */
                typedef BaseType base_type;

                /**
                   If true then this object uses classload_shared(),
                   otherwise it uses classload().
                */
                static const bool shared_instances = UseSharedInstances;
                
                /**
                   Takes a class name as an argument. Returns a
                   pointer to a new BaseType, or 0 on error. If
                   shared_instances is false then the caller owns the
                   returned pointer, otherwise it is owned by the
                   classloader and should not be destroyed by the
                   client.
                */
                base_type * operator()( const std::string & classname ) const
                {
                        return shared_instances
                                ? cllite::classload_shared<base_type>( classname )
                                : cllite::classload<base_type>( classname );
                }
        };


}

#undef LITECERR
#endif // cllite_CLLITE_H_INCLUDED
