#ifndef cl_dll_DLL_UTIL_HPP_INCLUDED
#define cl_dll_DLL_UTIL_HPP_INCLUDED 1

#include "cllite.hpp"

namespace cllite {

/**
   The dll namespace encapsulates some utility code for working with
   DLLs using the cllite framework.
*/
namespace dll {
        /**
           A helper class intended to hold information about a DLLs to
           open via the cllite framework.

           Normally they are used indirectly by client code, via
           something like this:

           <pre>
std::list<cllite::dll::dlentry> dlllist;
if( 0 != cllite::dll::parse_dlentry_flags( argc -1 , argv+1, dlllist ) )
{
    try
    {
        if( 0 < cllite::dll::dlload( dlllist ) ) return ErrorDLL;
    }
    catch( const cllite::cl_exception & ex )
    {
        std::cerr << ex.what() << "\n";
        return ErrorDLL;
    }
}
</pre>

        If you include the s11n headers before including this header then
        your dlentry objects will be made Serializable.
        */
        struct dlentry
        {
                dlentry() : name(""), path(""), tolerant(false) {}
                explicit dlentry( const std::string & n ) : name(n), path(""), tolerant(false) {}
                dlentry( const std::string & n, bool t ) : name(n), path(""), tolerant(t) {}
                dlentry( const std::string & n, const std::string & p, bool t ) : name(n), path(p), tolerant(t) {}
                ~dlentry(){}

                /**
                   Expected to be a name which cllite::open_dll() can
                   use to find a DLL. Normally this is the base
                   filename part of the DLL's path, minus any
                   extension.
                */
                std::string name;

                /**
                   The path where this dlentry's DLL is supposed to
                   live. This is expected to be modified by algorithms
                   which open a DLL using a dlentry.
                */
                std::string path;
                /**
                   The tolerant flag is a hint to algorithms saying
                   whether a failure to load this dlentry should be
                   considered an error. Failure to load a tolarant
                   dlentry should not be considered an error, and it
                   may be ignored (perhaps silently, parhaps with a
                   warning).
                */
                bool tolerant;

        };

        /**

        On successful loads it updates each entry's path member.
        For finding a DLL an entry's path is used, if set,
        otherwise its name is used.

        If an entry is intolerant and it fails to load then
        a cllite::cl_exception will be thrown.

        Returns true on success and false on error.

        */
        bool dlload( dlentry & dle ) throw( cllite::cl_exception )
        {
                std::string dlpath;
                std::string dlarg = dle.path.empty() ? dle.name : dle.path;
                bool worked = false;
                try
                {
                        std::string path = cllite::open_dll( dlarg ); // *might* throw (depends on cl_use_exceptions()).
                        worked = ! path.empty();
                        if( ! worked && ! dle.tolerant )
                        {
                                throw cllite::cl_exception( "dlload(dlentry &) failed: " + dlarg );
                        }
                        dle.path = path;
                }
                catch( const cllite::cl_exception & ex )
                {
                        if( ! dle.tolerant ) throw;
                }
                return worked;
        }

        /**
           Tries to open the DLLs associated with the dlentry objects
           in the given list, which must be compatible with
           <code>std::list<dlentry></code>.

           For each entry in list it calls dlload(), which might
           throw(cllite::cl_exception).

           Returns the number of dlload() failures for tolerant
           dlentries. That is, it returns 0 on success, non-zero on a
           warning, and throws an exception on error.
        */
        template <typename ListType>
        size_t dlload( ListType & list ) throw( cllite::cl_exception )
        {
                typedef typename ListType::value_type DLE;
                typename ListType::iterator it = list.begin(), et = list.end();
                int fails = 0;
                for( ; et != it; ++it )
                {
                        if( ! dlload( (*it) ) ) // might throw
                        {
                                ++fails;
                        }
                }
                return fails;
        }

        /**
           Looks for flags in the given list of arguments and tries to
           create dlentry objects out of them.

           See dlload( ListType & ) for the requirements imposed on
           ListType.

           The flags parsed from argv by this function are:

           [-]-intolerantFlag NAME == an intolerant dlentry named NAME.

           [-]-tolerantFlag NAME == a tolerant dlentry named NAME.

           Note that it checks for both -FLAG and --FLAG, and both
           forms are equivalent.

           i.e., it can accept arguments passed in via the command-line:

           <pre>
myapp -dl somedll -foo this_is_ignored -DL anotherdll --whatever=also_ignored
//----^^^ an intolerant dlentry -------^^^ a tolerant dlentry
</pre>

           Returns the number of dlentry items parsed out.

           It does not modify argv.
        */
        template <typename ListType>
        size_t parse_dlentry_flags( int argc, char ** argv,
                                    ListType & target,
                                    const std::string intolerantFlag = "dl",
                                    const std::string tolerantFlag = "DL" )
        {
                std::string dlarg;
                std::string nflag = "-"+intolerantFlag;
                std::string tflag = "-"+tolerantFlag;
                size_t count = 0;
                for( int i = 0; i < (argc-1); i++ )
                {
                        dlarg = std::string( argv[i] );
                        if( nflag == dlarg || ("-"+nflag) == dlarg ) // intolerant dlentry
                        {
                                target.push_back( dlentry( std::string( argv[1+i] ), false ) );
                        }
                        else if( tflag == dlarg || ("-"+tflag) == dlarg ) // tolerant dlentry
                        {
                                target.push_back( dlentry( std::string( argv[1+i] ), true ) );
                        }
                        else
                        {
                                continue;
                        }
                        ++i;
                        ++count;
                }
                return count;
        }

} } // namespace cllite::dll

#endif // cl_dll_DLL_UTIL_HPP_INCLUDED

#ifndef cllite_DLENTRY_S11N_DEFINED
#  ifdef s11n_S11N_INCLUDED
#  define cllite_DLENTRY_S11N_DEFINED
// Set up the dlentry_s11n serialization proxy for dlentry...
namespace cllite { namespace dll {

        /** An s11n serialization proxy for dlentry. */
        struct dlentry_s11n
        {
                /** Serialization operator. */
                template <typename NodeT>
                bool operator()( NodeT & dest, const dlentry & src ) const
                {
                        typedef s11n::node_traits<NodeT> NTR;
                        NTR::class_name( dest, "cllite::dll::dlentry" );
#define SER(PROP) NTR::set( dest, # PROP, src.PROP );
                        SER(name);
                        SER(path);
                        SER(tolerant);
#undef SER
                        return true;
                }

                /** Deserialization operator. */
                template <typename NodeT>
                bool operator()( const NodeT & src, dlentry & dest ) const
                {
                        typedef s11n::node_traits<NodeT> NTR;
#define DESER(PROP) dest.PROP = NTR::get( src, # PROP, dest.PROP );
                        DESER(name);
                        DESER(tolerant);
                        DESER(path);
#undef DESER
                        return true;
                }

        }; // struct dlentry_s11n

} } // namespace cllite::dll
#  define S11N_TYPE cllite::dll::dlentry
#  define S11N_TYPE_NAME "cllite::dll::dlentry"
#  define S11N_SERIALIZE_FUNCTOR cllite::dll::dlentry_s11n
#  include <s11n.net/s11n/reg_serializable_traits.hpp>
#endif // s11n_S11N_INCLUDED
#endif // cllite_DLENTRY_S11N_DEFINED

