#ifndef sigslot_SIGSLOT_HPP_INCLUDED
#define sigslot_SIGSLOT_HPP_INCLUDED

#include <set>
#include <list>


#if !defined(SIGSLOT_DEFAULT_MT_POLICY)
#  if SIGSLOT_PURE_ISO
#    define SIGSLOT_DEFAULT_MT_POLICY single_threaded
#  elif SIGSLOT_USE_POSIX_THREADS
#    include <pthread.h>
#    define SIGSLOT_HAS_POSIX_THREADS 1
     ////////////////////////////////////////////////////////////////////////
     // #define SIGSLOT_PTHREAD_CLASS to a mutex class implemented in
     // terms of pthreads, and that type will be used when
     // SIGSLOT_USE_POSIX_THREADS is true.
#    ifndef SIGSLOT_PTHREAD_CLASS
#      define SIGSLOT_PTHREAD_CLASS pthread_global
#    endif
#    define SIGSLOT_DEFAULT_MT_POLICY SIGSLOT_PTHREAD_CLASS
#  elif defined(WIN32)
#    include <windows.h>
#    define SIGSLOT_HAS_WIN32_THREADS 1
#    define SIGSLOT_DEFAULT_MT_POLICY win32_multi_threaded_global
#  else
#    define SIGSLOT_DEFAULT_MT_POLICY single_threaded
#  endif
#endif // SIGSLOT_DEFAULT_MT_POLICY

#ifndef SIGSLOT_HAS_POSIX_THREADS
#  define SIGSLOT_HAS_POSIX_THREADS 0
#endif

#ifndef SIGSLOT_HAS_WIN32_THREADS
#  define SIGSLOT_HAS_WIN32_THREADS 0
#endif

////////////////////////////////////////////////////////////////////////
// Debug stuff...
// Remove the SSCERR calls in the code and you can remove this block:
#ifndef SIGSLOT_DEBUG
   // achtung: enabling debug output can cause hangs in
   // multi-threading modes!
#  define SIGSLOT_DEBUG 0
#endif
#include <iostream> // sorry!
#if SIGSLOT_DEBUG
#  define SSCERR std::cerr << __FILE__<<":"<<std::dec<<__LINE__<<": "
#else
#  define SSCERR if( 0 ) std::cerr
#endif
////////////////////////////////////////////////////////////////////////


////////////////////////////////////////////////////////////////////////
// SIGSLOT_EMIT and SIGSLOT_CONNECT are here to allow this lib and Qt
// to play together.  (In libqt 'emit' and 'connect' are a macros.)
// Set them to the desired name of your 'emit' and 'connect' functions.
#ifndef SIGSLOT_EMIT
#  define SIGSLOT_EMIT emit_signal
#endif
#ifndef SIGSLOT_CONNECT
#  define SIGSLOT_CONNECT connect_slot
#endif
//
////////////////////////////////////////////////////////////////////////


/**

sigslot.hpp: Signal/Slot classes

Written by Sarah Thompson (sarah@telergy.com) 2002.

License: Public domain. You are free to use this code however you
like, with the proviso that the author takes on no responsibility or
liability for any use.

========================================================================
Hacked and re-released under the s11n.net project by stephan beal
(stephan@s11n.net) September, 2004. That is: the features are Sarah's,
the bugs are mine.

Changes by stephan:

24 Sept 2004:

- Added slot_proxy_funcN proxy class to allow sending slots to
  global/static functions.


23 Sept 2004:

- Replaced function name 'emit' with macro SIGSLOT_EMIT and 'connect'
  with SIGSLOT_CONNECT, to allow clients to rename emit so that this
  lib and Qt can play together :/.
  (In Qt emit is a macro.) To get around this problem altogether in
  client code, never call signalN::emit(), but call signalN::operator()
  instead.

- Renamed the functions emit() and connect() to emit_signal() and
  connect_slot(). Renamed disconnect() to disconnect_slot(), for
  symetry.

- Added slot_proxyN, to allow proxying of slots to types which cannot
  inherit has_slots.


19 Sept 2004:

- Removed signal_baseN classes, as they effectively added
  nothing and provided no code re-use over having the
  code all in signalN.

- Removed the mt_policy from the majority of the interfaces.
  It's much prettier this way. TODO: allow setting of the global
  policy at runtime. Locking is still used, but at a lower level,
  still settable by the appropriate #define macros.

- Documentation cleanups.

- Added doxygen support to build tree.


18 Sept 2004:

- Added signal_base::free_connections(), to ease maintenace
  in subclasses.

- Significant refactoring to move duplicated functionality
  into parent classes where possible (i.e., where template
  typing will allow us to).

- Significantly changed the way the threading macros
  work.

- Changed the way locking works: each object now holds
  it's own mutex, instead of subclassing a mutex type.

- Included test.cpp test app in the source bundle.

- Updated the docs.


1-4 September, 2004:

- brought typename usage up to date with gcc 3.

- replaced the {signal,slot}[2..8]-related classes with the
  generatesigslots script.


========================================================================

QUICK DOCUMENTATION 
       
  (see also the full documentation at http://sigslot.sourceforge.net/)

ACHTUNG: most of these docs were written by Sarah, and any notes about
mutexes, threading policies and locking may not apply after my changes
(stephan).


#define switches, which may be set before including sigslot.hpp.

SIGSLOT_PURE_ISO:

Define this to force ISO C++ compliance. This also disables all of the
thread safety support on platforms where it is available.

SIGSLOT_USE_POSIX_THREADS:

Force use of Posix threads. Client must link with the pthreads lib, if
appropriate (when linking with gcc this is not needed).


PLATFORM NOTES:

Win32:

On Win32, the WIN32 symbol must be #defined. Most mainstream compilers
do this by default, but you may need to define it yourself if your
build environment is less standard. This causes the Win32 thread
support to be compiled in and used automatically.

Unix/Linux/BSD, etc.:
  
If want to use Posix threads, you need to #define
SIGSLOT_USE_POSIX_THREADS to 1 and (if necessary)
link with -lpthread. i believe linking with pthread
is unnecessary.

ISO C++:

If no threads-implementation-specific options are defined, or if
SIGSLOT_PURE_ISO is defined to 1, all multithreading support is turned
off, along with any code that might cause a pure ISO C++ environment
to complain. Before you ask, gcc -ansi -pedantic won't compile this
library, but gcc -ansi is fine. Pedantic mode seems to throw a lot of
errors that aren't really there. If you feel like investigating this,
please contact the author.


       
THREADING MODES:

single_threaded:

Your program is assumed to be single threaded from the point of view
of signal/slot usage (i.e. all objects using signals and slots are
created and destroyed from a single thread). Behaviour if objects are
destroyed concurrently is undefined (i.e. you'll get the occasional
segmentation fault/memory exception).


(win32,pthread)_global:

Your program is assumed to be multi threaded. Objects using signals
and slots can be safely created and destroyed from any thread, even
when connections exist. In multi_threaded_global mode, this is
achieved by a single global mutex (actually a critical section on
Windows because they are faster). This option uses less OS resources,
but results in more opportunities for contention, possibly resulting
in more context switches than are strictly necessary.


(win32,pthread)_local:

Behaviour in this mode is essentially the same as
multi_threaded_global, except that each signal, and each object that
inherits has_slots, all have their own mutex/critical section. In
practice, this means that mutex collisions (and hence context
switches) only happen if they are absolutely essential. However, on
some platforms, creating a lot of mutexes can slow down the whole OS,
so use this option with care.

*/

namespace sigslot {


        class single_threaded
        {
        public:
                single_threaded()
                {
                        ;
                }

                single_threaded( const single_threaded & )
                {
                        ;
                }

                virtual ~single_threaded()
                {
                        ;
                }

                virtual void lock()
                {
                        //SSCERR << "single_threaded::lock()\n";
                        ;
                }

                virtual void unlock()
                {
                        //SSCERR << "single_threaded::unlock()\n";
                        ;
                }
        };

#if SIGSLOT_HAS_WIN32_THREADS
        // The multi threading policies only get compiled in if they are enabled.
        class win32_global
        {
        public:
                win32_global()
                {
                        static bool isinitialised = false;

                        if(!isinitialised)
                        {
                                InitializeCriticalSection(get_critsec());
                                isinitialised = true;
                        }
                }

                win32_global(const win32_global&)
                {
                        ;
                }

                virtual ~win32_global()
                {
                        ;
                }

                virtual void lock()
                {
                        EnterCriticalSection(get_critsec());
                }

                virtual void unlock()
                {
                        LeaveCriticalSection(get_critsec());
                }

        private:
                CRITICAL_SECTION* get_critsec()
                {
                        static CRITICAL_SECTION g_critsec;
                        return &g_critsec;
                }
        };

        class win32_local
        {
        public:
                win32_local()
                {
                        InitializeCriticalSection(&m_critsec);
                }

                win32_local(const win32_local&)
                {
                        InitializeCriticalSection(&m_critsec);
                }

                virtual ~win32_local()
                {
                        DeleteCriticalSection(&m_critsec);
                }

                virtual void lock()
                {
                        EnterCriticalSection(&m_critsec);
                }

                virtual void unlock()
                {
                        LeaveCriticalSection(&m_critsec);
                }

        private:
                CRITICAL_SECTION m_critsec;
        };
#endif // SIGSLOT_HAS_WIN32_THREADS

#if SIGSLOT_HAS_POSIX_THREADS
        // The multi threading policies only get compiled in if they are enabled.

        /**
           Maintains a global mutex.
         */
        class pthread_global
        {
        public:
                pthread_global()
                {
                        pthread_mutex_init(get_mutex(), NULL);
                }

                pthread_global(const pthread_global&)
                {
                        ;
                }

                virtual ~pthread_global()
                {
                        ;
                }

                virtual void lock()
                {
                        pthread_mutex_lock(get_mutex());
                        SSCERR << "pthread_global::lock("<<std::hex<<get_mutex()<<")\n";
                }

                virtual void unlock()
                {
                        pthread_mutex_unlock(get_mutex());
                        SSCERR << "pthread_global::unlock("<<std::hex<<get_mutex()<<")\n";
                }

        private:
                pthread_mutex_t* get_mutex()
                {
                        static pthread_mutex_t g_mutex;
                        return &g_mutex;
                }
        };

        /**
           Maintains a mutex local to this object.
         */
        class pthread_local
        {
        public:
                pthread_local()
                {
                        pthread_mutex_init(&m_mutex, NULL);
                }

                pthread_local(const pthread_local&)
                {
                        pthread_mutex_init(&m_mutex, NULL);
                }

                virtual ~pthread_local()
                {
                        pthread_mutex_destroy(&m_mutex);
                }

                virtual void lock()
                {
                        pthread_mutex_lock(&m_mutex);
                        SSCERR << "pthread_local::lock("<<std::hex<<&m_mutex<<")\n";
                }

                virtual void unlock()
                {
                        pthread_mutex_unlock(&m_mutex);
                        SSCERR << "pthread_local::unlock("<<std::hex<<&m_mutex<<")\n";
                }

        private:
                pthread_mutex_t m_mutex;
        };
#endif // SIGSLOT_HAS_POSIX_THREADS

        typedef SIGSLOT_DEFAULT_MT_POLICY mt_policy;


        /**
           This class locks a mutex on construction and unlocks in on
           destruction.

           mt_policy must comply with the interface used by the
           classes single_threaded, pthread_local, pthread_global,
           etc.
        */
        template<class lock_policy>
        class lock_block
        {
        public:
                lock_policy *m_mutex;
                lock_block(lock_policy &mtx)
                        : m_mutex(&mtx)
                {
                        m_mutex->lock();
                }
                ~lock_block()
                {
                        m_mutex->unlock();
                }
        };

        class has_slots;
        typedef has_slots slotted_base;

        /**
           The base-most type for all signals.
        */
        class signal_base
        {
        public:
                virtual ~signal_base() {}


                /**
                   Returns this->slot_disconnect(pslot). This is the one
                   which clients should use, rather than slot_disconnect().
                */
                bool disconnect_slot(slotted_base* pslot) { return this->slot_disconnect(pslot); }

                /**
                 */
                virtual bool slot_duplicate(const slotted_base* poldslot, slotted_base* pnewslot) = 0;
        protected:
                virtual bool slot_disconnect(slotted_base* pslot) = 0;

                /**
                   When subclasses perform locked operations, they
                   should use this mutex.
                */
                mt_policy & mutex() { return this->m_mutex; }

                /**
                   Frees all entries in list, which must contain
                   (connection_baseX *) objects, and must not contain
                   duplicates. All connections to this object are
                   disconnected, deleted, and removed from list.

                   After calling this function list will be empty.

                   This function does not lock mutex(). (Why not?)
                */
                template <typename ListT>
                void free_connections( ListT & list)
                {
                        typename ListT::const_iterator it = list.begin();
                        typename ListT::const_iterator itEnd = list.end();
                        for( ; it != itEnd; ++it)
                        {
                                (*it)->getdest()->remove_signaler(this);
                                delete *it;
                        }
                        list.clear();
                }

                /**
                   SlottedT must be a has_slots type and list must
                   contain (connection_baseX *) objects.

                   The connection object in list matching pslot is
                   deleted from list.

                   Returns true if it disconnect an object, otherwise
                   false.

                   This function locks mutex().
                */
                template <typename ListT>
                bool disconnect_slot( slotted_base *pslot, ListT & list )
                {
                        lock_block<mt_policy> lock(this->mutex());
                        typename ListT::iterator it = list.begin();
                        typename ListT::iterator itEnd = list.end();
                        for( ; it != itEnd; ++it )
                        {
                                if((*it)->getdest() == pslot)
                                {
                                        delete *it;
                                        list.erase(it);
                                        pslot->remove_signaler(this);
                                        return true;
                                }
                        }
                        return false;
                }

                /**
                   If this function finds oldtarget in list, it
                   duplicates the connection, using newtarget as the
                   new target.

                   SlottedT must be a has_slots type and list must
                   contain (_connects_base *) objects.

                   Returns true on success, false on error.

                   This function does not lock mutex(). (Why not?)
                */
                template <typename SlottedT, typename SetT>
                bool duplicate_connection(const SlottedT* oldtarget, SlottedT* newtarget, SetT & list )
                {
                        typename SetT::iterator it = list.begin();
                        typename SetT::iterator itEnd = list.end();

                        bool ret = false;
                        for( ; it != itEnd; ++it)
                        {
                                if((*it)->getdest() == oldtarget)
                                {
                                        list.push_back((*it)->duplicate(newtarget));
                                        ret = true;
                                }
                        }
                        return ret;
                }

                /**
                   Duplicates all connections in src to tgt,
                   connecting them to this object.

                   If src and tgt are the same list, nothing happens.

                   SetT must contain (connection_baseX *) objects.

                   This function locks mutex().
                */
                template <typename SetT>
                void copy_connections(const SetT & src, SetT & tgt)
                {
                        if( &src == &tgt ) return;
                        lock_block<mt_policy> lock(this->mutex());
                        typename SetT::const_iterator it = src.begin();
                        typename SetT::const_iterator itEnd = src.end();
                        typedef typename SetT::value_type ConT;
                        ConT clone = 0;
                        for( ; it != itEnd; ++it )
                        {
                                clone = (*it)->clone();
                                if( ! clone )
                                {
                                        SSCERR << "signal @ " << std::hex<<this<<std::dec << " Failed cloning connection @ " << std::hex << (*it)<<"\n";
                                        continue;
                                }
//                                 (*it)->getdest()->add_signaler(this);
                                clone->getdest()->add_signaler( this );
                                tgt.push_back( clone );
                        }
                }
        private:
                mutable mt_policy m_mutex;
        };

        /**
           Base type for connections involving no-arg signals/slots.
        */
        class connection_base0// : public connection_base<mt_policy>
        {
        public:
                typedef connection_base0 this_type;
                virtual slotted_base * getdest() const = 0;
                virtual void SIGSLOT_EMIT() const = 0;
                /**
                   Copies this connection's targeting information.
                */
                virtual this_type * clone() const  = 0;
                /**
                   Similar to clone, but the clone connects to
                   pnewdest, which must have the same slot as whatever
                   is already connected to this object.
                */
                virtual this_type * duplicate(slotted_base* pnewdest) const = 0;
        };


        /**
           Base type for connections involving 1-arg signals/slots.
        */
        template<class arg1_type>
        class connection_base1// : public connection_base<mt_policy>
        {
        public:
                typedef connection_base1<arg1_type> this_type;
                virtual slotted_base* getdest() const = 0;
                virtual void SIGSLOT_EMIT(arg1_type) const = 0;
                /**
                   Copies this connection's targeting information.
                */
                virtual this_type * clone() const = 0;
                /**
                   Similar to clone, but the clone connects to
                   pnewdest, which must have the same slot as whatever
                   is already connected to this object.
                */
                virtual this_type * duplicate(slotted_base* pnewdest) const = 0;
        };



        /**
           Connection type for no-arg signals/slots.
        */
        template<class dest_type>
        class connection0 : public connection_base0
        {
        public:
                typedef connection_base0 base_type;
                typedef connection0<dest_type> this_type;

                typedef void (dest_type::* slot_func)();
                connection0() : m_pobject(0), m_pmemfun(0)
                {
                }

                virtual ~connection0() {}

                connection0(dest_type* pobject, slot_func pmemfun )
                        : m_pobject(pobject), m_pmemfun(pmemfun)
                {
                }

                virtual base_type * clone() const 
                {
                        return new this_type(*this);
                }

                virtual base_type * duplicate(slotted_base* pnewdest) const
                {
                        return new this_type(dynamic_cast<dest_type *>(pnewdest), m_pmemfun);
                }

                virtual void SIGSLOT_EMIT() const
                {
                        (m_pobject->*m_pmemfun)();
                }

                virtual slotted_base* getdest() const
                {
                        return m_pobject;
                }

        private:
                dest_type* m_pobject;
                slot_func m_pmemfun;
        };

        /**
           Connection type for 1-arg signals/slots.
        */
        template<class dest_type, class arg1_type>
        class connection1 : public connection_base1<arg1_type>
        {
        public:
                typedef connection_base1<arg1_type> base_type;
                typedef connection1<dest_type,arg1_type> this_type;
                typedef void (dest_type::*slot_func)(arg1_type);
                connection1() : m_pobject(0),m_pmemfun(0)
                {
                }

                connection1(dest_type* pobject, slot_func pmemfun )
                        : m_pobject(pobject), m_pmemfun(pmemfun)
                {
                }

                virtual ~connection1() {}

                virtual base_type * clone() const 
                {
                        return new this_type(*this);
                }

                virtual base_type * duplicate(slotted_base* pnewdest) const
                {
                        return new this_type(dynamic_cast<dest_type *>(pnewdest), m_pmemfun);
                }

                virtual void SIGSLOT_EMIT(arg1_type a1) const
                {
                        (m_pobject->*m_pmemfun)(a1);
                }

                virtual slotted_base* getdest() const
                {
                        return m_pobject;
                }

        private:
                dest_type* m_pobject;
                slot_func m_pmemfun;
        };


        /**
           Client classes which implement slots must publically
           subclass this type.
        */
        class has_slots
        {
        private:
                typedef std::set<signal_base *> sender_set;
                typedef sender_set::const_iterator const_iterator;
                typedef sender_set::iterator iterator;
                sender_set m_senders;
                sender_set & signalers() { return this->m_senders; }
                const sender_set & signalers() const { return this->m_senders; }
        public:

                has_slots()
                {
                        ;
                }

                has_slots(const has_slots& rhs)
                {
                        this->copy_signalers(rhs);
                } 

                has_slots & operator=(const has_slots& rhs)
                {
                        this->disconnect_all();
                        this->copy_signalers(rhs);
                        return *this;
                } 

                /**
                   For internal use.

                   Adds sender to this object's list of signal
                   senders. Ownership of sender is unchanged.

                   This function does not actually connect sender with
                   this object.
                */
                void add_signaler(signal_base* sender)
                {
                        this->signalers().insert(sender);
                }

                /**
                   For internal use.

                   Removes sender from this objects list of
                   signalers. Ownership of sender is unchanged.

                   This function does not actually disconnect sender
                   from this object.
                */
                void remove_signaler(signal_base* sender)
                {
                        this->signalers().erase(sender);
                }

                /**
                   Disconnects this object from all signals.
                */
                virtual ~has_slots()
                {
                        disconnect_all();
                }

        protected:

                /**
                   Copies signal connections from rhs.
                */
                void copy_signalers( const has_slots & rhs )
                {
                        if( &rhs == this ) return;
                        const_iterator it = rhs.signalers().begin();
                        const_iterator itEnd = rhs.signalers().end();
                        for( ; it != itEnd; ++it )
                        {
                                (*it)->slot_duplicate(&rhs, this);
                                this->signalers().insert(*it);
                        }
                }

                /**
                   Disconnects this object from all signals which have
                   attached themselves via add_signaler().
                */
                void disconnect_all()
                {
                        iterator it = this->signalers().begin();
                        iterator et = this->signalers().end();
                        for( ; et != it; ++it )
                        {
                                (*it)->disconnect_slot( this );
                        }
                        this->signalers().clear();
                }


        };


        /**
           A signal accepting no arguments.
        */
        class signal0 : public signal_base
        {
        public:
                typedef std::list<connection_base0 *>  connections_list;

                signal0()
                {
                        ;
                }

                signal0(const signal0& s)
                        : signal_base(s)
                {
                        this->copy_connections( s.connections(), this->connections() );
                }

                signal0 & operator=(const signal0& s)
                {
                        this->copy_connections( s.connections(), this->connections() );
                        return *this;
                }

                virtual ~signal0()
                {
                        this->disconnect_all();
                }

                void disconnect_all()
                {
                        lock_block<mt_policy> lock(this->mutex());
                        this->free_connections(this->connections());
                }


                template<class desttype>
                void SIGSLOT_CONNECT(desttype* pclass, void (desttype::*pmemfun)())
                {
                        lock_block<mt_policy> lock(this->mutex());
                        connection_base0* conn = 
                                new connection0<desttype>(pclass, pmemfun);
                        this->connections().push_back(conn);
                        pclass->add_signaler(this);
                }


                bool slot_duplicate(const slotted_base* oldtarget, slotted_base* newtarget)
                {
                        return this->duplicate_connection(oldtarget, newtarget,this->connections());
                }


                void SIGSLOT_EMIT() const
                {
                        //lock_block<mt_policy> lock(this);
                        connections_list::const_iterator it = this->connections().begin();
                        connections_list::const_iterator end = this->connections().end();

                        for( ; end != it; ++it )
                        {
                                (*it)->SIGSLOT_EMIT();
                        }
                }

                void operator()() const
                {
                        this->SIGSLOT_EMIT();
                }

	protected:
		connections_list & connections() { return this->m_connected_slots; }
		const connections_list & connections() const { return this->m_connected_slots; }
                /**
                   Removes any connection between this object and pslot.

                   Returns true if it disconnect an object, otherwise
                   false.
                */
                bool slot_disconnect(slotted_base* pslot)
                {
                        return this->disconnect_slot( pslot,
                                                      this->connections() );
                }
	private:
		connections_list m_connected_slots;

        };

        /**
           A signal accepting 1 argument.
        */
        template<class arg1_type>
        class signal1 : public signal_base
        {
        public:
                typedef std::list<connection_base1<arg1_type> *>  connections_list;

                signal1()
                {
                        ;
                }

                signal1(const signal1<arg1_type>& s)
                        : signal_base(s)
                {
                        ;
                }

                template<class dest_type>
                void SIGSLOT_CONNECT(dest_type* pclass, void (dest_type::*pmemfun)(arg1_type))
                {
                        SSCERR << std::hex<<this<<std::dec<<"->signal1::SIGSLOT_CONNECT("<<std::hex<<pclass<<")\n";
                        lock_block<mt_policy> lock(this->mutex());
                        connection_base1<arg1_type>* conn = 
                                new connection1<dest_type, arg1_type>(pclass, pmemfun);
                        this->connections().push_back(conn);
                        pclass->add_signaler(this);
                }

                void SIGSLOT_EMIT(arg1_type a1) const
                {
                        //lock_block<mt_policy> lock(this->mutex());
                        //SSCERR << std::hex<<this<<std::dec<<"->signal1::SIGSLOT_EMIT("<<a1<<")\n";
                        typename connections_list::const_iterator it = this->connections().begin();
                        typename connections_list::const_iterator itEnd = this->connections().end();
                        for( ; itEnd != it; ++it )
                        {
                                (*it)->SIGSLOT_EMIT(a1);
                        }
                }

                void operator()(arg1_type a1) const
                {
                        this->SIGSLOT_EMIT( a1 );
                }

                signal1 & operator=(const signal1& s)
                {
                        this->copy_connections( s.connections(), this->connections() );
                        return *this;
                }

                bool slot_duplicate(const slotted_base* oldtarget, slotted_base* newtarget)
                {
                        return this->duplicate_connection(oldtarget, newtarget,this->connections());
                }

                ~signal1()
                {
                        this->disconnect_all();
                }

                void disconnect_all()
                {
                        lock_block<mt_policy> lock(this->mutex());
                        this->free_connections(connections());
                }

	protected:
		connections_list & connections() { return this->m_connected_slots; }
		const connections_list & connections() const { return this->m_connected_slots; }
                bool slot_disconnect(slotted_base* pslot)
                {
                        return this->disconnect_slot( pslot,
                                                      this->connections() );
                }

	private:
		connections_list m_connected_slots;
        };




        /**
           Proxies slots for type T, which is assumed to not
           be able to inherit has_slots.

           Connect it to a signal like this:

<pre>
typedef slot_proxy1&lt;MyType,int&gt; ProxyT;
ProxyT proxy( myobj, &MyType::myfunc );
mysignal.connect_slot( &proxy, &ProxyT::proxy_slot );
</pre>

	Note that the proxy has no way of knowing when it's proxied
	object is destroyed, and thus that object must outlive the
	proxy. The proxy is a normal has_slots object, and all normal
        automatic disconnections apply to it.

        This type can be used to proxy calls to, e.g., QWidget types
        created via Qt Designer, which does not allow you to use
        multiple inheritence to inherit has_slots.
        */
        template <typename T>
        struct slot_proxy0 : public has_slots
        {
                typedef void (T::* slot_func)();
                T * obj;
                slot_func memfunc;
                /**
                   Creates a proxy object for t.f().
                */
                slot_proxy0(T&t, slot_func f) : obj(&t),memfunc(f){}


                /**
                   Sets this object to proxy t.f().
                */
                void proxy(T&t, slot_func f)
                {
                        this->obj = &t;
                        this->memfunc = f;
                }

                /**
                   Calls the object/function passed to the ctor
                   or set via proxy().
                 */
                void proxy_slot()
                {
                        ((this->obj)->*memfunc)();
                }
        };

        /**
           See slot_proxy0. This class is identical except that it
           proxies functions with 1 argument.
        */
        template <typename T, typename ArgT1>
        struct slot_proxy1 : public has_slots
        {
                typedef void (T::* slot_func)( ArgT1 );
                T * obj;
                slot_func memfunc;
                slot_proxy1(T&t, slot_func f) : obj(&t),memfunc(f){}
                void proxy_slot( ArgT1 a1 )
                {
                        ((this->obj)->*memfunc)( a1 );
                }
                void proxy(T&t, slot_func f)
                {
                        this->obj = &t;
                        this->memfunc = f;
                }

        };

        /**
           A helper class to proxy calls to global/static functions
	   taking no arguments.

           Use this object by instantiating one, passing it a pointer
           to a global or static function. Then connect this object
           to your signals as you would any other object. The global
           function will be called when the signal is emitted.
        */
        struct slot_proxy_func0 : public has_slots
        {
                typedef void (*slot_func)();
                slot_func func;
                /**
                   Creates a proxy object for f.
                */
                slot_proxy_func0(slot_func f) : func(f){}

                /**
                   Calls the proxied function.
                */
                void proxy()
                {
                        return this->func();
                }
                /**
                   Sets this object's proxied function to f.
                */
                void proxy_func(slot_func f)
                {
                        this->func = f;
                }
        };

        /**
           A helper class to proxy calls to global/static functions
	   taking 1 argument.

           See slot_proxy_func0 for the documentation.
        */
        template <typename arg_t1>
        struct slot_proxy_func1 : public has_slots
        {
                typedef void (*slot_func)( arg_t1 );
                slot_func func;
                /**
                   Creates a proxy object for f.
                */
                slot_proxy_func1(slot_func f) : func(f){}

                /**
                   Passes on the arguments to the proxied function.
                */
                void proxy( arg_t1 a1 )
                {
                        return func( a1 );
                }
                /**
                   Sets this object's proxied function to f.
                */
                void proxy_func(slot_func f)
                {
                        this->func = f;
                }
        };


} // namespace sigslot


#include "sigslot.generated.hpp" // for signals with >1 parameters

#endif // sigslot_SIGSLOT_HPP_INCLUDED

