#ifndef s11n_net_str_STR_HPP_INCLUDED
#define s11n_net_str_STR_HPP_INCLUDED 1

#include <string>
#include <sstream>
#include <map>

#include <s11n.net/stringutil/string_util.hpp> // translate_entities()
#include <s11n.net/phoenix/phoenix.hpp> // phoenix<> class

/**
   The str namespace offer is TINY yet powerful interface
   for lexical casting, entity-translation, and [un]escaping strings.

   General Conventions:

   - ValueT templatized types must be i/ostreamable.




*/
namespace str {



        /**
           The functions in the Private namespace should not be used
           by client code.
        */
        namespace Private
        {


                template <typename value_type>
                value_type from_string( const std::string & str, const value_type & errorVal )
                {
                        std::istringstream is( str );
                        if ( !is )
                                return errorVal;
                        value_type foo = value_type();
                        if ( is >> foo )
                                return foo;
                        return errorVal;
                }

                /**
                   A quasi-bogus overload to avoid the
                   first-token-only problem of from_string<>().
                */
                inline std::string from_string( const std::string & str, const std::string & errorVal )
                {
                        return str;
                }

                /**
                   A quasi-bogus overload to avoid the
                   first-token-only problem of from_string<>().
                */
                inline std::string from_string( const char *str, const char *errorVal )
                {
                        return str;
                }


                /**
                   Returns a string representation of the given object, which must be ostreamble.
                */
                template <typename value_type>
                std::string to_string( const value_type & obj )
                {
                        std::ostringstream os;
                        os << std::fixed;
                        os << obj;
                        return os.str();
                }

                /**
                   Overloaded for strings-via-streams reasons.
                */
                inline std::string to_string( const char *obj )
                {
                        return obj;
                }

                /**
                   Overloaded for strings-via-streams reasons.
                */
                inline std::string to_string( const std::string & obj )
                {
                        return obj;
                }



        } // end Private namespace
        
        /**
           Lexically casts v to a string.
        */
        template <typename ValueT>
        std::string to( const ValueT & v )
        {
                return Private::to_string( v );
        }

        /**
           Lexically casts v to a ValueT, or returns dflt if
           conversion fails.
        */
        template <typename ValueT>
        ValueT from( const std::string & v, const ValueT & dflt = ValueT() )
        {
                return Private::from_string( v, dflt );
        }


        /**
           See stringutil::translate_entities() for details.
        */
        typedef std::map<std::string,std::string> entity_map;


        // YAGNI!
        struct entity_translator
        {
                entity_translator( const entity_map & map, bool reverse ) : m_map(map),m_rev(reverse)
                {
                }
                void operator()( std::string & str ) const
                {
                        stringutil::translate_entities( str, this->m_map, this->m_rev );
                }
        private:
                entity_map m_map;
                bool m_rev;
                         
        };

        /**
           Internal-use initializer for setting up an entity
           translation map for default quote-escaping behaviour.
        */
        struct default_escapes_initializer
        {
                inline void operator()( entity_map & map )
                {
                        map["\\"] = "\\\\";
                        map["\'"] = "\\\'";
                        map["\""] = "\\\"";
                }
        };


        /** Internal marker type. */
        template <typename ContextT> struct str_sharing_context {};

        /**
           Returns the default entity translation map, which can be used to
           [un]slash-escape the folling entities: '\\', '\'', '"'.
        */
        inline const entity_map &
        default_escapes_translations()
        {
                typedef phoenix::phoenix<entity_map,
                        str_sharing_context<entity_map>,
                        default_escapes_initializer
                        > TMap;
                return TMap::instance();
        }

        /**
           Converts v to a string, applies translate_entities(...,trans,reverse ),
           and returns the resulting string.
        */
        template <typename ValueT>
        std::string translate( const ValueT & v,
                               const entity_map & trans,
                               bool reverse )
        {
                std::string val = to( v );
                stringutil::translate_entities( val, trans, reverse );
                return val;
        }


        /**
           Calls translate( v,trans, false);
        */
        template <typename ValueT>
        std::string escape( const ValueT & v, const entity_map & trans = default_escapes_translations() )
        {
                return translate( v, trans, false );
        }


        /**
           Calls translate( v, trans, true );
        */
        template <typename ValueT>
        std::string unescape( const ValueT & v, const entity_map & trans = default_escapes_translations() )
        {
                return translate( v, trans, true );
        }

        /**
           Returns v as a quoted string, using the given quote
           character.
        */
        template <typename ValueT>
        std::string quote( const ValueT & v, const std::string & quote = "\'" )
        {
                return quote + to( v ) + quote;
        }

} // namespace str





#endif // s11n_net_str_STR_HPP_INCLUDED
