#ifndef stringutil_STRING_H_INCLUDED
#define stringutil_STRING_H_INCLUDED 1

#include <string>
#include <map>

/**
   The stringutil namespace encapsulates a number of functions and
   classes which have proven useful when working with std::strings.
*/
namespace stringutil {

        /**
           Convenience typedef for use with translate_entities().
         */
        typedef std::map<std::string,std::string> EntityMap;

        /**
           For each entry in the input string, the characters are
           mapped to string sequences using the given
           translation_map. Where no mappings exist, the input
           sequence is left as-is.

           It returns the number of translations made.

           If reverse_translation == true then a reverse mapping is
           done: map values are treated as keys. 

           This is useful, for example, for doing XML-entity-to-char
           conversions.

           It is amazingly INefficient, but seems to work okay.

        */
        size_t translate_entities( std::string &, const EntityMap & translation_map, bool reverse_translation = false );


        /**
           A policy enum used by trim_string().
        */
        enum TrimPolicy {
        /**
           Trim only leading spaces.
         */
        TrimLeading = 0x01,
        /**
           Trim only trailing spaces.
         */
        TrimTrailing = 0x02,
        /**
           Trim leading and trailing spaces.
         */
        TrimAll = TrimLeading + TrimTrailing
        };

        /**
           Trims leading and trailing whitespace from the input string
           and returns the number of whitespace characters removed.
         */
        size_t trim_string( std::string &, TrimPolicy = TrimAll );
        /**
           Trims leading and trailing whitespace from the input string
           and returns the trimmed string.
         */
        std::string trim_string( const std::string &, TrimPolicy = TrimAll );


        /**
           Attempts to remove all backslash-escaped chars from str.

           Removes backslash-escaped newlines from the input string, including
           any whitespace immediately following each backslash.

           The optional slash parameter defines the escape character.
        */
        size_t strip_slashes( std::string &str, const char slash = '\\' );

        /**
           Adds an escape sequence in front of any characters in
           instring which are also in the list of chars_to_escape.
           Returns the number of escapes added.

           e.g., to escape (with a single backslash) all $, % and \ in
           mystring with a backslash:

           <pre>
           escape_string( mystring, "$%\\", "\\" );
           </pre>

           (WARNING: the doxygen-generated HTML version of these docs
           may incorrectly show single backslashes in the above example!)


        */
        size_t escape_string( std::string & instring, const std::string & chars_to_escape, const std::string & escape_seq = "\\" );

        /**
           normalize_string() is like trim_string() and
           strip_slashes(), combined, plus it removes leading/trailing
           quotes:

           <pre>
           "this is a \
           sample multi-line, backslash-escaped \
           string."
           </pre>
           
           Will translate to:
           <pre>
           this is a sample multi-line, backslash-escaped string.
           </pre>
        */
        void normalize_string( std::string & );


        /**
           Returns the first whitespace-delimited token from the given
           string, or an empty string if there is no such token.
        */
        std::string first_token( const std::string & );

        /**
           Returns the passed-in string, minus the first
           whitespace-delimited token. An empty string is returned if
           there is no second token.
         */
        std::string after_first_token( const std::string & );



        /**
           Returns int values for chars '0'-'9', 'a'-'f' and 'A'-'F',
           else -1.
        */
        int int4hexchar( char c );

        /**
           Returns decimal value of wd, which is assumed to be a
           hex-encoded number. wd may optionally be prefixed with '#',
           as in \#ff00ff. Case is insignificant.

           On error 0 is returned, but 0 is also a valid number, so
           there is really no way of knowing if it fails or not. :/
        */
        int hex2int( const std::string & wd );



} // namespace stringutil

#endif // stringutil_STRING_H_INCLUDED
