#ifndef tostring_TO_STRING_H_INCLUDED
#define tostring_TO_STRING_H_INCLUDED 1
// Author: stephan beal <stephan@s11n.net>
// License: Public Domain

#include <string>
#include <iostream>
#include <sstream>
/**
   The tostring namespace encapsulates the to_string() and
   from_string() free functions. These functions do what is commonly
   called "lexical casting" - converting i/ostreamable types to/from
   strings.
*/
namespace tostring
{
	/**
           to_string() and from_string() provide an interface for
           converting strings into arbitrary other types and vice
           versa.

           It supports any types which to which std::istream>>X and
           std::ostream<<X can be applied.

           One immediate use i can imagine for this class is
           converting arguments read in via the command-line or
           libreadline. Another is using it to simulate a multi-type
           std::map, where you store string representations of the
           data, which can be easily converted to the proper types via
           from_string().

           Tries to from_string() a value of type value_type
           from the string str.  Returns an object of type
           value_type, hopefully properly populated from the
           string, or errorVal if it cannot convert the value
           poperly.  As errorVal is necessarily
           client/case-specific, you must supply your own
           error value. The error value need not be an error
           at all: this parameter is often used as a default
           value when pulling objects out of (e.g.) 
           configuration object (with the intention being,
           "the one i ask for may not be there - in that case
           assume errorVal is the value i want").

           Note that if you call from_string<T>(...) and T is
           one of std::string or char *, you will only get the
           first token from the string. To get around this,
           use the non-template forms, from_string( const
           std::string & ) and from_string( const char * ).

           This function is known to have problems with
           booleans:

           bool b = tostring::from_string( "foo", true );

           because bool is often an int of some type and the compiler
           may complain about ambiguity or type loss or some such. In
           this case explicitely pass the templatized type:

           bool b = tostring::from_string&lt;bool&gt;( "foo", true );
        */
	template <typename value_type>
        value_type from_string( const std::string & str, const value_type & errorVal )
	{
		std::istringstream is( str );
		if ( !is )
			return errorVal;
		value_type foo = value_type();
		if ( is >> foo )
			  return foo;
		  return errorVal;
	}

	/**
           A quasi-bogus overload to avoid the
           first-token-only problem of from_string<>().
        */
	inline std::string from_string( const std::string & str, const std::string & errorVal )
	{
		return str;
	}

	/**
           A quasi-bogus overload to avoid the
           first-token-only problem of from_string<>().
        */
	inline std::string from_string( const char *str, const char *errorVal )
	{
		return std::string( str );
	}


        /**
           Returns a string representation of the given object, which must be ostreamble.
        */
	template <typename value_type>
        std::string to_string( const value_type & obj )
	{
		std::ostringstream os;
                os << std::fixed;
		os << obj;
		return os.str(  );
	}

        /**
           Overloaded for strings-via-streams reasons.
        */
	inline std::string to_string( const char *obj )
	{
		return std::string( obj ? obj : "" );
	}

        /**
           Overloaded for strings-via-streams reasons.
        */
	inline std::string to_string( const std::string & obj )
	{
		return obj;
	}



}; // namespace tostring

#endif // tostring_TO_STRING_H_INCLUDED

