// a simple random number generator for shellish

#include <s11n.net/shellish/shellish.hpp>
#include <s11n.net/shellish/strtool.hpp>
#include <s11n.net/shellish/shellish_debuggering_macros.hpp> // CERR

#include <map>

#if shellish_ENABLE_S11N
#include <s11n.net/s11n/s11nlite.hpp>
#include <s11n.net/s11n/micro_api.hpp>
#include <s11n.net/s11n/proxy/std/map.hpp>
#include <s11n.net/s11n/proxy/pod/string.hpp>
#endif //!shellish_ENABLE_S11N


namespace shellish {

	/**
	   shellish::props provides an interface for mapping arbitrary
	   key/value pairs to "contexts". In practice this has shown
	   useful as a way of providing a common way to manipulate
	   application-specific metadata. e.g., this is used in one
	   client app to map information about game pieces, like their
	   combat strengths.

	   If shellish is built with s11n support then load/save
	   support for the property maps is added, otherwise these
	   are no-op commands.

	   Sample usage, assuming input is given at the shellish
	   console:

<pre>
prompt: prop set MyData SomeKey 123
prompt: prop set MyData Key2 456789
prompt: prop get MyData
        Key2=456789
        SomeKey=123
prompt: prop unset MyData Key2
prompt: prop get MyData
        SomeKey=123
</pre>   

	*/
namespace props {

	typedef std::map<std::string,std::string> prop_map;
	typedef std::map<std::string, prop_map > prop_map_map;

	static prop_map_map & props()
	{
		static prop_map_map bob;
		return bob;
	}

	static prop_map & props( const std::string & key )
	{
		return props()[key];
	}

	static size_t show_props( const prop_map & src )
	{
		size_t ret = 0;
 		std::ostream & os = ::shellish::ostream();
		prop_map::const_iterator it = src.begin();
		prop_map::const_iterator et = src.end();
		for( ; et != it; ++it )
		{
			++ret;
			os << '\t'<<(*it).first << "="<<(*it).second << "\n";
		}
		return ret;
	}

	template <typename MapType>
	bool save_map( const MapType src, const std::string & dest )
	{
		std::ostream & os = ::shellish::ostream();
#if !shellish_ENABLE_S11N
		os << "saving of properties not supported: shellish built without s11n support.\n";
		return false;
#else // use s11n...
		os << "Saving properties map to '"<<dest<<"'... ";
		try
		{
			if( s11nlite::save( src, dest ) )
			{
				os << "succeeded.\n";
				return true;
			}
			else
			{
				os << "FAILED!\n";
				return false;
			}
		}
		catch(const std::exception & ex)
		{
			os << "caught std::exception: " << ex.what() << "\n";
			return false;
		}
		catch(...)
		{
			os << "caught UNKNOWN EXCEPTION!\n";
			return false;
		}
		return false; // won't get this far
#endif //shellish_ENABLE_S11N
	}

	int handle_properties_save( const shellish::arguments & args )
	{
		// argc==1: $1=filename
		// argc== 2: $1=context $2=filename
		// argc<0 or argc>2: usage error
		shellish::arguments a(args);
		std::string a0 = a.shift(); // pop $0
		size_t ac = a.argc();
		if( 1 == ac )
		{
			std::string fname = a.shift();
			return save_map( props(), fname ) ? 0 : ::shellish::ErrorResourceAcquisition;
		}
		else if( 2 == ac )
		{
			std::string ctx = a.shift();
			std::string fname = a.shift();
			return save_map( props(ctx), fname ) ? 0 : ::shellish::ErrorResourceAcquisition;
		}
		else
		{
			std::ostream & os = ::shellish::ostream();
			os << "try one of the following:\n"
			   << '\t' << a0 << " FILENAME\n"
			   << '\t' << a0 << " CONTEXT FILENAME\n";
			return ::shellish::ErrorUsageError;
		}
		return -666; // won't get this far
	}


	template <typename MapType>
	bool load_map( const std::string & src, MapType & dest )
	{
#if !shellish_ENABLE_S11N
		os << "loading of properties not supported: shellish built without s11n support.\n";
		return false;
#else // use s11n...
		std::ostream & os = ::shellish::ostream();
		os << "Loading properties map from '"<<src<<"'... ";
		try
		{
			::s11n::cleanup_ptr<MapType> cp( s11nlite::load_serializable<MapType>( src ) );
			if( ! cp.get() )
			{
				os << "FAILED!\n";
				return false;
			}
			dest = *cp;
			os << "succeeded.\n";
			return true;
		}
		catch(const std::exception & ex)
		{
			os << "caught std::exception: " << ex.what() << "\n";
			return false;
		}
		catch(...)
		{
			os << "caught UNKNOWN EXCEPTION!\n";
			return false;
		}
		return false; // won't get this far
#endif //shellish_ENABLE_S11N
	}

	int handle_properties_load( const shellish::arguments & args )
	{
		// argc==1: $1=filename
		// argc== 2: $1=context $2=filename
		// argc<0 or argc>2: usage error
		shellish::arguments a(args);
		std::string a0 = a.shift(); // pop $0
		size_t ac = a.argc();
		if( 1 == ac )
		{
			std::string fname = a.shift();
			return load_map( fname, props() ) ? 0 : ::shellish::ErrorResourceAcquisition;
		}
		else if( 2 == ac )
		{
			std::string ctx = a.shift();
			std::string fname = a.shift();
			return load_map( fname, props(ctx) ) ? 0 : ::shellish::ErrorResourceAcquisition;
		}
		else
		{
			std::ostream & os = ::shellish::ostream();
			os << "try one of the following:\n"
			   << '\t' << a0 << " FILENAME\n"
			   << '\t' << a0 << " CONTEXT FILENAME\n";
			return ::shellish::ErrorUsageError;
		}
		return -666; // won't get this far
	}


	int handle_properties_set( const shellish::arguments & args )
	{ // $1=context, $2=key, $3=value
 		std::ostream & os = ::shellish::ostream();
		shellish::arguments a(args);
		std::string a0 = a.shift(); // pop $0
		size_t ac = a.argc();
		if( (ac < 2) )
		{
			os << "usage error: try: " << a0 << " CONTEXT KEY [VALUE]\n";
			return ::shellish::ErrorUsageError;
		}

		std::string c = a.shift();
		std::string k = a.shift();
		std::string v = a.str(); // the rest
		props(c)[k] = v;
		os << "set property: [" << c << "] " << k<<"="<<v <<std::endl;
		return 0;
	}

	int handle_properties_unset( const shellish::arguments & args )
	{ // [$1=context [$2..$N=key]]
		shellish::arguments a(args);
		a.shift(); // pop $0

		size_t ac = a.argc();
		if( 0 == ac ) // clear everything!
		{
			props().clear();
			return 0;
		}
		else if( 1 == ac ) // clear context
		{
			props().erase( a.shift() );
			return 0;
		}

		// else treat $2..$N as specific keys to get
		std::string ctx = a.shift();
		prop_map_map::iterator findcheck = props().find(ctx);
		if( props().end() == findcheck )
		{
			return 0;
		}
		prop_map & pm = (*findcheck).second;
		std::string key;
		while(1)
		{
			key = a.shift();
			if( key.empty() ) break;
			pm.erase(key);
		}
		return 0;
	}

	int handle_properties_get( const shellish::arguments & args )
	{ // [$1=context [$2..$N=key]]
		shellish::arguments a(args);
		a.shift(); // pop $0
		std::ostream & os = ::shellish::ostream();

		size_t ac = a.argc();
		if( 0 == ac )
		{ // list all properties
			os << "Listing all properties...\n";
			prop_map_map::const_iterator mit = props().begin();
			prop_map_map::const_iterator met = props().end();
			for( ; met != mit; ++mit )
			{
				os << "Properties for context ["<<(*mit).first<<"]:\n";
				if( 0 == show_props((*mit).second) )
				{
					os << "\t(none)\n";
				}
			}
			return 0;
		}
		else if( 1 == ac )
		{ // list all props in context $1
			std::string ctx = a.shift();
			if( 0 == show_props(props(ctx)) )
			{
				os << "\t(none)\n";
			}
			return 0;
		}

		// else treat $2..$N as specific keys to get
		std::string ctx = a.shift();

		prop_map_map::iterator findcheck = props().find(ctx);
		if( props().end() == findcheck )
		{
			return 0;
		}
		prop_map & pm = (*findcheck).second;
		prop_map::const_iterator pit = pm.begin();
		prop_map::const_iterator pet = pm.end();
		std::string key;
		while(1)
		{
			key = a.shift();
			if( key.empty() ) break;
			pit = pm.find(key) ;
			if( pet == pit ) continue;
			os << '\t'<<(*pit).first << "="<<(*pit).second << "\n";
		}
		return 0;
	}

	/**
	*/
	int handle_properties( const shellish::arguments & args )
	{
		shellish::arguments a( args );
		std::ostream & os = ::shellish::ostream();
		std::string a0 = a.shift(); // pop $0
		std::string cmd = a[0];

		if( "set" == cmd )
		{
			return handle_properties_set(a);
		}
		else if( "get" == cmd )
		{
			return handle_properties_get(a);
		}
		else if( "unset" == cmd )
		{
			return handle_properties_unset(a);
		}
		else if( "save" == cmd )
		{
			return handle_properties_save(a);
		}
		else if( "load" == cmd )
		{
			return handle_properties_load(a);
		}
			

		std::ostringstream usage;
		usage << "Usage: " << a0 << " COMMAND [arguments]\n"
		      << "where the COMMAND may be one of the following:\n"
		      << "\n\tset CONTEXT KEY VALUE\n"
		      << "Associates a key/value pair in the given context.\n"
		      << "\n\tget [CONTEXT [KEY [KEY2..KEYN]]]\n"
		      << "Echos the value(s) of the given context/key(s), or all properties if no context is given.\n"
		      << "\n\tunset [CONTEXT [KEY]]\n"
		      << "Clears all properties, or all in the given context, or only that of the given key.\n"
		      << "\nCONTEXTs may be any single token.\n"
#if shellish_ENABLE_S11N
		      << "\n\t(save|load) FILENAME\n"
		      << "Saves/loads all properties to the given file.\n"
		      << "\n\t(save|load) CONTEXT FILENAME\n"
		      << "Saves/loads all properties for the given context to the given file.\n"
#endif
			;

		os << usage.str() << std::endl;
		return ::shellish::ErrorUsageError;
	}

        void init_shellish_properties()
        {
                shellish::map_commander( "prop", handle_properties, "Manage arbitrary key/value pairs." );
        }

        static int bogus_properties_init = (init_shellish_properties(),0);
} }
